
/*
 *  $Id: tperf.c,v 1.7 2006/02/01 18:44:30 terry Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <math.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "config.h"

const char version[] = "Tperf 1.4 (20060202)";

void tcp_server_setup();
void udp_server_setup();
void client_setup();
void write_loop();
void read_loop();
void send_loop();
void receive_loop();
void server_socket_setup();
void client_socket_setup();
void timer_setup();
void buffer_setup();
void create_socket();
void set_signal_handler();
void set_socket_buffer_size();
void cleanup();
void print_info();
void print_stats();
void int_handler(int);
void alrm_handler(int);
void get_options(int, char **);
void usage();

const uint32_t stop = 0xffffffffU;
int server = 0;
int client = 0;
int udp = 0;
int verbose = 1;
u_short port = SERVER_PORT;
uint32_t bw = 0;
size_t sbsize = 0;
size_t len = RW_SIZE;
uint32_t iter = 0xffffffffU;
time_t sec = DURATION;
char server_ipaddr[16];

int s;
u_char *buf;
struct itimerval itv;
struct timespec start;
struct timespec ts;
struct timespec elapsed = { 0, 0 };
struct timespec next = { 0, 0 };
uint32_t interval = 0;
uint32_t count = 0;
uint32_t rcount = 0;
uint32_t lost = 0;
uint32_t ooo = 0;
uint32_t duped = 0;
ssize_t last_len = 0;
int start_flag = 0;
int end_flag = 0;
int len_flag = 0;
int intr_flag = 0;

inline void
start_time_measure()
{
  if (clock_gettime(CLOCK_REALTIME, &start) < 0) {
    perror("clock_gettime");
    exit(1);
  }
  start_flag = 1;
}

inline void
record_time_measure()
{
  if (clock_gettime(CLOCK_REALTIME, &ts) < 0) {
    perror("clock_gettime");
    exit(1);
  }
}

inline void
update_time_measure()
{
  elapsed.tv_sec += ts.tv_sec-start.tv_sec;
  elapsed.tv_nsec += ts.tv_nsec-start.tv_nsec;
}

inline double
get_elapsed_time()
{
  return elapsed.tv_sec+(double)elapsed.tv_nsec*1E-9;
}

inline void
set_timer()
{
  if (sec <= 0)  return;

  if (setitimer(ITIMER_REAL, &itv, NULL) < 0) {
    perror("setitimer");
    exit(1);
  }
}

inline void
delay_loop()
{
  struct timespec now;

  do {
    if (clock_gettime(CLOCK_REALTIME, &now) < 0) {
      perror("clock_gettime");
      exit(1);
    }
  } while (now.tv_sec < next.tv_sec ||
           (now.tv_sec == next.tv_sec && now.tv_nsec < next.tv_nsec));
}

inline void
insert_delay()
{
  if (!interval)  return;

  if (!next.tv_sec) {
    if (clock_gettime(CLOCK_REALTIME, &next) < 0) {
      perror("clock_gettime");
      exit(1);
    }
  }
  else
    delay_loop();

  next.tv_nsec += interval;
  if (next.tv_nsec >= 1000000000) {
    next.tv_sec++;
    next.tv_nsec -= 1000000000;
  }
}

inline ssize_t
read_buffer()
{
  ssize_t left, i, n;

  left = len;  i = 0;

  while (left > 0) {
    if ((n = read(s, buf+i, left)) < 0)  return n;
    if (n == 0)  break;
    rcount++;

    left -= n;  i += n;
  }

  return i;
}

int
main(int argc, char **argv)
{
  get_options(argc, argv);

  if (server) {
    if (udp) {
      udp_server_setup();
      receive_loop();
    }
    else {
      tcp_server_setup();
      read_loop();
    }
  }
  else if (client) {
    client_setup();
    if (udp)  send_loop();
    else  write_loop();
  }
  else {
    usage();
    return 1;
  }

  return 0;
}

void
tcp_server_setup()
{
  server_socket_setup();
  buffer_setup();

  if (listen(s, 1) < 0) {
    perror("listen");
    exit(1);
  }

  print_info();

  if ((s = accept(s, NULL, NULL)) < 0) {
    perror("accept");
    exit(1);
  }
}

void
udp_server_setup()
{
  server_socket_setup();
  buffer_setup();

  print_info();
}

void
client_setup()
{
  client_socket_setup();
  timer_setup();
  buffer_setup();

  print_info();
}

void
write_loop()
{
  set_timer();
  start_time_measure();

  while (count < iter && !intr_flag) {
    if ((last_len = write(s, buf, len)) < 0) {
      perror("write");
      exit(1);
    }
    count++;
  }

  record_time_measure();
}

void
read_loop()
{
  ssize_t n;

  while (1) {
    if ((n = read_buffer()) < 0) {
      perror("read");
      exit(1);
    }
    record_time_measure();
    if (n == 0)  break;
    last_len = n;
    count++;
    if ((size_t)n < len)  break;

    if (!start_flag)  start_time_measure();
  }

  end_flag = 1;
}

void
send_loop()
{
  set_timer();
  start_time_measure();

  while (count < iter && !end_flag) {
    *(uint32_t *)buf = htonl(++count);
    if (count == iter || intr_flag) {
      *((uint32_t *)buf+1) = htonl(stop);
      end_flag = 1;
    }

    insert_delay();
    while ((last_len = send(s, buf, len, 0)) < 0) {
      if (errno == ENOBUFS)  continue;
      perror("send");
      exit(1);
    }
    if (verbose && (size_t)last_len < len)
      printf("sent %d byte datagram (#%u)\n", last_len, count);
  }

  record_time_measure();
}

void
receive_loop()
{
  uint32_t last, seq;

  last = 0;
  while (1) {
    if ((last_len = recv(s, buf, len, 0)) < 0) {
      perror("recv");
      exit(1);
    }
    record_time_measure();
    count++;
    seq = ntohl(*(uint32_t *)buf);
    if (verbose && (size_t)last_len < len)
      printf("received %u byte datagram (#%u)\n", last_len, seq);

    if (seq > last+1) {
      if (verbose)
        printf("lost %d datagrams (#%u - #%u)\n", seq-last-1, last+1, seq-1);
      lost += seq-last-1;
    }
    else if (seq < last) {
      if (verbose)
        printf("received out-of-order datagram (#%u < #%u)\n", seq, last);
      ooo++;
      seq = last;
    }
    else if (seq == last) {
      if (verbose)
        printf("received duplicated datagram (#%u)\n", seq);
      duped++;
    }
    last = seq;

    if (ntohl(*((uint32_t *)buf+1)) == stop)  break;

    if (!start_flag)  start_time_measure();
  }

  end_flag = 1;
}

void
server_socket_setup()
{
  struct sockaddr_in sin;

  create_socket();
  set_signal_handler();
  set_socket_buffer_size();

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(port);
  sin.sin_addr.s_addr = htonl(INADDR_ANY);
  if (bind(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
    perror("bind");
    exit(1);
  }
}

void
client_socket_setup()
{
  struct sockaddr_in sin;

  create_socket();
  set_signal_handler();
  set_socket_buffer_size();

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(port);
  if (inet_pton(AF_INET, server_ipaddr, &sin.sin_addr) < 0) {
    perror("inet_pton");
    exit(1);
  }
  if (connect(s, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
    perror("connect");
    exit(1);
  }
}

void
timer_setup()
{
  memset(&itv, 0, sizeof(itv));
  itv.it_value.tv_sec = sec;
  itv.it_value.tv_usec = 1;

  if (bw > 0)  interval = rint((double)8000*len/bw);
}

void
buffer_setup()
{
  if ((buf = calloc(len, 1)) == NULL) {
    perror("calloc");
    exit(1);
  }
}

void
create_socket()
{
  if ((s = socket(PF_INET, udp ? SOCK_DGRAM : SOCK_STREAM, 0)) < 0) {
    perror("socket");
    exit(1);
  }

  if (atexit(cleanup) < 0) {
    perror("atexit");
    exit(1);
  }
}

void
set_signal_handler()
{
  struct sigaction sa_int, sa_alrm;

  memset(&sa_int, 0, sizeof(sa_int));
  sa_int.sa_handler = int_handler;
  sa_int.sa_flags = SA_RESTART;
  if (sigaction(SIGINT, &sa_int, NULL) < 0) {
    perror("sigaction");
    exit(1);
  }

  memset(&sa_alrm, 0, sizeof(sa_alrm));
  sa_alrm.sa_handler = alrm_handler;
  sa_alrm.sa_flags = SA_RESTART;
  if (sigaction(SIGALRM, &sa_alrm, NULL) < 0) {
    perror("sigaction");
    exit(1);
  }
}

void
set_socket_buffer_size()
{
  int option;
  socklen_t slen;

  option = server ? SO_RCVBUF : SO_SNDBUF;
  slen = sizeof(sbsize);
  if (sbsize > 0 &&
      setsockopt(s, SOL_SOCKET, option, &sbsize, slen) < 0) {
    perror("setsockopt");
    exit(1);
  }

  if (getsockopt(s, SOL_SOCKET, option, &sbsize, &slen) < 0) {
    perror("getsockopt");
    exit(1);
  }
}

void
cleanup()
{
  update_time_measure();

  free(buf);

  if (close(s) < 0) {
    perror("close");
    exit(1);
  }

  print_stats();
}

void
print_info()
{
  printf("%s size: %u bytes\n"
         "%s socket buffer size: %u bytes\n",
         server ? (udp ? "receive" : "read buffer") : (udp ? "send" : "write"),
         len, server ? "receive" : "send", sbsize);
}

void
print_stats()
{
  double lrate, etime, bandwidth;

  if (server) {
    if (udp) {
      if (!end_flag)  puts("not received last datagram");
      lrate = (double)100*lost/(count+lost);
      printf("received %u datagrams\n"
             "lost %u datagrams (%.3f%%)\n", count, lost, lrate);
      if (ooo)  printf("received %u our-of-order datagrams\n", ooo);
      if (duped)  printf("received %u duplicated datagrams\n", duped);
    }
    else {
      if (!end_flag)  puts("not read end of file");
      printf("received %u datas (read %u times)\n", count, rcount);
    }
  }
  else
    printf(udp ? "sent %u datagrams\n" : "wrote %u times\n", count);

  etime = get_elapsed_time();
  bandwidth = ((double)8*len*(count-(server ? 2 : 1))+last_len)/(etime*1E6);
  printf("elapsed %f seconds\n"
         "%.3f Mbps bandwidth\n", etime, bandwidth);
}

void
int_handler(int sig)
{
  if (sig != SIGINT)  return;

  if (server)  exit(0);

  intr_flag = 1;
}

void
alrm_handler(int sig)
{
  if (sig != SIGALRM)  return;

  intr_flag = 1;
}

void
get_options(int argc, char **argv)
{
  for (argc--, argv++ ; argc > 0; argc--, argv++) {
    if (!strcmp(argv[0], "-s")) {
      server = 1;  client = 0;
      continue;
    }
    else if (!strcmp(argv[0], "-c")) {
      client = 1;  server = 0;
      strcpy(server_ipaddr, argv[1]);
    }
    else if (!strcmp(argv[0], "-u")) {
      udp = 1;
      if (!len_flag)  len = DGRAM_SIZE;
      continue;
    }
    else if (!strcmp(argv[0], "-p"))
      port = strtol(argv[1], NULL, 10);
    else if (!strcmp(argv[0], "-w"))
      sbsize = strtol(argv[1], NULL, 10);
    else if (!strcmp(argv[0], "-l")) {
      len = strtol(argv[1], NULL, 10);
      len_flag = 1;
    }
    else if (!strcmp(argv[0], "-t")) {
      sec = strtol(argv[1], NULL, 10);
      iter = sec > 0 ? 0xffffffffU : 0;
    }
    else if (!strcmp(argv[0], "-i")) {
      iter = strtol(argv[1], NULL, 10);
      sec = 0;
    }
    else if (!strcmp(argv[0], "-b")) {
      bw = strtol(argv[1], NULL, 10);
      udp = 1;
      if (!len_flag)  len = DGRAM_SIZE;
    }
    else if (!strcmp(argv[0], "-q")) {
      verbose = 0;
      continue;
    }
    else if (!strcmp(argv[0], "-h")) {
      usage();
      exit(0);
    }
    else if (!strcmp(argv[0], "-v")) {
      puts(version);
      exit(0);
    }
    else if (!strncmp(argv[0], "-", 1)) {
      fprintf(stderr, "get_options: unknown option %s\n", argv[0]);
      exit(1);
    }
    else  break;

    argc--; argv++;
  }
}

void
usage()
{
  fprintf(stderr,
          "usage: tperf -s [options...]\n"
          "       tperf -c server_ipaddr [options...]\n"
          "options: -u    use UDP rather than TCP (implies -l %u)\n"
          "         -p #  server port to listen on/connect to (default %u)\n"
          "         -w #  send/receive socket buffer size to request\n"
          "         -l #  length in bytes to read/write (default %u)\n"
          "         -t #  time in seconds to transmit for (default %d)\n"
          "         -i #  number of iterations to transmit (instead of -t)\n"
          "         -b #  bandwidth to send at in Mbps (implies -u)\n"
          "         -q    suppress messages during execution\n"
          "         -h    show this help message and quit\n"
          "         -v    show version information and quit\n",
          DGRAM_SIZE, SERVER_PORT, RW_SIZE, DURATION);
}

