/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"
`include "dunit.vh"
`include "nrm.vh"

module dunit(CLK,
             RST,

             SET,  SADDR, SDIN,
             MRE,  MADDR, MDOUT,
             START,

             IREQ, IRDY,
             DIN,
             OREQ, ORDY,
             DOUT,
             ID);
   
   input          CLK, RST;

   input [5:0]    SET;
   input [9:0]    SADDR;
   input [31:0]   SDIN;

   input          START;
   
   input [1:0]    IREQ;
   output         IRDY;
   input [31:0]   DIN;

   output [1:0]   OREQ;
   input          ORDY;
   output [31:0]  DOUT;

   input          MRE;
   input [9:0]    MADDR;
   output [31:0]  MDOUT;
   
   input [7:0]    ID;
   
   reg [4:0]      STATE;
   
   wire           fifo_io_we,  fifo_io_re;
   wire           fifo_io_emp, fifo_io_full;
   
   wire [31:0]    fifo_io_din, fifo_io_dout;

   wire           mwea, mweb;
   wire [9:0]     maddra, maddrb;
   wire [31:0]    mdina, mdinb, mdouta, mdoutb;
   
   wire           pwea, pweb;
   wire [9:0]     paddra, paddrb;
   wire [31:0]    pdina, pdinb, pdouta, pdoutb;

   wire [2:0]     ipq_inst;
   wire [9:0]     ipq_rin;
   wire [41:0]    ipq_din, ipq_dout;
   wire           ipq_busy;
   
   reg            fifo_io_re_reg;

   reg [1:0]      OREQ;
   reg [31:0]     DOUT;

   reg [7:0]      MNUM;

   reg [3:0]      CYCLE_FLAG;
   reg [41:0]     IPQ_ROOT;

   reg [31:0]     N_ADDR[0:7];

   wire           TM_PASS_FLAG;
   wire           TC_PASS_FLAG;
   wire           PC_PASS_FLAG;
   wire           TM_TC_FLAG;

   wire [8:0] fifo_io_dc;

   reg [4:0]  SPFLG;
   reg [2:0]  MUFLG;
   reg [3:0]  RSFLG;
   reg [3:0]  PCFLG;
   reg [2:0]  TCFLG;
   reg        LPFLG;

   reg [7:0]  DNUM;

   reg [2:0]  RSSTATE;
   reg [4:0]  PCSTATE;

   reg        TCAL_FLG1, TCAL_FLG2;
   reg        PASS_FLG;
   reg        PASS_MOD;

   reg [31:0] DATA, DATB, DATC, DATD;

   reg        END_FLG;

   assign     IRDY = ~fifo_io_dc[8];
   assign     MDOUT = mdouta;
   
   assign     fifo_io_we  = |IREQ & IRDY;
   assign     fifo_io_din = DIN;

   assign     fifo_io_re  = ((STATE == `DUNIT_STATE_IDLE) & ~fifo_io_emp) | fifo_io_re_reg;

   assign     mwea   = (MUFLG[2]);
   assign     maddra = (MUFLG[0] | MUFLG[2] | RSFLG[0]) ? fifo_io_dout[9:0] :
                       (MRE) ? MADDR : 10'd0;
   assign     mdina  = (MUFLG[2]) ? DATA : 32'd0;

   assign     mweb   = (SET == `SET_MT | MUFLG[2]);
   assign     maddrb = (SET == `SET_MT) ? SADDR : 
                       (MUFLG[0] | MUFLG[2]) ? fifo_io_dout[25:16] : 
                       (RSFLG[0]) ? fifo_io_dout[19:10] : 10'd0;
   assign     mdinb  = (SET == `SET_MT) ? SDIN :
                       (MUFLG[2]) ? DATB : 32'd0;

   assign     pwea   = (PCFLG[1]) ? `ENABLE : `DISABLE;
   assign     paddra = (PCFLG[0]) ? fifo_io_dout[9:0] :
                       (PCFLG[1]) ? DATC[9:0] : 10'd0;
   assign     pdina  = (PCFLG[1]) ? fifo_io_dout : 32'b0;

   assign     pweb   = (SET == `SET_PT);
   assign     paddrb = SADDR;
   assign     pdinb  = SDIN;

   assign     ipq_inst = (SET == `SET_TT) ? `IPQ_IWET :
                         (SET == `SET_TI) ? `IPQ_IWEI :
                         (PCFLG[0])       ? `IPQ_READ :
                         (TCFLG[1])       ? `IPQ_UPDT :
                                            `IPQ_IDLE;
   
   assign     ipq_rin  = (PCFLG[0]) ? fifo_io_dout[9:0] :
                         (TCFLG[1]) ? DATC[9:0] :
                         10'd0;
   assign     ipq_din  = (SET == `SET_TT | SET == `SET_TI) ? {SADDR, SDIN} :
                         (TCFLG[1] & ~PASS_MOD) ? {DATC[9:0], fifo_io_dout} :
                         (TCFLG[1] & PASS_MOD)  ? {DATC[9:0], 32'h7F80_0000} :
                         42'b0;


   always@(posedge CLK) begin
      if(SET == `SET_AR) N_ADDR[SADDR[2:0]] <= SDIN;
      if(MUFLG[1])      DATA <= mdouta + {{26{fifo_io_dout[15]}},fifo_io_dout[15:10]};
      else if(RSFLG[1]) DATA <= mdouta;
      else if(PCFLG[1]) DATA <= pdouta;

      if(MUFLG[1])      DATB <= mdoutb + {{26{fifo_io_dout[31]}},fifo_io_dout[31:26]};
      else if(RSFLG[1]) DATB <= mdoutb;
      else if(PCFLG[1]) DATB <= fifo_io_dout;

      if(RSFLG[0])      DATC <= {20'b0, fifo_io_dout[31:20]};
      else if(PCFLG[0] | (TCFLG[0] & ~PASS_MOD)) DATC <= fifo_io_dout;

      if(PCFLG[2])      DATD <= ipq_dout[31:0];

      if(PCFLG[0])      TCAL_FLG1 <= fifo_io_dout[10];
      if(PCFLG[1])      PASS_FLG  <= ~|fifo_io_dout[30:23];

      if(~CYCLE_FLAG[1] & (STATE == `DUNIT_STATE_FTCH) & (fifo_io_dout[31:28] == `OP_REAS)) DNUM <= fifo_io_dout[27:20] + 1;
      else if(TCFLG[1])             DNUM <= DNUM - 1;
   end

   always@(posedge CLK) begin
      if(RST) begin
         TCAL_FLG2 <= `DISABLE;
         STATE <= `DUNIT_STATE_INIT;
         OREQ  <= 2'b00;
         DOUT  <= 32'b0;
         fifo_io_re_reg <= `DISABLE;
         SPFLG    <= 5'b00000;
         MUFLG    <= 3'b000;
         RSFLG    <= 4'b0000;
         PCFLG    <= 4'b000;
         TCFLG    <= 3'b000;
         LPFLG    <= 1'b0;
         RSSTATE  <= 3'b000;
         PCSTATE  <= 5'b00000;
         CYCLE_FLAG <= 4'b0;
         PASS_MOD <= 1'b0;
         END_FLG <= `DISABLE;
      end
      else begin
         casex(STATE)
           `DUNIT_STATE_INIT : begin
              if(START) begin
                 IPQ_ROOT   <= ipq_dout;
                 SPFLG      <= 5'b00001;
                 CYCLE_FLAG <= 4'b0;
                 STATE      <= `DUNIT_STATE_STRT;
                 END_FLG    <= `ENABLE;
              end
           end
           `DUNIT_STATE_STRT : begin
              END_FLG <= `DISABLE;
              casex(SPFLG)
                5'b00001 : begin
                   OREQ  <= 2'b01; DOUT <= {`OP_UPDT, 8'd1, N_ADDR[`ADDR_UPDT]};
                   SPFLG <= 5'b00010;
                end
                5'b00010 : begin
                   if(ORDY) begin
                      OREQ  <= 2'b10; DOUT  <= {22'b0, IPQ_ROOT[41:32]};
                      SPFLG <= 5'b00100;
                   end
                end
                5'b00100 : begin
                   if(ORDY) begin
                      OREQ  <= 2'b01; DOUT <= {`OP_DGTB, 8'd1, N_ADDR[`ADDR_DGTB]};
                      SPFLG <= 5'b01000;
                   end
                end
                5'b01000 : begin
                   if(ORDY) begin
                      OREQ  <= 2'b10; DOUT <= {22'b0, IPQ_ROOT[41:32]};
                      SPFLG <= 5'b10000;
                   end
                end
                5'b10000 : begin
                   if(ORDY) begin
                      OREQ  <= 2'b00; DOUT <= 32'b0;
                      SPFLG <= 5'b00000;
                      STATE <= `DUNIT_STATE_IDLE;
                   end
                end
              endcase // casex(SPFLG)
           end // case: `DUNIT_STATE_STRT
           `DUNIT_STATE_IDLE : begin
              fifo_io_re_reg         <= ~fifo_io_emp;
              if(~fifo_io_emp) STATE <= `DUNIT_STATE_FTCH;
           end
           `DUNIT_STATE_FTCH : begin
              casex(fifo_io_dout[31:28])
                `OP_UPDT : begin
                   MNUM  <= fifo_io_dout[27:20] - 1;
                   fifo_io_re_reg <= `DISABLE;
                   MUFLG <= 3'b001;
                   STATE <= `DUNIT_STATE_UPDT;
                end
                `OP_REAS : begin
                   fifo_io_re_reg <= `DISABLE;
                   if(CYCLE_FLAG[0]) begin
                      RSFLG   <= 4'b0001;
                      RSSTATE <= 3'b001;
                      STATE <= `DUNIT_STATE_REAS;
                   end
                   else begin
                      OREQ  <= 2'b01;
                      LPFLG <= 1'b0;
                      DOUT  <= {`OP_REAS, 8'd1, N_ADDR[`ADDR_LOOP]};
                      STATE <= `DUNIT_STATE_LOOP;
                   end
                end // case: `OP_REAS
                `OP_PRCS : begin
                   STATE   <= `DUNIT_STATE_PRCS;
                   PCFLG   <= 4'b0001;
                end
                `OP_TCAL : begin
                   PASS_MOD <= 1'b0;
                   TCFLG    <= 3'b001;
                   STATE    <= `DUNIT_STATE_TCAL;
                end
                `OP_TMOD : begin
                   PASS_MOD <= 1'b0;
                   TCFLG    <= 3'b001;
                   STATE    <= `DUNIT_STATE_TCAL;
                end
                default : begin
                end
              endcase
           end // case: `DUNIT_STATE_FTCH
           `DUNIT_STATE_LOOP : begin
              if(~LPFLG) begin
                 if(ORDY) begin
                    OREQ <= 2'b10;
                    DOUT <= fifo_io_dout;
                    LPFLG <= 1'b1;
                 end
              end
              else begin
                 if(ORDY) begin
                    OREQ    <= 2'b00; DOUT <= 32'b0;
                    LPFLG <= 1'b0;
                    STATE   <= `DUNIT_STATE_IDLE;
                 end
              end
           end
           `DUNIT_STATE_TCAL : begin
              fifo_io_re_reg <= `DISABLE;
              if(TCFLG[2]) begin
                 if(~ipq_busy) begin
                    TCFLG <= 3'b000;
                    if(|DNUM) begin
                       STATE <= `DUNIT_STATE_IDLE;
                    end
                    else begin
                       IPQ_ROOT   <= ipq_dout;
                       SPFLG      <= 5'b00001;
                       CYCLE_FLAG <= 4'b0;
                       STATE <= `DUNIT_STATE_STRT;
                       END_FLG <= `ENABLE;
                    end
                 end
              end
              else begin
                 TCFLG <= {TCFLG[1:0], 1'b0};
              end
           end
           `DUNIT_STATE_PRCS : begin
              fifo_io_re_reg <= `DISABLE;
              PCFLG <= {PCFLG[2:0], 1'b0};
              if(PCFLG[1]) begin
                 PCSTATE <= 5'b00001;
              end
              else begin
                 casex(PCSTATE)
                   5'b00001 : begin
                      if(~PASS_FLG) begin
                         OREQ  <= 2'b01; 
                         if(&ipq_dout[30:23] | TCAL_FLG1) begin
                            TCAL_FLG2 <= `ENABLE;
                            DOUT      <= {`OP_TCAL, 8'd2, N_ADDR[`ADDR_TCAL]};
                         end
                         else begin
                            DOUT <= {`OP_TMOD, 8'd2, N_ADDR[`ADDR_TMOD][19:0]};
                         end
                         PCSTATE  <= 5'b00010;
                         PASS_MOD <= 1'b0;
                      end
                      else begin
                         PASS_MOD <= 1'b1;
                         TCFLG    <= 3'b001;
                         PCSTATE <= 5'b00000;
                         STATE    <= `DUNIT_STATE_TCAL;
                      end
                   end
                   5'b00010 : begin
                      if(ORDY) begin
                         OREQ  <= 2'b11; DOUT <= DATC; // Rid
                         PCSTATE <= 5'b00100;
                      end
                   end
                   5'b00100 : begin
                      if(ORDY) begin
                         OREQ  <= 2'b11; DOUT <= DATB; // a_new
                         PCSTATE <= 5'b01000;
                      end
                   end
                   5'b01000 : begin
                      if(ORDY) begin
                         DOUT <= IPQ_ROOT[31:0]; // t
                         if(TCAL_FLG2) begin
                            OREQ  <= 2'b10;
                            PCSTATE <= 5'b10010;
                         end
                         else begin
                            OREQ  <= 2'b11;
                            PCSTATE <= 5'b10000;
                         end
                      end
                   end
                   5'b10000 : begin
                      if(ORDY) begin
                         OREQ  <= 2'b11; DOUT <= DATA; // a_old
                         PCSTATE <= 5'b10001;
                      end
                   end
                   5'b10001 : begin
                      if(ORDY) begin
                         OREQ  <= 2'b10; DOUT <= DATD; // tau_old
                         PCSTATE <= 5'b10010;
                      end
                   end
                   5'b10010 : begin
                      TCAL_FLG2 <= `DISABLE;
                      if(ORDY) begin
                         OREQ    <= 2'b00; DOUT <= 32'b0;
                         PCSTATE <= 5'b00000;
                         STATE   <= `DUNIT_STATE_IDLE;
                      end
                   end
                   default : begin
                   end
                 endcase // casex(PCSTATE)
              end
           end
           `DUNIT_STATE_UPDT : begin
              if(MUFLG[1] & |MNUM)  fifo_io_re_reg <= `ENABLE;
              else                  fifo_io_re_reg <= `DISABLE;
              if(MUFLG[2] & ~|MNUM) STATE <= `DUNIT_STATE_IDLE;
              if(MUFLG[2] & ~|MNUM) CYCLE_FLAG <= {CYCLE_FLAG[3:1], 1'b1};
              if(MUFLG[2] &  |MNUM) MNUM <= MNUM - 1;
              if(MUFLG[2] & ~|MNUM) MUFLG <= 3'b0;
              else                  MUFLG <= {MUFLG[1:0],MUFLG[2]};
           end
           `DUNIT_STATE_REAS : begin
              casex(RSSTATE)
                3'b001 : begin
                   OREQ  <= 2'b01; DOUT <= {`OP_PRCS, 8'd1, N_ADDR[`ADDR_PRCS]};
                   RSSTATE <= 3'b010;
                end
                3'b010 : begin
                   if(ORDY) begin
                      OREQ    <= 2'b11; DOUT <= DATC;
                      RSSTATE <= 3'b011;
                   end
                end
                3'b011 : begin
                   if(ORDY) begin
                      OREQ    <= 2'b11; DOUT <= DATA;
                      RSSTATE <= 3'b100;
                   end
                end
                3'b100 : begin
                   if(ORDY) begin
                      OREQ    <= 2'b10; DOUT <= DATB;
                      RSSTATE <= 3'b101;
                   end
                end
                3'b101 : begin
                   if(ORDY) begin
                      OREQ    <= 2'b00; DOUT <= 32'b0;
                      RSSTATE <= 3'b000;
                      STATE   <= `DUNIT_STATE_IDLE;
                   end
                end
              endcase // casex(RSSTATE)
              if(~CYCLE_FLAG[1] & RSFLG[0]) begin
                 CYCLE_FLAG <= {CYCLE_FLAG[3:2], 1'b1, CYCLE_FLAG[0]};
              end
              RSFLG <= {RSFLG[2:0], 1'b0};
           end
           default : begin
           end
         endcase // casex(STATE)
      end
   end // always@ (posedge CLK)
   
   fifo32x9_br fifo_io(.clk(CLK),
                       .rst(RST),
                       .rd_en(fifo_io_re),
                       .wr_en(fifo_io_we),
                       .full(fifo_io_full),
                       .data_count(fifo_io_dc),
                       .empty(fifo_io_emp),
                       .din(fifo_io_din),
                       .dout(fifo_io_dout));

   dpram32x10 mtable(.clka(CLK), .clkb(CLK),
                     .wea(mwea), .addra(maddra), .dina(mdina), .douta(mdouta),
                     .web(mweb), .addrb(maddrb), .dinb(mdinb), .doutb(mdoutb));

   dpram32x10 ptable(.clka(CLK), .clkb(CLK),
                     .wea(pwea), .addra(paddra), .dina(pdina), .douta(pdouta),
                     .web(pweb), .addrb(paddrb), .dinb(pdinb), .doutb(pdoutb));

   ipq ipq(.CLK(CLK),
           .RST(RST),
           .INST(ipq_inst),
           .RIN(ipq_rin),
           .DIN(ipq_din),
           .DOUT(ipq_dout),
           .BUSY(ipq_busy));

   wire [3:0]     I_OPCODE, O_OPCODE;
   assign         I_OPCODE = (IREQ == 2'b01) ? DIN[31:28] : 4'b0000;
   assign         O_OPCODE = (OREQ == 2'b01) ? DOUT[31:28] : 4'b0000;

   reg [31:0]     CNUM;
   always@(posedge CLK) begin
      if(RST) CNUM <= 32'b0;
      else begin
         if(END_FLG) CNUM <= CNUM + 1;
      end
   end

endmodule // dunit
