/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"

module dgtb_core(CLK, RST,
                 DGWE,
                 DGDIN,
                 REQ,
                 HOLD,
                 DIN,
                 DOUT);

   input  CLK, RST;
   input  DGWE;
   input [31:0] DGDIN;
   input  REQ;
   input  [31:0] DIN;
   output [31:0]  DOUT;
   output        HOLD;

   reg           HOLD;
   
   wire        idx_wea;
   wire [9:0]  idx_addra, idx_addrb;
   wire [17:0] idx_dina;
   wire [17:0] idx_doutb;

   wire         dgtb_wea;
   wire [12:0]  dgtb_addra, dgtb_addrb;
   wire [9:0]   dgtb_dina;
   wire [9:0]   dgtb_doutb;

   wire         reas_wea;
   wire [9:0]   reas_addra, reas_addrb;
   wire [19:0]   reas_dina;
   wire [19:0]   reas_doutb;

   reg [1:0]     STATE;
   reg [12:0]    DADDR;

   reg  [4:0]    RNUM;

   reg [9:0]     RID;

   reg           DO;
   reg           NF; // Not first
   
   
   assign        idx_wea   = (DGWE & DGDIN[31:30] == 2'b01) ? `WRITE : `READ;
   assign        idx_addra = DGDIN[29:20];
   assign        idx_dina  = DGDIN[17:0];

   assign        idx_addrb = DIN[9:0];
   
   assign        dgtb_wea   = (DGWE & DGDIN[31:30] == 2'b10) ? `WRITE : `READ;
   assign        dgtb_addra = DGDIN[29:17];
   assign        dgtb_dina  = DGDIN[9:0];

//   assign        dgtb_addrb = (STATE == 2'b01) ? idx_doutb[10:0] : 
   assign        dgtb_addrb = (STATE == 2'b01) ? idx_doutb[12:0] : 
                                                DADDR;

   assign        reas_wea   = (DGWE & DGDIN[31:30] == 2'b11) ? `WRITE : `READ;
   assign        reas_addra = DGDIN[29:20];
   assign        reas_dina  = DGDIN[19:0];

   assign        reas_addrb = (STATE == 2'b01) ? RID :
                              (STATE == 2'b10) ? dgtb_doutb : 10'd0;

   assign        DOUT = (STATE == 2'b01) ? {27'b0, idx_doutb[17:13]} : 
                        (DO) ? {1'b0, ~NF, RID, reas_doutb} : 32'b0;
   
   always@(posedge CLK) begin
      if(STATE == 2'b00) DO <= `DISABLE;
      else               DO <= `ENABLE;
      NF <= DO;
   end

   always@(posedge CLK) begin
      if(RST) begin
         HOLD <= `DISABLE;
         STATE  <= 2'b00;
      end
      else begin
         casex(STATE)
           2'b00 : begin
              if(REQ) begin
                 HOLD  <= `ENABLE;
                 RID   <= DIN[9:0];
                 STATE <= 2'b01;
              end
              else begin
                 HOLD <= `DISABLE;
              end
           end
           2'b01 : begin
              if(~|idx_doutb[17:13]) begin
                 STATE <= 2'b00;
              end
              else begin
                 RNUM  <= idx_doutb[17:13] - 1;
                 DADDR <= idx_doutb[12:0] + 1;
                 STATE <= 2'b10;
              end
           end
           2'b10 : begin
              RID   <= dgtb_doutb;
              if(|RNUM) begin
                 DADDR <= DADDR + 1;
                 RNUM  <= RNUM - 1;
              end
              else begin
                 STATE <= 2'b00;
              end
           end
         endcase // casex(STATE)
      end
   end // always@ (posedge CLK)

   sdpram18x10 idx(.clka(CLK), .clkb(CLK),
                   .wea(idx_wea),
                   .addra(idx_addra),
                   .addrb(idx_addrb),
                   .dina(idx_dina),
                   .doutb(idx_doutb));

   sdpram10x13 dgtb(.clka(CLK), .clkb(CLK),
                    .wea(dgtb_wea),
                    .addra(dgtb_addra),
                    .addrb(dgtb_addrb),
                    .dina(dgtb_dina),
                    .doutb(dgtb_doutb));
   
   sdpram20x10 reas(.clka(CLK), .clkb(CLK),
                    .wea(reas_wea),
                    .addra(reas_addra),
                    .addrb(reas_addrb),
                    .dina(reas_dina),
                    .doutb(reas_doutb));

   
endmodule // dgtb
