/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"
`include "dunit.vh"

module cno_dgtb(CLK, RST,
                AWE, ADDR,
                DWE, DIN,
                CRDY, OST,
                OREQ, ORDY, DOUT,
                HEADER
                );

   input CLK;
   input RST;

   input        AWE;
   input [19:0] ADDR;
   input        DWE;
   input [31:0] DIN;

   output       CRDY;
   input        OST;

   output [1:0]  OREQ;
   input         ORDY;
   output [31:0] DOUT;
   input [3:0]   HEADER;

   reg [1:0]     OREQ;
   reg [31:0]    DOUT;

   wire        fifo_addr_we,  fifo_addr_re;
   wire        fifo_addr_fl,  fifo_addr_em;
   wire [19:0] fifo_addr_din, fifo_addr_dout;
   wire [4:0]  fifo_addr_dc;

   wire        fifo_data_we,  fifo_data_re;
   wire        fifo_data_fl,  fifo_data_em;
   wire [31:0] fifo_data_din, fifo_data_dout;
   wire [4:0]  fifo_data_dc;

   reg [2:0]   STATE;
   reg [4:0]   TNUM;

   reg [6:0] RET_NUM;
   reg [6:0] RNUM;
   reg       ST, ET, RT; // Start, End, Reaction
   
   assign      CRDY = ~fifo_data_dc[4];
   assign      fifo_addr_we  = AWE;
   assign      fifo_addr_din = ADDR;
   assign      fifo_addr_re  = (STATE == 3'b000 & |TNUM);

   assign      fifo_data_we  = DWE;
   assign      fifo_data_din = DIN;
   assign      fifo_data_re  = (STATE == 3'b000 & |TNUM) | RT;
   
   always@(posedge CLK) begin
      if(RST) TNUM <= 5'b0;
      else begin
         if(STATE == 3'b001 & ST) begin
            if(~OST) TNUM <= TNUM - 1;
         end
         else if(OST) TNUM <= TNUM + 1;
      end
   end

   always@(posedge CLK) begin
      if(RST) begin
         STATE <= 3'b000;
         ST <= `DISABLE;
         ET <= `DISABLE;
         RT <= `DISABLE;
         OREQ <= 2'b00;
         DOUT <= 32'b0;
      end
      else begin
         casex(STATE)
           3'b000 : begin
              if(|TNUM) begin
                 ST <= `ENABLE;
                 STATE <= 3'b001;
                 RT <= `ENABLE;
              end
           end
           3'b001 : begin
              RT      <= `DISABLE;
              RNUM    <= fifo_data_dout[6:0];
              RET_NUM <= fifo_data_dout[6:0];
              STATE <= 3'b010;

              OREQ <= 2'b01;
              DOUT <= {HEADER, 1'b0, fifo_data_dout[6:0], fifo_addr_dout};
           end
           3'b010 : begin
              if(ORDY) begin
                 OREQ <= 2'b10;
                 DOUT <= fifo_data_dout;
                 ST   <= `DISABLE;

                 STATE <= 3'b100;
                 if(|RNUM) begin
                    RT <= `ENABLE;
                 end
              end
           end
           3'b100 : begin
              RT <= `DISABLE;
              if(ORDY) begin
                 if(|RNUM) begin
                    RNUM <= RNUM - 1;

                    OREQ <= 2'b01;
                    DOUT <= {HEADER, 1'b0, RET_NUM, fifo_addr_dout};
                    STATE <= 3'b010;
                 end
                 else begin
                    OREQ <= 2'b00;
                    DOUT <= 32'b0;
                    STATE <= 3'b000;
                 end
              end
           end
         endcase // casex(STATE)
      end // else: !if(RST)
   end // always@ (posedge CLK)

   fifo20x5_dr fifo_addr(.clk(CLK), .rst(RST),
                         .wr_en(fifo_addr_we),
                         .rd_en(fifo_addr_re),
                         .full(fifo_addr_fl),
                         .empty(fifo_addr_em),
                         .din(fifo_addr_din),
                         .dout(fifo_addr_dout),
                         .data_count(fifo_addr_dc));

   fifo32x5_dr fifo_data(.clk(CLK), .rst(RST),
                         .wr_en(fifo_data_we),
                         .rd_en(fifo_data_re),
                         .full(fifo_data_fl),
                         .empty(fifo_data_em),
                         .din(fifo_data_din),
                         .dout(fifo_data_dout),
                         .data_count(fifo_data_dc));

endmodule // no
