/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"

module demux2(CLK, 
              IREQ, IRDY, DIN, DOUT,
              OREQ0, ORDY0,
              OREQ1, ORDY1,
              RST);
   input CLK;
   input RST;

   input [1:0]    IREQ;
   output         IRDY;
   input  [31:0]  DIN;
   output [31:0]  DOUT;

   output [1:0]   OREQ0, OREQ1;
   input          ORDY0, ORDY1;

   reg       STATE;
   wire      ORDY;
   reg [1:0] ENC;
   reg [1:0] PND;

   wire FIFO_WE, FIFO_RE;
   wire FIFO_FULL;
   wire [`FIFO_WIDTH - 1 : 0] FIFO_DIN, FIFO_DOUT;

   assign OREQ0 = (STATE & ENC[0]) ? FIFO_DOUT[33:32] : 2'b00;
   assign OREQ1 = (STATE & ENC[1]) ? FIFO_DOUT[33:32] : 2'b00;

   assign ORDY = (ENC[0]) ? ORDY0 :
                 (ENC[1]) ? ORDY1 : `DISABLE;

   assign IRDY = ~FIFO_FULL;
   assign FIFO_WE  = |IREQ & ~FIFO_FULL;
   assign FIFO_DIN = {IREQ, DIN};
   assign FIFO_RE  = STATE & ORDY;

   bus_fifo fifo(.CLK(CLK), .RST(RST),
                 .WE(FIFO_WE),
                 .RE(FIFO_RE),
                 .FULL(FIFO_FULL),
                 .DIN(FIFO_DIN),
                 .DOUT(FIFO_DOUT));

   function [1:0] encode;
      input [0:0] addr;
      begin
         casex(addr)
           1'b0 : encode = 2'b01;
           1'b1 : encode = 2'b10;
           default : encode = 2'b0;
         endcase // casex(addr)
      end
   endfunction // encode

   assign DOUT = (FIFO_DOUT[33:32] == 2'b01) ? {FIFO_DOUT[31:20], FIFO_DOUT[18:0], FIFO_DOUT[19:19]} : FIFO_DOUT[31:0];

   always@(posedge CLK) begin
      if(RST) begin
         STATE <= 1'b0;
         ENC   <= 2'b0;
         PND   <= 2'b0;
      end
      else begin
         if(STATE & IREQ == 2'b01) PND <= encode(DIN[19:19]);
         if(~STATE) begin
            if(|IREQ) begin
               STATE <= 1'b1;
               ENC   <= encode(DIN[19:19]);
            end
         end
         else begin
            if(FIFO_DOUT[33:32] == 2'b10 & ORDY) begin
               if(IREQ == 2'b01) ENC   <= encode(DIN[19:19]);
               else if(IREQ[1])  ENC   <= PND;
               else              STATE <= 1'b0;
            end
         end
      end
   end
endmodule // demux2
