/**
 * SBML-SSAlib bridge: a bridge library to connect SSAlib and SBML
 *
 * Copyright (C) 2004-2009, SBML-SSA Team, Keio/Seikei University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *                    
 * @file    SBMLSSASolver.cpp
 * @brief   SBML-SSA Bridge
 * @author  Keio/Seikei SBML-SSA Team.
 */

#include "SBMLSSASolver.h"
#include "SBMLDtoSSAPConverter.h"
#include "SSAPtoSBMLDConverter.h"
#include "SSASolverLauncher.h"

/**
 * Constructor
 */
SBMLSSASolver::SBMLSSASolver()
 : mProblem(0)
{
  try
  {     
    // set default converters
    mSBMLDtoSSAPConverter = new SBMLDtoSSAPConverter();
    mSSAPtoSBMLDConverter = new SSAPtoSBMLDConverter();
  }
  catch(...)
  {
    delete mSBMLDtoSSAPConverter;   
    delete mSSAPtoSBMLDConverter;
    throw;
  }  
}


/**
 * Destructor
 */
SBMLSSASolver::~SBMLSSASolver()
{
  //delete mProblem;   
  delete mSBMLDtoSSAPConverter;   
  delete mSSAPtoSBMLDConverter;
}

/**
 * loadSBML by file
 */   
Problem*
SBMLSSASolver::loadSBML(const string &filename)
{
  SBMLReader sbmlReader;
  
  //if (mProblem) delete mProblem;

  SBMLDocument *sbmlDocument = sbmlReader.readSBML(filename);
  if (!validateSBML(sbmlDocument))
  {
    delete sbmlDocument;
    return 0;
  }     

  mProblem = mSBMLDtoSSAPConverter->convert(sbmlDocument);
  if (!mProblem)
  {
    mSBMLDtoSSAPConverter->printErrors();
  }
  
  delete sbmlDocument;
  
  return mProblem;
}


/**
 * loadProblem by Problem (for debugging purpose)
 */
Problem* 
SBMLSSASolver::loadProblem(Problem *problem)
{
  if (mProblem) delete mProblem;

  mProblem = problem;

  return mProblem;
}

/**
 * getProblem (for debugging purpose)
 */
Problem*
SBMLSSASolver::getProblem()
{
  return mProblem;
}


/**
 * getParticipant
 */
Participant* 
SBMLSSASolver::getParticipant(const string &participantName) 
{
  for (InitialCondition** icl = mProblem->initialConditions; *icl; ++icl)
  {
    Participant* curp = (*icl)->participant;
    if (curp && curp->name && string(curp->name) == participantName)
      return curp;
  }
  return 0;
}

/**
 * getNumParticipants
 */
int
SBMLSSASolver::getNumParticipants()
{
  int num =0;
  InitialCondition** icl = mProblem->initialConditions;
  while (*(icl++)) { ++num;}
  return num;
}

/**
 * getListOfParticipant
 */
vector<Participant*>
SBMLSSASolver::getListOfParticipant()
{
  vector<Participant*> lstofp;
  for (InitialCondition** icl = mProblem->initialConditions; *icl; ++icl)
  {
    lstofp.push_back((*icl)->participant);
  }

  // offset
  lstofp.push_back(0); 

  return vector<Participant*>(lstofp);
}


/**
 * saveSBML
 *
 * save an SBMLDocument which is normalized for ssaProblem
 */   
bool 
SBMLSSASolver::saveSBML(const string &filename, int level, int version)
{
  if (!mProblem) return false;

  SBMLDocument* sbmlDocument = mSSAPtoSBMLDConverter->convert(mProblem, level, version);
  if (!validateSBML(sbmlDocument))
  {
    delete sbmlDocument;
    return 0;
  }     

  SBMLWriter sbmlWriter;
  
  bool result = sbmlWriter.writeSBML(sbmlDocument, filename);
  delete sbmlDocument;

  return result;
}

/**
 * runSimulation
 */   
SSATrajectory*
SBMLSSASolver::runSimulation(SSASolverType_t solverType, SSATimeLine* timeLine)
{
   return SSASolverLauncher::runSimulation(solverType, mProblem, timeLine);
}

/**
 * getNumSolvers
 */   
int 
SBMLSSASolver::getNumSolvers() const
{
  return SSASolverLauncher::getNumSolvers();
}


/**
 * validateSBML (private member)
 */
bool 
SBMLSSASolver::validateSBML(SBMLDocument *sbmlDocument)
{
  if (!sbmlDocument) return false;

  if (sbmlDocument->getNumErrors() > 0)
  {
    sbmlDocument->printErrors();
    return false;
  }
  else
  {
    unsigned int numValidationErrors = 0;
    unsigned int numCheckFailures = sbmlDocument->checkConsistency();
    if ( numCheckFailures > 0 )
    {
      for (unsigned int i = 0; i < numCheckFailures; i++)
      {
        const SBMLError* sbmlErr = sbmlDocument->getError(i);
        if ( sbmlErr->isFatal() || sbmlErr->isError() )
        {
          ++numValidationErrors;
        }
      } 
    }

    if (numValidationErrors > 0) return false;
  }
  
  return true;
}
