/**
 * SBML-SSAlib bridge: a bridge library to connect SSAlib and SBML
 *
 * Copyright (C) 2004-2009, SBML-SSA Team, Keio/Seikei University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * @file    ASTNodeCalculator.cpp
 * @brief   Calculating ASTNode
 * @author  Keio/Seikei SBML-SSA Team.
 */

#include "ASTNodeCalculator.h"

double 
ASTNodeCalculator::evalMath(const ASTNode *math, const Model *model)
{
  mStack.clear();

  if (!math)
    throw invalid_argument("NULL ASTNode");
  else if (!model)
    throw invalid_argument("NULL Model");

  _evalMath(math,model,0);
  return mStack.back();
}

double 
ASTNodeCalculator::evalMathOfMassAction(const KineticLaw *kl, const Model *model)
{
  mStack.clear();

  if (!kl)
    throw invalid_argument("NULL KineticLaw");
  else if (!model)
    throw invalid_argument("NULL Model");

  //cout << "Math " << SBML_formulaToString(kl->getMath()) << endl;

  _evalMathOfMassAction(model,kl);

  return mStack.back();
}

void
ASTNodeCalculator::_evalMathOfMassAction(const Model *model, const KineticLaw *kl)
{
  _evalMath(kl->getMath(), model, kl);
}

void
ASTNodeCalculator::_evalMath(const ASTNode *node, const Model *model, const KineticLaw *kl)
{
  if (node)
  {
    _evalMath(node->getLeftChild(),  model, kl);
    _evalMath(node->getRightChild(), model, kl);

    string name;

    double d1, d2;
    switch (node->getType()) {
    case AST_INTEGER: 
      mStack.push_back( static_cast<double>(node->getInteger()) );
      //cout << "AST_INTEGER " << mStack.back() << endl;
      break;
    case AST_REAL: 
      mStack.push_back( static_cast<double>(node->getReal()) );
      //cout << "AST_REAL " << mStack.back() << endl;
      break;
    case AST_NAME:
      name.assign(node->getName());
      if (!kl)
        mStack.push_back(_getValueById(name, model));
      else
        mStack.push_back(_getValueById(name, kl, model));

      //cout << "AST_NAME " << mStack.back() << endl;
      break;
    case AST_PLUS:
      d2 = mStack.back();
      mStack.pop_back();
      d1 = mStack.back();
      mStack.pop_back();
      mStack.push_back(d1 + d2);
      break;
    case AST_MINUS:
      d2 = mStack.back();
      mStack.pop_back();
      d1 = mStack.back();
      mStack.pop_back();
      mStack.push_back(d1 - d2);
      break;
    case AST_TIMES:
      d2 = mStack.back();
      mStack.pop_back();
      d1 = mStack.back();
      mStack.pop_back();
      mStack.push_back(d1 * d2);
      break;
    case AST_DIVIDE:
      d2 = mStack.back();
      mStack.pop_back();
      d1 = mStack.back();
      mStack.pop_back();
      mStack.push_back(d1 / d2);
      break;
    case AST_FUNCTION_POWER:
    case AST_POWER:
      d2 = mStack.back();
      mStack.pop_back();
      d1 = mStack.back();
      mStack.pop_back();
      mStack.push_back(pow(d1,d2));
      break;
    default:
      throw invalid_argument("Unsupported ASTNode Type.");
    }
  }
}


double 
ASTNodeCalculator::_getValueById(const string &name, const Model *model)
{
  const InitialAssignment *ia = model->getInitialAssignment(name);
  if (ia)
  {
    ASTNodeCalculator ac;
    return ac.evalMath(ia->getMath(), model);
  }

  const Rule *rule = model->getRule(name);
  if (rule)
  {
    ASTNodeCalculator ac;
    return ac.evalMath(rule->getMath(), model);
  }

  const Parameter *gpara = model->getParameter(name);
  if (gpara)
    return gpara->getValue();

  const Species *sp = model->getSpecies(name);
  if (sp)
  {
    if (sp->isSetInitialAmount())
    {
      return sp->getInitialAmount();
    }
    else if (sp->isSetInitialConcentration())
    {
      return sp->getInitialConcentration();
    }

    throw invalid_argument("Initial value of Species (" + name + ") is undefined.");
  }

  const Compartment *cp = model->getCompartment(name);
  if (cp)
  {
    if (cp->isSetSize())
    {
      return cp->getSize();
    }

    throw invalid_argument("Initial value of Compartment (" + name + ") is undefined.");
  }

  throw invalid_argument("Initial value for " + name + " is undefined.");

  return 0;
}

double 
ASTNodeCalculator::_getValueById(const string &name, const KineticLaw *kl, const Model *model)
{
  if (!kl)    throw invalid_argument("NULL KineticLaw");
  if (!model) throw invalid_argument("NULL Model");

  const Species *sp = model->getSpecies(name);
  if (sp) return 1;

  const Compartment *cp = model->getCompartment(name);
  if (cp) return 1;

  const Parameter *lpara = kl->getParameter(name);
  if (lpara)
    return lpara->getValue();

  const InitialAssignment *ia = model->getInitialAssignment(name);
  if (ia)
  {
    ASTNodeCalculator ac;
    return ac.evalMath(ia->getMath(), model);
  }

  const Rule *rule = model->getRule(name);
  if (rule)
  {
    ASTNodeCalculator ac;
    return ac.evalMath(rule->getMath(), model);
  }

  const Parameter *gpara = model->getParameter(name);
  if (gpara)
    return gpara->getValue();


  throw invalid_argument("Initial value for " + name + " is undefined.");

  return 0;
}


