/**
 * SBML-SSAlib bridge: a bridge library to connect SSAlib and SBML
 *
 * Copyright (C) 2004-2009, SBML-SSA Team, Keio/Seikei University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * @file    ssaSBMLSolver.cpp
 * @brief   A simple command-line SSASBMLSolver program
 * @author  Keio/Seikei SBML-SSA Team.
 */

#include "SBMLSSASolver.h"

//
// Helper functions
//
void printPlot(SSATrajectory *tr, vector<Participant*> &ps, const SSATimeLine *tl);
vector<Participant*> getParticipantList(SBMLSSASolver &solver, const string &plist);

// usage
const string usage = "Usage: ssaSBMLSolver [-s start_time] [-e end_time] [-i interval] [-p p1[,p2,...]] filename\n";


int main(int argc, char** argv)
{
  string filename;

  //
  // default values
  //
  double start_time = 0;
  double end_time   = 100;
  double interval   = 0.1;
  string plottedParticipantList;

  //
  // parse command-line options
  //
  int c;
  while((c = getopt(argc, argv, "s:e:i:p:")) != -1) {
    switch (c) {
    case 's':
      start_time = atof(optarg);
      break;
    case 'e':
      end_time = atof(optarg);
      break;
    case 'i':
      interval = atof(optarg);
      break;
    case 'p':
      plottedParticipantList.assign(optarg);
      break;
    case 'h':
    case '?':
    default:
      cerr << usage;
      return 1;
    }
  }

  if (optind < argc)
    filename.assign(argv[optind]);
  else
  {
    cerr << usage;
    return 1;
  }

  cout << "-----------------------------------" << endl;
  cout << " ssaSBMLSolver                     " << endl;
  cout << "-----------------------------------" << endl;
  cout << "filename   : " << filename << endl;
  cout << "algorithm  : DirectLinear" << endl;
  cout << "-----------------------------------" << endl;
  cout << "simulation conditions    " << endl;
  cout << " Start time : " << start_time << endl;
  cout << " End time   : " << end_time   << endl;
  cout << " Interval   : " << interval   << endl;
  cout << "-----------------------------------" << endl;

  SBMLSSASolver solver;
  SSATimeLine       tl = {0, start_time, end_time, interval};
  SSATrajectory  *traj = 0;
  
  solver.loadSBML(filename);
  traj = solver.runSimulation(LIBSSA_DirectLinear, &tl);

  if (plottedParticipantList.empty())
  {
    vector<Participant*> plist = solver.getListOfParticipant();
    printPlot(traj, plist, &tl);
  }
  else
  {
    vector<Participant*> plist = getParticipantList(solver, plottedParticipantList);
    printPlot(traj, plist, &tl);
  }

  return 0;   
}

////////////////////////////////////////////////////////////////////////////////
// Helper functions
////////////////////////////////////////////////////////////////////////////////

/**
 * printPlot
 *
 * Creates a gnuplot-generated graph with the given SSATrajectory, participant list
 * and time line.
 */
void printPlot(SSATrajectory *tr, vector<Participant*> &ps, const SSATimeLine *tl)
{
  FILE* o = fopen("gp.d","wc");  
  trajectoryWritePts(o,tr);
  fclose(o);  
  trajectoryWriteGnuplotScript(tr,&ps[0], tl->start, tl->end,"time","population");
  system("./gp.r");
}


/**
 * getPariticipantList
 *
 * get the list of Participant objects corresponding to the given comma separated 
 * participant list 
 *
 */
vector<Participant*> getParticipantList(SBMLSSASolver &solver, const string &plist)
{
  vector<Participant*> vplist;

  size_t   cur_pos = 0;
  size_t comma_pos = 0;
  comma_pos = plist.find(',',cur_pos); 
  while (comma_pos != string::npos)
  {
    //cout << "cur_pos " << cur_pos << " comma_pos " << comma_pos << endl;
    if (cur_pos != comma_pos)
    {
      string pname = plist.substr(cur_pos, comma_pos - cur_pos);
      //cout << "plotted pname " << pname << endl;
      Participant *pp = solver.getParticipant(pname);
      if (pp) 
	vplist.push_back(pp);
    }
    cur_pos = comma_pos+1;
    comma_pos = plist.find(',',cur_pos);
  }

  if (cur_pos != comma_pos)
  {
    string pname = plist.substr(cur_pos, comma_pos - cur_pos);
    //cout << "plotted pname " << pname << endl;
    Participant *pp = solver.getParticipant(pname);
    if (pp) 
      vplist.push_back(pp);
  }

  // terminating
  vplist.push_back(0);

  return vector<Participant*>(vplist);
}
