/*
 * <<< memory_control_unit_test.cc >>>
 *
 * --- Test program for memory_control_unit_test class
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstddef>
#include <cstdlib>
#include <iostream>
#include <isis/bus_port.h>
#include <isis/mapped_memory.h>
#include "memory_control_unit.h"

typedef unsigned int word;

static void single_read_test(mapped_memory<word, word>&,
							 mapped_memory<word, word>&, 
							 memory_control_unit<word>&, bus_port<word>&, bool);
static void single_write_test(mapped_memory<word, word>&,
							  mapped_memory<word, word>&, 
							  memory_control_unit<word>&, bus_port<word>&,
							  bool);
static void multi_read_test(mapped_memory<word, word>&,
							mapped_memory<word, word>&, 
							memory_control_unit<word>&, bus_port<word>&, bool,
							size_t);
static void multi_write_test(mapped_memory<word, word>&,
							 mapped_memory<word, word>&, 
							 memory_control_unit<word>&, bus_port<word>&, bool,
							 size_t);

int main(int, char** argv)
{
	const word top = 0, size = 0x20;
	const int multi_count = 4;
	int split_flag = 0, wait_count = 0;
	while (*++argv != NULL && **argv == '-') {
		switch (*++*argv) {
		case 's':
			split_flag = 1;
			break;
		case 'w':
			wait_count = atoi(*argv + 1);
			break;
		}
	}
	bus_port<word> bus_if;
	mapped_memory<word, word> mem_a(top, size), mem_b(top, size);
	memory_control_unit<word> mem_ctl;
	mem_ctl.port_ref().connect(bus_if);
	mem_ctl.connect_memory(mem_a);
	mem_ctl.set_read_wait(wait_count);
	mem_ctl.set_write_wait(wait_count);
	if (split_flag) mem_ctl.enable_split_transaction();

	cout << "=== single read test" << endl;
	cerr << "=== single read test" << endl;
	single_read_test(mem_a, mem_b, mem_ctl, bus_if, split_flag);
	cout << "src:" << endl << mem_a << endl
		 << "dst:" << endl << mem_b << endl;

	cout << "=== single write test" << endl;
	cerr << "=== single write test" << endl;
	single_write_test(mem_a, mem_b, mem_ctl, bus_if, split_flag);
	cout << "src:" << endl << mem_b << endl
		 << "dst:" << endl << mem_a << endl;

	cout << "=== multi read test" << endl;
	cerr << "=== multi read test" << endl;
	multi_read_test(mem_a, mem_b, mem_ctl, bus_if, split_flag, multi_count);
	cout << "src:" << endl << mem_b << endl
		 << "dst:" << endl << mem_a << endl;

	cout << "=== multi write test" << endl;
	cerr << "=== multi write test" << endl;
	multi_write_test(mem_a, mem_b, mem_ctl, bus_if, split_flag, multi_count);
	cout << "src:" << endl << mem_b << endl
		 << "dst:" << endl << mem_a << endl;

	return 0;
}

void single_read_test(mapped_memory<word, word>& mem_a,
					  mapped_memory<word, word>& mem_b, 
					  memory_control_unit<word>& mem_ctl,
					  bus_port<word>& bus_if, bool split_flag)
{
	word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_a.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		if (split_flag) {
			bus_if.request_ownership();
			bus_if.is_owner();
		}
		bus_if.send_single_read_request(adr);
		int flag = 0;
		do {
			cerr << "in:  " << bus_if << ", " << mem_ctl << endl;
			mem_ctl.clock_in();
			if (bus_if.is_single_read_data()) {
				mem_b.write(adr, bus_if.data());
				flag = 1;
			} else if (bus_if.is_single_read_grant()) {
				bus_if.release_ownership();
				bus_if.clear();
			}
			cerr << "out: " << bus_if << ", " << mem_ctl << endl;
			mem_ctl.clock_out();
		} while (!flag);
		bus_if.clear();
		if (split_flag) {
			bus_if.release_ownership();
			bus_if.is_owner();
		}
	}
}

void single_write_test(mapped_memory<word, word>& mem_a,
					   mapped_memory<word, word>& mem_b, 
					   memory_control_unit<word>& mem_ctl,
					   bus_port<word>& bus_if, bool split_flag)
{
	word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_b.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		if (split_flag) {
			bus_if.request_ownership();
			bus_if.is_owner();
		}
		bus_if.send_single_write_request(adr);
		int flag = 0;
		do {
			cerr << "in:  " << bus_if << ", " << mem_ctl << endl;
			mem_ctl.clock_in();
			if (bus_if.is_single_write_grant()) {
				bus_if.release_ownership();
				bus_if.clear();
			}
			flag = bus_if.is_single_write_data();
			int ack_flag = bus_if.is_single_write_ack();
			cerr << "out: " << bus_if << ", " << mem_ctl << endl;
			if (ack_flag) {
				bus_if.send_single_write_data(mem_b.read(adr));
			}
			if (flag) bus_if.clear();
			mem_ctl.clock_out();
		} while (!flag);
		if (split_flag) {
			bus_if.release_ownership();
			bus_if.is_owner();
		}
	}
}

void multi_read_test(mapped_memory<word, word>& mem_a,
					 mapped_memory<word, word>& mem_b, 
					 memory_control_unit<word>& mem_ctl,
					 bus_port<word>& bus_if, bool split_flag,
					 size_t multi_count)
{
	word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_a.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word) * multi_count) {
		if (split_flag) {
			bus_if.request_ownership();
			bus_if.is_owner();
		}
		bus_if.send_multi_read_request(adr, multi_count);
		int flag = 0;
		do {
			cerr << "in:  " << bus_if << ", " << mem_ctl << endl;
			mem_ctl.clock_in();
			if (bus_if.is_multi_read_data()) {
				mem_b.write(adr + bus_if.packet_number() * sizeof(word),
					bus_if.data());
				if (bus_if.packet_number() == multi_count - 1) {
					flag = 1;
				}
			} else if (bus_if.is_multi_read_grant()) {
				bus_if.release_ownership();
				bus_if.clear();
			}
			cerr << "out: " << bus_if << ", " << mem_ctl << endl;
			mem_ctl.clock_out();
		} while (!flag);
		bus_if.clear();
		if (split_flag) {
			bus_if.release_ownership();
			bus_if.is_owner();
		}
	}
}

void multi_write_test(mapped_memory<word, word>& mem_a,
					  mapped_memory<word, word>& mem_b, 
					  memory_control_unit<word>& mem_ctl,
					  bus_port<word>& bus_if, bool split_flag,
					  size_t multi_count)
{
	word size = mem_a.size();
	mem_a.clear();
	mem_b.clear();
	for (word adr = 0; adr < size; adr += sizeof(word)) {
		mem_b.write(adr, adr);
	}
	for (word adr = 0; adr < size; adr += sizeof(word) * multi_count) {
		bus_if.send_multi_write_request(adr, multi_count);
		bool flag = false;
		size_t i = 0;
		do {
			cerr << "in:  " << bus_if << ", " << mem_ctl << endl;
			mem_ctl.clock_in();
			if (bus_if.is_multi_write_grant()) {
				bus_if.release_ownership();
				bus_if.clear();
			}
			flag = (i == multi_count);
			bool ack_flag =
				(bus_if.is_multi_write_ack() ||
				 (i > 0 && i < multi_count));
			cerr << "out: " << bus_if << ", " << mem_ctl << endl;
			if (ack_flag) {
				bus_if.send_multi_write_data(
					mem_b.read(adr + sizeof(word) * i), multi_count, i);
				i++;
			}
			if (flag) bus_if.clear();
			mem_ctl.clock_out();
		} while (!flag);
		if (split_flag) {
			bus_if.release_ownership();
			bus_if.is_owner();
		}
	}
}
