/*
 * <<< r3010_rd.cc >>>
 *
 * --- Copyright (C) 1996-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * current state:
 *
 * 1996/12/5 6:30  ȡ⡼ɤʤ
 *				 ä cfc1 $r, $f31 ˤ
 */

#include <cstring> // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include "r3010_fgr.h"
#include "r3010_fgrbuf.h"
#include "r3010_fcr.h"
#include "r3010_forward.h"
#include "r3010_forwarder.h"
#include "r3010_inst.h"
#include "r3010_stage.h"
#include "r3010_add.h"
#include "r3010_ext.h"
#include "r3010.h"
#include "r3010_debug.h"

bool
r3010_stage::have_inst() const
{
	return inst.have_inst();
}

void
r3010_stage::instruction_fetch(r3000_word w)
{
	inst.decode(w);
	idx[r3010_inst::REG_SRC1] = inst.src( r3010_inst::REG_SRC1 );
	idx[r3010_inst::REG_SRC2] = inst.src( r3010_inst::REG_SRC2 );
	idx[r3010_inst::REG_DST] = inst.dst();
}

void
r3010_stage::rd_clock()
{
	char *debug = "";
	// bool flag;
	// int num;

	switch( rd_state ) {
	  case rd_none:
		if( ! inst.have_inst() ) {
			debug = "Not have instruction, yet";
			break;
		}
		rd_state = rd_wait;

	  case rd_wait:
		if( fpa->s_alu ) {
			debug = "predecessor is stalling.";
			break;
		}
		if( inst.is_calc() ) {
			bool go_next = true;

			switch( inst.func() ) {
			  case r3010_inst::FUNC_ADD:
			  case r3010_inst::FUNC_SUB:
			  case r3010_inst::FUNC_ABS:
			  case r3010_inst::FUNC_MOV:
			  case r3010_inst::FUNC_NEG:
			  case r3010_inst::FUNC_CVT_S:
			  case r3010_inst::FUNC_CVT_D:
			  case r3010_inst::FUNC_CVT_W:
				if( fpa->ex_add().in_use() ) {
					debug = "can't use adder";
					go_next = false;
				}
				break;
			  case r3010_inst::FUNC_MUL:
				if( fpa->ex_mul().in_use() ) {
					debug = "can't use multiplyer";
					go_next = false;
				}
				else if( fpa->ex_add().in_use() ) {
					debug = "can't use adder";
					go_next = false;
				}
				break;
			  case r3010_inst::FUNC_DIV:
				if( fpa->ex_div().in_use() ) {
					debug = "can't use divider";
					go_next = false;
				}
				else if( fpa->ex_add().in_use() ) {
					debug = "can't use adder";
					go_next = false;
				}
				break;
			  default:
				if( inst.func() >= r3010_inst::FUNC_C_F &&
					inst.func() <= r3010_inst::FUNC_C_NGT ) {
					if( fpa->ex_add().in_use() ) {
						debug = "Can't use adder";
						go_next = false;
					}
				}
			}

			if( !go_next )
				break;
		}
		rd_state = rd_read_data;

	  case rd_read_data:
		if (inst.use_src(r3010_inst::REG_SRC1)) {
			if (!get_data(idx[r3010_inst::REG_SRC1],
						  reg[r3010_inst::REG_SRC1])) {
				debug = "Can't get data for src1";
				break;
			}
			valid[r3010_inst::REG_SRC1] = true;
		}
		if (inst.use_src(r3010_inst::REG_SRC2)) {
			if (!get_data(idx[r3010_inst::REG_SRC2],
						  reg[r3010_inst::REG_SRC2])) {
				debug = "Can't get data for src2";
				break;
			}
			valid[r3010_inst::REG_SRC2] = true;
		}
		rd_state = rd_lock_reg;

	  case rd_lock_reg:
		if( inst.use_dst() ) {
			if( inst.is_calc() ) {
				if (!tas(idx[r3010_inst::REG_DST])) {
					debug = "Can't reserve dst";
					break;
				}
				need_release = true;
			}
		}

		if( (inst.inst() & 0xffe0f800) == 0x4440f800 /* CFC1 $r,$31 */ ) {
			need_wait = true;
		}

		rd_state = rd_wait_stall;

	  case rd_wait_stall:
		if( need_wait ) {
			if( rd_have_calc_predecessor() ) {
				debug = "wait to finish predecessor calculations";
				break;
			}
		}
		if( rd_have_stall_mode_predecessor() ) {
			debug = "there is stall mode instruction";
			break;
		}

		rd_get_csr();
		rd_go_next(); // advance stage
		break;

	  default:
		cerr << "R3010: RD: illegal state" << endl;
		break;
	}

#ifdef R3010_DEBUG
	if( fpa->debug_level() & R3010_DEBUG_STALL )
		cerr << "RD:" << debug << endl;
#endif // R3010_DEBUG
}

bool
r3010_stage::is_locked( int regnum )
{
	return fpa->lock_reg_p( regnum );
}

bool
r3010_stage::get_data( int regnum, r3010_fgr& regval )
{
	r3010_forward f0, *f1, *f2;
	bool f = false;

	f0.regnum = regnum;

	if( is_locked( regnum ) ) {
#ifdef R3010_DEBUG
		if( fpa->debug_level() & R3010_DEBUG_STALL )
			cerr << "RD:get_data:reg #" << regnum << " is locked."
				<< endl;
#endif // R3010_DEBUG
		return false;
	}
	else if( (f1 = fpa->forward->get(f0)) != 0 ) {
#ifdef R3010_DEBUG
		if( fpa->debug_level() & R3010_DEBUG_FORWARDING )
			cerr << "RD:get_data:get from forwarding path "
				 << regnum << endl;
#endif // R3010_DEBUG
		regval.write_word( regnum%2, f1->val.read_word( regnum%2 ) );

		f = true;
	}
	else if( regnum >= NFGR )
		regval.write_word(regnum - NFGR, fpa->fcr().read_word(regnum - NFGR));
	else
		regval = fpa->fgr().reg( regnum );

	if( inst.is_calc() ) {
		f0.regnum = regnum ^ 1;
		if( is_locked( regnum^1 ) ) {
#ifdef R3010_DEBUG
			if( fpa->debug_level() & R3010_DEBUG_STALL )
				cerr << "RD:get_data:reg #" << (regnum^1) << " is locked."
					 << endl;
#endif // R3010_DEBUG
			return false;
		}
		else if( (f2 = fpa->forward->get(f0)) != 0 ) {
#ifdef R3010_DEBUG
			if( fpa->debug_level() & R3010_DEBUG_FORWARDING )
				cerr << "RD:get_data:get from forwarding path "
					 << (regnum^1) << endl;
#endif // R3010_DEBUG
			regval.write_word((regnum%2)^1, f2->val.read_word((regnum%2)^1));
			return true;
		}
		else if( f && regnum >= NFGR )
			regval.write_word((regnum%2)^1,
							  fpa->fcr().read_word((regnum - NFGR)^1));
		else
			regval.write_word((regnum%2)^1, fpa->fgr().read_word(regnum^1));
	}

	return true;
}

bool
r3010_stage::tas( int regnum )
{
	if( is_locked( regnum ) )
		return false;

	if( inst.is_calc() ) {
		if( is_locked( regnum^1 ) )
			return false;

		fpa->lock_reg( regnum^1 );
	}
	fpa->lock_reg( regnum );

	return true;
}

void
r3010_stage::rd_get_csr()
{
	r3010_forward f0, *f1;

	f0.regnum = NFGR + R3010_CSR;
	if( (f1 = fpa->forward->get(f0)) != 0 )
		csr.write_word( f1->val.read_word( R3010_CSR ) );
	else
		csr = fpa->csr();
}

void
r3010_stage::rd_go_next()
{
	fpa->s_alu = this;
	fpa->s_rd = 0;

	stage = alu_stage;
}

bool
r3010_stage::rd_have_calc_predecessor()
{
	if( fpa->s_alu )
		return fpa->s_alu->inst.is_calc();
	if( fpa->s_mem )
		return fpa->s_mem->inst.is_calc();
	if( fpa->s_wb )
		return fpa->s_wb->inst.is_calc();
	if( fpa->s_fwb )
		return fpa->s_fwb->inst.is_calc();

	if( fpa->ex_add().in_use() )
		return true;
	if( fpa->ex_mul().in_use() )
		return true;
	if( fpa->ex_div().in_use() )
		return true;

	return false;
}

bool
r3010_stage::rd_have_stall_mode_predecessor()
{
	if( fpa->s_alu && fpa->s_alu->need_stall )
		return true;
	if( fpa->s_mem && fpa->s_mem->need_stall )
		return true;
	if( fpa->s_wb && fpa->s_wb->need_stall )
		return true;
	if( fpa->s_fwb && fpa->s_fwb->need_stall )
		return true;
	if( fpa->ex_add().in_use() && fpa->ex_add().is_stall_mode() )
		return true;
	if( fpa->ex_mul().in_use() && fpa->ex_mul().is_stall_mode() )
		return true;
	if( fpa->ex_div().in_use() && fpa->ex_div().is_stall_mode() )
		return true;

	return false;
}
