/* -*- C++ -*-
 *
 * <<< bus_packet_base.h >>>
 *
 * --- Abstract bus packet class 'bus_packet_base'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef BUS_PACKET_BASE_H
#define BUS_PACKET_BASE_H 1

#include <cstddef>
#include <isis/packet.h>

template <class A, class D = A>
class bus_packet_base : public packet
{
public:
	typedef A address_type;
	typedef D data_type;
private:
	typedef bus_packet_base<A, D> thisclass;
	typedef packet inherited;
	address_type address_;
	data_type data_;
	size_t total_packet_count_, packet_number_;
public:
	bus_packet_base(void) {}
	virtual ~bus_packet_base() {}
	address_type address(void) const { return address_; }
	address_type& address(void) { return address_; }
	data_type data(void) const { return data_; }
	data_type& data(void) { return data_; }
	size_t total_packet_count(void) const { return total_packet_count_; }
	size_t packet_number(void) const { return packet_number_; }
	bool is_single(void) const { return total_packet_count() == 1; }
	bool is_multi(void) const { return total_packet_count() > 1; }
	void set_single(void) { total_packet_count_ = 1; }
	void set_multi(void) { if (!is_multi()) set_multi(2); }
	void set_multi(size_t a) { total_packet_count_ = a; }
	void set_multi(size_t a, size_t b)
		{ total_packet_count_ = a, packet_number_ = b; }
	// basic virtual functions: define these functions in inherited class
	virtual bool is_read(void) const = 0;
	virtual bool is_write(void) const = 0;
	virtual bool is_request(void) const = 0;
	virtual bool is_grant(void) const = 0;
	virtual bool is_ack(void) const = 0;
	virtual bool is_nack(void) const = 0;
	virtual bool is_data(void) const = 0;
	virtual void set_read(void) = 0;
	virtual void set_write(void) = 0;
	virtual void set_request(void) = 0;
	virtual void set_grant(void) = 0;
	virtual void set_ack(void) = 0;
	virtual void set_nack(void) = 0;
	virtual void set_data(void) = 0;
	// useful virtual functions: define for performance
	// is_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_read_request(void) const
		{ return is_request() && is_read(); }
	virtual bool is_read_grant(void) const
		{ return is_grant() && is_read(); }
	virtual bool is_read_ack(void) const
		{ return is_ack() && is_read(); }
	virtual bool is_read_nack(void) const
		{ return is_nack() && is_read(); }
	virtual bool is_read_data(void) const
		{ return is_data() && is_read(); }
	virtual bool is_write_request(void) const
		{ return is_request() && is_write(); }
	virtual bool is_write_grant(void) const
		{ return is_grant() && is_write(); }
	virtual bool is_write_ack(void) const
		{ return is_ack() && is_write(); }
	virtual bool is_write_nack(void) const
		{ return is_nack() && is_write(); }
	virtual bool is_write_data(void) const
		{ return is_data() && is_write(); }
	// is_single_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_single_read_request(void) const
		{ return is_single() && is_read_request(); }
	virtual bool is_single_read_grant(void) const
		{ return is_single() && is_read_ack(); }
	virtual bool is_single_read_ack(void) const
		{ return is_single() && is_read_ack(); }
	virtual bool is_single_read_nack(void) const
		{ return is_single() && is_read_nack(); }
	virtual bool is_single_read_data(void) const
		{ return is_single() && is_read_data(); }
	virtual bool is_single_write_request(void) const
		{ return is_single() && is_write_request(); }
	virtual bool is_single_write_grant(void) const
		{ return is_single() && is_write_grant(); }
	virtual bool is_single_write_ack(void) const
		{ return is_single() && is_write_ack(); }
	virtual bool is_single_write_nack(void) const
		{ return is_single() && is_write_nack(); }
	virtual bool is_single_write_data(void) const
		{ return is_single() && is_write_data(); }
	// is_multi_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_multi_read_request(void) const
		{ return is_multi() && is_read_request(); }
	virtual bool is_multi_read_grant(void) const
		{ return is_multi() && is_read_grant(); }
	virtual bool is_multi_read_ack(void) const
		{ return is_multi() && is_read_ack(); }
	virtual bool is_multi_read_nack(void) const
		{ return is_multi() && is_read_nack(); }
	virtual bool is_multi_read_data(void) const
		{ return is_multi() && is_read_data(); }
	virtual bool is_multi_write_request(void) const
		{ return is_multi() && is_write_request(); }
	virtual bool is_multi_write_grant(void) const
		{ return is_multi() && is_write_grant(); }
	virtual bool is_multi_write_ack(void) const
		{ return is_multi() && is_write_ack(); }
	virtual bool is_multi_write_nack(void) const
		{ return is_multi() && is_write_nack(); }
	virtual bool is_multi_write_data(void) const
		{ return is_multi() && is_write_data(); }
	// set_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_read_request(void)
		{ set_request(), set_read(); }
	virtual void set_read_request(address_type a)
		{ set_read_request(), address() = a; }
	virtual void set_read_grant(void)
		{ set_grant(), set_read(); }
	virtual void set_read_ack(void)
		{ set_ack(), set_read(); }
	virtual void set_read_nack(void)
		{ set_nack(), set_read(); }
	virtual void set_read_data(void)
		{ set_data(), set_read(); }
	virtual void set_read_data(data_type a)
		{ set_read_data(), data() = a; }
	virtual void set_write_request(void)
		{ set_request(), set_write(); }
	virtual void set_write_request(address_type a)
		{ set_write_request(), address() = a; }
	virtual void set_write_grant(void)
		{ set_grant(), set_write(); }
	virtual void set_write_ack(void)
		{ set_ack(), set_write(); }
	virtual void set_write_nack(void)
		{ set_nack(), set_write(); }
	virtual void set_write_data(void)
		{ set_data(), set_write(); }
	virtual void set_write_data(data_type a)
		{ set_write_data(), data() = a; }
	// set_single_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_single_read_request(void)
		{ set_single(), set_read_request(); }
	virtual void set_single_read_request(address_type a)
		{ set_single(), set_read_request(a); }
	virtual void set_single_read_grant(void)
		{ set_single(), set_read_grant(); }
	virtual void set_single_read_ack(void)
		{ set_single(), set_read_ack(); }
	virtual void set_single_read_nack(void)
		{ set_single(), set_read_nack(); }
	virtual void set_single_read_data(void)
		{ set_single(), set_read_data(); }
	virtual void set_single_read_data(data_type a)
		{ set_single(), set_read_data(a); }
	virtual void set_single_write_request(void)
		{ set_single(), set_write_request(); }
	virtual void set_single_write_request(address_type a)
		{ set_single(), set_write_request(a); }
	virtual void set_single_write_grant(void)
		{ set_single(), set_write_grant(); }
	virtual void set_single_write_ack(void)
		{ set_single(), set_write_ack(); }
	virtual void set_single_write_nack(void)
		{ set_single(), set_write_nack(); }
	virtual void set_single_write_data(void)
		{ set_single(), set_write_data(); }
	virtual void set_single_write_data(data_type a)
		{ set_single(), set_write_data(a); }
	// set_multi_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_multi_read_request(void)
		{ set_multi(), set_read_request(); }
	virtual void set_multi_read_request(address_type a, size_t b)
		{ set_multi(b), set_read_request(a); }
	virtual void set_multi_read_grant(void)
		{ set_multi(), set_read_grant(); }
	virtual void set_multi_read_ack(void)
		{ set_multi(), set_read_ack(); }
	virtual void set_multi_read_nack(void)
		{ set_multi(), set_read_nack(); }
	virtual void set_multi_read_data(void)
		{ set_multi(), set_read_data(); }
	virtual void set_multi_read_data(data_type a, size_t b, size_t c)
		{ set_multi(b, c), set_read_data(a); }
	virtual void set_multi_write_request(void)
		{ set_multi(), set_write_request(); }
	virtual void set_multi_write_request(address_type a, size_t b)
		{ set_multi(b), set_write_request(a); }
	virtual void set_multi_write_grant(void)
		{ set_multi(), set_write_grant(); }
	virtual void set_multi_write_ack(void)
		{ set_multi(), set_write_ack(); }
	virtual void set_multi_write_nack(void)
		{ set_multi(), set_write_nack(); }
	virtual void set_multi_write_data(void)
		{ set_multi(), set_write_data(); }
	virtual void set_multi_write_data(data_type a, size_t b, size_t c)
		{ set_multi(b, c), set_write_data(a); }
};

#endif /* BUS_PACKET_BASE_H */
