/* -*- C++ -*-
 *
 * <<< set_associative_cache.h >>>
 *
 * --- Set-associative cache class 'set_associative_cache'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef SET_ASSOCIATIVE_CACHE_H
#define SET_ASSOCIATIVE_CACHE_H 1

#ifdef DEBUG
# include <cassert>
#endif // DEBUG
#include <iostream>
#include <iomanip>
#include <vector>
#include <isis/root_object.h>
#include <isis/cache_line_set.h>

template <class L>
class set_associative_cache : public root_object
{
public:
	typedef typename L::address_type address_type;
	typedef typename L::data_type data_type;
	typedef L line_type;
	typedef cache_line_set<L> line_set_type;
private:
	typedef set_associative_cache<L> thisclass;
	typedef root_object inherited;
	typedef vector<line_set_type> container_type;
	static const int default_max_lru_counter = 1;
	container_type buf;
	address_type line_size_, cache_size_;
	size_t upper_shift, lower_shift;
	address_type upper_mask, lower_mask, tag_mask;
	size_t index_address(address_type adr) const
		{ return size_t((adr & upper_mask) >> upper_shift); }
	size_t line_address(address_type adr) const
		{ return size_t((adr & lower_mask) >> lower_shift); }
public:
	set_associative_cache(void);
	explicit set_associative_cache(address_type, address_type, size_t);
	set_associative_cache(const thisclass&);
	virtual ~set_associative_cache();
	virtual void output(ostream&) const;
	const line_set_type& cache_line_set(address_type a) const
		{ return buf[index_address(a)]; }
	line_set_type& cache_line_set(address_type a)
		{ return buf[index_address(a)]; }
	address_type cache_size(void) const { return cache_size_; }
	address_type line_size(void) const { return line_size_; }
	size_t way_size(void) const { return (buf.size() > 0) ? buf[0].size() : 0; }
	size_t line_size_in_word(void) const
		{ return (buf.size() > 0) ? buf[0].line_size() : 0; }
	int max_lru_counter(void)
		{ return (buf.size() > 0) ?
			buf[0].max_lru_counter() : default_max_lru_counter; }
	address_type tag_address(address_type adr) const
		{ return adr & tag_mask; }
	size_t line_offset(address_type adr) const
		{ return size_t(line_address(adr)); }
	void clear(void);
	void resize(address_type, address_type, size_t);
	void set_max_lru_counter(int);
};

template <class L>
set_associative_cache<L>::set_associative_cache(void)
{
	resize(0, 0, 0);
}

template <class L>
set_associative_cache<L>::set_associative_cache
	(typename set_associative_cache<L>::address_type a,
	 typename set_associative_cache<L>::address_type b, size_t c)
{
	resize(a, b, c);
}

template <class L>
set_associative_cache<L>::set_associative_cache
	(const set_associative_cache<L>& a)
	: buf(a.buf),
	  line_size_(a.line_size_),
	  cache_size_(a.cache_size_),
	  upper_shift(a.upper_shift),
	  lower_shift(a.lower_shift),
	  upper_mask(a.upper_mask),
	  lower_mask(a.lower_mask),
	  tag_mask(a.tag_mask)
{}

template <class L>
set_associative_cache<L>::~set_associative_cache()
{}

template <class L>
void set_associative_cache<L>::set_max_lru_counter(int a)
{
	for (size_t i = 0; i < buf.size(); i++) buf[i].set_max_lru_counter(a);
}

template <class L>
void set_associative_cache<L>::resize
	(typename set_associative_cache<L>::address_type cache_sz,
	 typename set_associative_cache<L>::address_type line_sz,
	 size_t way_sz)
{
#ifdef DEBUG
	assert((cache_sz > 0 || cache_sz == 0) && (line_sz > 0 || line_sz == 0));
#endif // DEBUG
	if (cache_sz == 0 || line_sz == 0 || way_sz == 0 || cache_sz < way_sz) {
		cache_size_ = line_size_ = 0;
		buf.resize(0);
		return;
	}
	size_t max_lru_counter_ = max_lru_counter();
	{
		size_t i = 0;
		for (size_t tmp = line_sz; tmp > 1; tmp >>= 1) i++;
		line_size_ = (1 << i);
		upper_shift = i;
	}
	{
		size_t i = 0;
		for (size_t tmp = (cache_sz / way_sz) >> upper_shift;
			 tmp > 1; tmp >>= 1) i++;
		cache_size_ = (1 << i) << upper_shift;
		upper_mask = (~(~address_type(0) << i) << upper_shift);
	}
	{
		size_t i = 0;
		for (size_t tmp = sizeof(data_type); tmp > 1; tmp >>= 1) i++;
		lower_shift = i;
		lower_mask = (~(~address_type(0) << (upper_shift - lower_shift))
					  << lower_shift);
	}
	tag_mask = (~address_type(0) << upper_shift);
	buf.resize(size_t(cache_size_ / line_size_));
	for (size_t i = 0; i < buf.size(); i++) {
		buf[i].resize(way_sz);
		buf[i].line_resize(line_size_ >> lower_shift);
		buf[i].set_max_lru_counter(max_lru_counter_);
	}
	clear();
}

template <class L>
void set_associative_cache<L>::clear(void)
{
	for (size_t i = 0; i < buf.size(); i++) buf[i].clear();
}

template <class L>
void set_associative_cache<L>::output(ostream& os) const
{
	if (cache_size() == 0 || line_size() == 0 || way_size() == 0) return;
	size_t width;
	{
		size_t tmp = buf.size() - 1;
		int i = 1;
		while (tmp >= 16) tmp >>= 4, i++;
		width = i;
	}
	const char fill = os.fill();
	const long flags = os.flags();
	os << hex << setfill('0');
	for (size_t i = 0; i < buf.size(); i++) {
		for (size_t j = 0; j < buf[0].size(); j++) {
			if (j == 0) {
				os << '[' << setw(width) << i << "] ";
			} else {
				for (size_t k = 0; k < width + 3; k++) os << ' ';
			}
			os << buf[i][j];
			if (buf[i][j].is_valid()) os << ' ' << buf[i].lru_counter(j);
			if (j < buf[0].size() - 1) os << '\n';
		}
		if (i < buf.size() - 1) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* SET_ASSOCIATIVE_CACHE_H */
