/* -*- C++ -*-
 *
 * <<< debugger.h >>>
 *
 * --- Debugger class 'debugger'
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef DEBUGGER_H
#define DEBUGGER_H 1

#include <algorithm>
#include <functional>
#include <iostream>
#include <vector>
#include "synchronous_unit.h"
#include "single_processing_element.h"
#include "multiprocessor.h"

template <class T>
class debugger
{
public:
	typedef debugger<T> thisclass;
	typedef T processing_element_type;
	typedef typename processing_element_type::address_type address_type;
	typedef typename processing_element_type::data_type data_type;
	typedef typename processing_element_type::char_type char_type;
	typedef typename processing_element_type::size_type size_type;
	typedef typename processing_element_type::processor_type processor_type;
	typedef typename processing_element_type::memory_map_type memory_map_type;
	typedef multiprocessor<processing_element_type> multiprocessor_type;
	static const size_t sizeof_data_type
		= processing_element_type::sizeof_data_type;
private:
	typedef vector<processing_element_type*>
		processing_element_pointer_container_type;
	typedef vector<address_type> breakpoint_container_type;
	synchronous_unit* sync_unit;
	multiprocessor_type* multiprocessor_unit;
	processing_element_pointer_container_type pe_buf;
	breakpoint_container_type breakpoint_buf;
	size_t curr_pe_id;
	processing_element_type* curr_pe;
protected:
	const processing_element_type& current_pe(void) const { return *curr_pe; }
	processing_element_type& current_pe(void) { return *curr_pe; }
public:
	debugger(void);
	debugger(const thisclass&);
	virtual ~debugger();
	// setup target
	void set(processing_element_type&);
	void set(multiprocessor_type&);
	void unset(void);
	// check setting
	bool is_valid(void) const { return sync_unit != NULL; }
	size_t processor_number(void) const { return pe_buf.size(); }
	size_t current_processor_id(void) const { return curr_pe_id; }
	// check state
	inline bool is_halt(void) const;
	inline bool is_bus_error(void) const;
	// execute
	inline void step(void);
	void execute(void);
	// reset
	void reset(void);
	// breakpoint
	void insert_breakpoint(const address_type&);
	void erase_breakpoint(const address_type&);
	void clear_breakpoints(void);
	template <class Cont> void dump_breakpoints(Cont&) const;
	// memory access
	bool load(const char*);
	void dump(ostream&, const address_type&, const size_type&) const;
	void disassemble(ostream&, const address_type&, const size_type&) const;
};

template <class T>
debugger<T>::debugger(void)
	: sync_unit(NULL),
	  multiprocessor_unit(NULL)
{}

template <class T>
debugger<T>::debugger(const debugger<T>& a)
	: sync_unit(a.sync_unit),
	  multiprocessor_unit(a.multiprocessor_unit),
	  pe_buf(a.pe_buf),
	  breakpoint_buf(a.breakpoint_buf),
	  curr_pe_id(a.curr_pe_id),
	  curr_pe(&pu_table[curr_pe_id])
{}

template <class T>
debugger<T>::~debugger()
{}

template <class T>
void debugger<T>::set(debugger<T>::processing_element_type& a)
{
	pe_buf.resize(1);
	pe_buf[0] = &a;
	sync_unit = &a;
	multiprocessor_unit = NULL;
	curr_pe_id = 0;
	curr_pe = pe_buf[0];
}

template <class T>
void debugger<T>::set(debugger<T>::multiprocessor_type& a)
{
	pe_buf.resize(a.processor_number());
	for (size_t i = 0; i < a.processor_number(); i++) {
		pe_buf[i] = &(a.processing_element(i));
	}
	sync_unit = &a;
	multiprocessor_unit = &a;
	curr_pe_id = 0;
	curr_pe = pe_buf[0];
}

template <class T>
void debugger<T>::unset(void)
{
	sync_unit = NULL;
	multiprocessor_unit = NULL;
}

template <class T>
inline bool debugger<T>::is_halt(void) const
{
	if (multiprocessor_unit == NULL) {
		return curr_pe->is_halt();
	} else {
		return multiprocessor_unit->is_halt();
	}
}

template <class T>
inline bool debugger<T>::is_bus_error(void) const
{
	if (pe_buf.size() == 1) {
		return curr_pe->is_bus_error();
	} else {
		return multiprocessor_unit->is_bus_error();
	}
}

template <class T>
inline void debugger<T>::step(void)
{
	sync_unit->clock_in();
	sync_unit->clock_out();
}

template <class T>
void debugger<T>::execute(void)
{
	if (multiprocessor_unit == NULL) {
		if (breakpoint_buf.size() == 0) {
			while (!curr_pe->is_halt() && !curr_pe->is_bus_error()) {
				sync_unit->clock_in();
				sync_unit->clock_out();
			}
		} else {
			address_type pc = curr_pe->processor().program_counter();
			while (!curr_pe->is_halt() && !curr_pe->is_bus_error()) {
				sync_unit->clock_in();
				sync_unit->clock_out();
				address_type new_pc = curr_pe->processor().program_counter();
				if (new_pc != pc) {
					pc = new_pc;
					for (size_t i = 0; i < breakpoint_buf.size(); i++) {
						if (breakpoint_buf[i] == pc) return;
					}
				}
			}
		}
	} else {
		if (breakpoint_buf.size() == 0) {
			while (!multiprocessor_unit->is_halt() &&
				   !multiprocessor_unit->is_bus_error()) {
				sync_unit->clock_in();
				sync_unit->clock_out();
			}
		} else {
			address_type pc = curr_pe->processor().program_counter();
			while (!multiprocessor_unit->is_halt() &&
				   !multiprocessor_unit->is_bus_error()) {
				sync_unit->clock_in();
				sync_unit->clock_out();
				address_type new_pc = curr_pe->processor().program_counter();
				if (new_pc != pc) {
					pc = new_pc;
					for (size_t i = 0; i < breakpoint_buf.size(); i++) {
						if (breakpoint_buf[i] == pc) return;
					}
				}
			}
		}
	}
}

template <class T>
void debugger<T>::reset(void)
{
	sync_unit->reset();
}

template <class T>
void debugger<T>::insert_breakpoint(const debugger<T>::address_type& a)
{
	typename breakpoint_container_type::iterator p;
	breakpoint_container_type& buf(breakpoint_buf);
	p = find(buf.begin(), buf.end(), a);
	if (p != buf.end()) return;
	// p = find_if(buf.begin(), buf.end(), less_equal<address_type>(a));
	for (p = buf.begin(); p != buf.end(); p++) {
		if (a < *p) break;
	}
	buf.insert(p, a);
}

template <class T>
void debugger<T>::erase_breakpoint(const debugger<T>::address_type& a)
{
	typename breakpoint_container_type::iterator p;
	breakpoint_container_type& buf(breakpoint_buf);
	p = find(buf.begin(), buf.end(), a);
	if (p != buf.end()) buf.erase(p);
}

template <class T>
void debugger<T>::clear_breakpoints(void)
{
	breakpoint_buf.clear();
}

template <class T> template <class Cont>
void debugger<T>::dump_breakpoints(Cont& c) const
{
	copy(breakpoint_buf.begin(), breakpoint_buf.end(), back_inserter(c));
}

template <class T>
bool debugger<T>::load(const char* a)
{
	for (size_t i = 0; i < pe_buf.size(); i++) {
		if (!pe_buf[i]->load(a)) return false;
	}
	return true;
}

template <class T>
void debugger<T>::dump
	(ostream& os,
	 const debugger<T>::address_type& a,
	 const debugger<T>::size_type& b) const
{
	curr_pe->memory_map().dump(os, a, b);
	os << '\n';
}

template <class T>
void debugger<T>::disassemble
	(ostream& os,
	 const debugger<T>::address_type& a,
	 const debugger<T>::size_type& b) const
{
	curr_pe->disassemble(os, a, b);
	os << '\n';
}

#endif /* DEBUGGER_H */
