/* -*- C++ -*-
 *
 * <<< cui_debugger.h >>>
 *
 * --- Character-based user interface debugger class 'cui_debugger'
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef CUI_DEBUGGER_H
#define CUI_DEBUGGER_H 1

// #define CUI_DEBUGGER_DEBUG

#include <cctype>
#include <iostream>
#include <map>
#include <string>
#include <vector>
#include "debugger.h"

template <class T>
class cui_debugger : public debugger<T>
{
public:
	typedef cui_debugger<T> thisclass;
	typedef debugger<T> inherited;
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
	typedef typename inherited::char_type char_type;
	typedef typename inherited::size_type size_type;
	typedef typename inherited::processor_type processor_type;
	typedef typename inherited::memory_map_type memory_map_type;
	typedef typename inherited::processing_element_type processing_element_type;
	typedef typename inherited::multiprocessor_type multiprocessor_type;
	static const size_t sizeof_data_type = inherited::sizeof_data_type;
private:
	typedef vector<string> command_args_type;
	typedef void (thisclass::*command_function_type)(const command_args_type&);
	struct command_map_entry_type {
		command_function_type function;
		size_t minimal_argument_number, maximal_argument_number;
		bool repeatable;
		command_map_entry_type(void) {}
		command_map_entry_type
			(command_function_type b, size_t c, size_t d, bool e)
			: function(b),
			  minimal_argument_number(c),
			  maximal_argument_number(d),
			  repeatable(e) {}
	};
	typedef map<string, command_map_entry_type> command_map_type;
	typedef map<string, string> command_completion_map_type;
	command_map_type command_map_;
	command_completion_map_type command_completion_map_;
	istream* input_stream_;
	ostream* output_stream_;
	ostream* error_stream_;
	string prompt_string_, version_string_;
	address_type dump_address_;
	size_type dump_size_;
	address_type list_address_;
	size_type list_size_;
	size_t step_count_;
	static bool is_dec_string(const string&);
	static bool is_hex_string(const string&);
	static bool is_integer_string(const string&);
	static unsigned long string_to_int_dec(const string&);
	static unsigned long string_to_int_hex(const string&);
	static data_type to_integer(const string& a) { return to_integer_dec(a); }
	static data_type to_integer_dec(const string&);
	static data_type to_integer_hex(const string&);
	template <class Cont>
		static void read_commandline_input(istream&, Cont&);
	void setup_command_map(void);
	void setup_variables(void);
	void help_command(const command_args_type&);
	void step_command(const command_args_type&);
	void run_command(const command_args_type&);
	void continue_command(const command_args_type&);
	void print_command(const command_args_type&);
	void memory_dump_command(const command_args_type&);
	void list_command(const command_args_type&);
	void breakpoint_command(const command_args_type&);
	void delete_breakpoint_command(const command_args_type&);
	void version_command(const command_args_type&);
	void already_halted_report(void);
	void bus_error_report(void);
	void commandline_status_report(void);
public:
	cui_debugger(void);
	cui_debugger(const cui_debugger<T>&);
	virtual ~cui_debugger();
	void set_prompt(const string&);
	void set_version(const string&);
	void set_standard_input_stream(istream&);
	void set_standard_output_stream(ostream&);
	void set_standard_error_stream(ostream&);
	void interactive_mode(void);
};

template <class T>
bool cui_debugger<T>::is_dec_string(const string& a)
{
	size_t len = a.length(), i = 0;
	while (i < len && isdigit(a[i])) i++;
	return (i == len);
}

template <class T>
bool cui_debugger<T>::is_hex_string(const string& a)
{
	size_t len = a.length(), i;
	i = (len > 2 && a[0] == '0' && (a[1] == 'x' || a[1] == 'X')) ? 2 : 0;
	while (i < len && isxdigit(a[i])) i++;
	return (i == len);
}

template <class T>
bool cui_debugger<T>::is_integer_string(const string& a)
{
	return is_dec_string(a) || is_hex_string(a);
}

template <class T>
unsigned long cui_debugger<T>::string_to_int_dec(const string& a)
{
	unsigned long val = 0;
	for (size_t i = 0; i < a.length(); i++) {
		int tmp;
		switch (a[i]) {
		case '0': tmp =  0; break;
		case '1': tmp =  1; break;
		case '2': tmp =  2; break;
		case '3': tmp =  3; break;
		case '4': tmp =  4; break;
		case '5': tmp =  5; break;
		case '6': tmp =  6; break;
		case '7': tmp =  7; break;
		case '8': tmp =  8; break;
		case '9': tmp =  9; break;
		default:  tmp =  0; break;
		}
		val = (val * 10) + tmp;
	}
	return val;
}

template <class T>
unsigned long cui_debugger<T>::string_to_int_hex(const string& a)
{
	unsigned long val = 0;
	for (size_t i = 0; i < a.length(); i++) {
		int tmp;
		switch (a[i]) {
		case '0': tmp =  0; break;
		case '1': tmp =  1; break;
		case '2': tmp =  2; break;
		case '3': tmp =  3; break;
		case '4': tmp =  4; break;
		case '5': tmp =  5; break;
		case '6': tmp =  6; break;
		case '7': tmp =  7; break;
		case '8': tmp =  8; break;
		case '9': tmp =  9; break;
		case 'A':
		case 'a': tmp = 10; break;
		case 'B':
		case 'b': tmp = 11; break;
		case 'C':
		case 'c': tmp = 12; break;
		case 'D':
		case 'd': tmp = 13; break;
		case 'E':
		case 'e': tmp = 14; break;
		case 'F':
		case 'f': tmp = 15; break;
		default:  tmp =  0; break;
		}
		val = (val << 4) + tmp;
	}
	return val;
}

template <class T>
cui_debugger<T>::data_type
cui_debugger<T>::to_integer_dec(const string& a)
{
	data_type val;
	if (is_dec_string(a)) {
		val = string_to_int_dec(a);
	} else if (is_hex_string(a)) {
		if (a.length() > 2 && a[0] == '0' && (a[1] == 'x' || a[1] == 'X')) {
			val = data_type(string_to_int_hex(a.substr(2)));
		} else {
			val = data_type(string_to_int_hex(a));
		}
	} else {
		val = data_type(0); // unknown type, error.
	}
	return val;
}

template <class T>
cui_debugger<T>::data_type
cui_debugger<T>::to_integer_hex(const string& a)
{
	data_type val;
	if (is_hex_string(a)) {
		if (a.length() > 2 && a[0] == '0' && (a[1] == 'x' || a[1] == 'X')) {
			val = data_type(string_to_int_hex(a.substr(2)));
		} else {
			val = data_type(string_to_int_hex(a));
		}
	} else if (is_dec_string(a)) {
		val = data_type(string_to_int_dec(a));
	} else {
		val = data_type(0); // unknown type, error.
	}
	return val;
}

template <class T> template <class Cont>
void cui_debugger<T>::read_commandline_input(istream& is, Cont& buf)
{
	while (is) {
		string a;
		char c;
		while (is.get(c) && isspace(c) && c != '\n');
		if (!is || c == '\n') break;
		a += c;
		while (is.get(c) && !isspace(c)) a += c;
		buf.push_back(a);
		if (!is || c == '\n') break;
	}
}

template <class T>
cui_debugger<T>::cui_debugger(void)
	: inherited(),
	  input_stream_(&cin),
	  output_stream_(&cout),
	  error_stream_(&cerr)
{
	setup_command_map();
}

template <class T>
cui_debugger<T>::cui_debugger(const cui_debugger<T>& a)
	: inherited(a),
	  input_stream_(a.input_stream_),
	  output_stream_(a.output_stream_),
	  error_stream_(a.error_stream_),
	  prompt_string_(a.prompt_string_),
	  version_string_(a.version_string_)
{
	setup_command_map();
}

template <class T>
cui_debugger<T>::~cui_debugger()
{}

template <class T>
void cui_debugger<T>::set_prompt(const string& a)
{
	prompt_string_ = a;
}

template <class T>
void cui_debugger<T>::set_version(const string& a)
{
	version_string_ = a;
}

template <class T>
void cui_debugger<T>::set_standard_input_stream(istream& a)
{
	input_stream_ = &a;
}

template <class T>
void cui_debugger<T>::set_standard_output_stream(ostream& a)
{
	output_stream_ = &a;
}

template <class T>
void cui_debugger<T>::set_standard_error_stream(ostream& a)
{
	error_stream_ = &a;
}

template <class T>
void cui_debugger<T>::setup_command_map(void)
{
	command_map_.clear();
	command_completion_map_.clear();
	command_completion_map_type tmp_comp_map_;
	{
		command_map_type& p(command_map_);
		command_completion_map_type& q(tmp_comp_map_);
		typedef command_map_entry_type ET;
		p["breakpoint"]	 = ET(&thisclass::breakpoint_command, 1, 2, false);
		q["breakpoint"]	 = "breakpoint";
		p["continue"]	 = ET(&thisclass::continue_command, 1, 1, true);
		q["continue"]	 = "continue";
		p["delete_breakpoint"]
			= ET(&thisclass::delete_breakpoint_command, 1, 2, false);
		q["delete_breakpoint"]
			= "delete_breakpoint";
		p["help"]		 = ET(&thisclass::help_command, 1, 1, false);
		q["help"]		 = "help";
		q["?"]			 = "help";
		p["list"]		 = ET(&thisclass::list_command, 1, 3, true);
		q["list"]		 = "list";
		p["memory_dump"] = ET(&thisclass::memory_dump_command, 1, 3, true);
		q["memory_dump"] = "memory_dump";
		q["md"]			 = "memory_dump";
		p["print"]		 = ET(&thisclass::print_command, 1, 2, false);
		q["print"]		 = "print";
		q["quit"]		 = "quit";
		p["run"]		 = ET(&thisclass::run_command, 1, 1, false);
		q["run"]		 = "run";
		p["step"]		 = ET(&thisclass::step_command, 1, 2, true);
		q["step"]		 = "step";
		p["version"]	 = ET(&thisclass::version_command, 1, 1, false);
		q["version"]	 = "version";
	}
	typedef command_completion_map_type::const_iterator CI;
	for (CI p = tmp_comp_map_.begin(); p != tmp_comp_map_.end(); p++) {
		// set complete command name
		command_completion_map_[p->first] = p->second;
		// set alias command name
		const string& name(p->second);
		for (size_t i = 1; i < name.length(); i++) {
			string alias = name.substr(0, i);
			CI q;
			for (q = tmp_comp_map_.begin(); q != tmp_comp_map_.end(); q++) {
				const string& t(q->first);
				if (t.length() >= i && name.substr(0, i) == alias) break;
			}
			if (q != tmp_comp_map_.end()) {
				command_completion_map_[alias] = name;
			}
		}
	}
}

template <class T>
void cui_debugger<T>::setup_variables(void)
{
	dump_address_ = address_type(0);
	dump_size_ = size_type(8 * 4);
	list_address_ = current_pe().processor().program_counter();
	list_size_ = size_type(8 * 4);
	step_count_ = 1;
}

template <class T>
void cui_debugger<T>::interactive_mode(void)
{
	if (!is_valid()) return;
	setup_variables();
	istream& is(*input_stream_);
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	command_args_type cmd_args;
	string prev_cmd;
	while (is) {
		os << prompt_string_ << flush;
		cmd_args.clear();
		read_commandline_input(is, cmd_args);
		if (cmd_args.size() == 0) {
			if (!is) {
				os << endl;
				break;
			} else if (prev_cmd.size() > 0) {
				cmd_args.push_back(prev_cmd);
			} else {
				continue;
			}
		}
#ifdef CUI_DEBUGGER_DEBUG
		{
			cerr << "command:";
			for (size_t i = 0; i < cmd_args.size(); i++) {
				cerr << " \"" << cmd_args[i] << '"';
			}
			cerr << endl;
		}
#endif // CUI_DEBUGGER_DEBUG
		const string& cmd(cmd_args[0]);
		command_completion_map_type::const_iterator p
			= command_completion_map_.find(cmd);
		if (p == command_completion_map_.end()) {
			es << cmd << ": no such command." << endl;
		} else if (p->second == "quit") {
			os << endl;
			break;
		} else {
			const string& name(p->second);
			const command_map_entry_type& t(command_map_[name]);
			if (cmd_args.size() < t.minimal_argument_number) {
				es << name << ": too few arguments." << endl;
			} else if (cmd_args.size() > t.maximal_argument_number) {
				es << name << ": too many arguments." << endl;
			} else {
				(this->*(t.function))(cmd_args);
				if (t.repeatable) prev_cmd = cmd;
				os << flush;
				es << flush;
			}
		}
	}
}

template <class T>
void cui_debugger<T>::help_command(const command_args_type& cmd)
{
	ostream& os(*output_stream_);
	os << "breakpoint        : set breakpoint"				<< endl
	   << "continue          : continue execution"			<< endl
	   << "delete_breakpoint : delete breakpoint"			<< endl
	   << "list              : disassemble"					<< endl
	   << "help, ?           : show this help"				<< endl
	   << "md                : memory dump"					<< endl
	   << "print             : print state"					<< endl
	   << "quit              : exit program"				<< endl
	   << "run               : execute from beginning"		<< endl
	   << "step              : step"						<< endl
	   << "version           : show version"				<< endl;
}

template <class T>
void cui_debugger<T>::step_command(const command_args_type& cmd)
{
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	if (is_halt() || is_bus_error()) {
		already_halted_report();
		return;
	}
	if (cmd.size() == 2) {
		int tmp = 0;
		if (!is_integer_string(cmd[1]) || (tmp = to_integer(cmd[1])) <= 0) {
			es << "step: bad argument \"" << cmd[1] << "\"." << endl;
			return;
		}
		step_count_ = size_t(tmp);
	}
	for (size_t i = 0; i < step_count_; i++) {
		step();
		if (current_pe().is_halt() || current_pe().is_bus_error()) break;
	}
	if (is_halt()) {
		commandline_status_report();
	} else if (is_bus_error()) {
		bus_error_report();
	} else {
		os << current_pe() << endl;
	}
}

template <class T>
void cui_debugger<T>::run_command(const command_args_type& cmd)
{
	// not implemented
	continue_command(cmd);
}

template <class T>
void cui_debugger<T>::continue_command(const command_args_type& cmd)
{
	ostream& os(*output_stream_);
	if (is_halt() || is_bus_error()) {
		already_halted_report();
		return;
	}
	execute();
	if (is_halt()) {
		commandline_status_report();
	} else if (is_bus_error()) {
		bus_error_report();
	} else {
		const long flags = os.flags();
		os << hex
		   << "Break at 0x" << current_pe().processor().program_counter()
		   << endl;
		os.flags(flags);
	}
}

template <class T>
void cui_debugger<T>::print_command(const command_args_type& cmd)
{
	ostream& os(*output_stream_);
	if (cmd.size() == 1) {
		os << current_pe() << endl;
	} else {
		current_pe().output(os, cmd[1]);
		os << endl;
	}
}

template <class T>
void cui_debugger<T>::memory_dump_command(const command_args_type& cmd)
{
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	if (cmd.size() >= 2) {
		if (!is_integer_string(cmd[1])) {
			es << "memory_dump: bad argument \"" << cmd[1] << "\"." << endl;
			return;
		}
		dump_address_ = address_type(to_integer_hex(cmd[1]));
	}
	if (cmd.size() == 3) {
		if (!is_integer_string(cmd[2])) {
			es << "memory_dump: bad argument \"" << cmd[2] << "\"." << endl;
			return;
		}
		dump_size_ = size_type(to_integer_hex(cmd[2]));
	}
	dump(os, dump_address_, dump_size_);
	dump_address_ += dump_size_;
}

template <class T>
void cui_debugger<T>::list_command(const command_args_type& cmd)
{
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	if (cmd.size() >= 2) {
		if (!is_integer_string(cmd[1])) {
			es << "list: bad argument \"" << cmd[1] << "\"." << endl;
			return;
		}
		list_address_ = address_type(to_integer_hex(cmd[1]));
	}
	if (cmd.size() == 3) {
		if (!is_integer_string(cmd[2])) {
			es << "list: bad argument \"" << cmd[2] << "\"." << endl;
			return;
		}
		list_size_ = size_type(to_integer_hex(cmd[2]));
	}
	disassemble(os, list_address_, list_size_);
	list_address_ += list_size_;
}

template <class T>
void cui_debugger<T>::breakpoint_command(const command_args_type& cmd)
{
	typedef vector<address_type> BCT;
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	if (cmd.size() == 1) {
		BCT bp_buf;
		dump_breakpoints(bp_buf);
		if (bp_buf.size() == 0) {
			os << "No breakpoints." << endl;
		} else {
			long flags = os.flags();
			os << "num where" << endl;
			for (size_t i = 0; i < bp_buf.size(); i++) {
				os << dec << setw(3) << i << " 0x" << hex << bp_buf[i] << endl;
			}
			os.flags(flags);
		}
	} else {
		if (!is_integer_string(cmd[1])) {
			es << "breakpoint: bad argument \"" << cmd[1] << "\"." << endl;
			return;
		}
		address_type adr = address_type(to_integer_hex(cmd[1]));
		insert_breakpoint(adr);
	}
}

template <class T>
void cui_debugger<T>::delete_breakpoint_command
	(const command_args_type& cmd)
{
	typedef vector<address_type> BCT;
	typedef typename BCT::const_iterator CI;
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	BCT bp_buf;
	dump_breakpoints(bp_buf);
	if (cmd.size() == 1) {
		if (bp_buf.size() == 0) {
			os << "No breakpoints." << endl;
		} else {
			clear_breakpoints();
			os << "All breakpoints were deleted." << endl;
		}
	} else {
		if (!is_integer_string(cmd[1])) {
			es << "delete_breakpoint: bad argument \"" << cmd[1] << "\"."
			   << endl;
			return;
		}
		address_type adr = to_integer_hex(cmd[1]);
		CI p = find(bp_buf.begin(), bp_buf.end(), adr);
		if (p != bp_buf.end()) {
			erase_breakpoint(adr);
		} else if (size_t(adr) < bp_buf.size()) {
			erase_breakpoint(bp_buf[adr]);
		} else {
			es << "No such breakpoint." << endl;
		}
	}
}

template <class T>
void cui_debugger<T>::version_command(const command_args_type& cmd)
{
	if (version_string_.length() == 0) return;
	ostream& os(*output_stream_);
	os << version_string_ << endl;
}

template <class T>
void cui_debugger<T>::already_halted_report(void)
{
	ostream& es(*error_stream_);
	es << "System has already halted." << endl;
}

template <class T>
void cui_debugger<T>::bus_error_report(void)
{
	ostream& es(*error_stream_);
	processing_element_type& pe(current_pe());
	const long flags = es.flags();
	es << hex << "Bus error at 0x" << pe.processor().program_counter()
	   << ", requested address is 0x" << pe.bus_error_address() << '.' << endl;
	es.flags(flags);
}

template <class T>
void cui_debugger<T>::commandline_status_report(void)
{
	ostream& os(*output_stream_);
	ostream& es(*error_stream_);
	int status = current_pe().commandline_status();
	if (status == 0) {
		os << "Program exited normally." << endl;
	} else {
		const long flags = es.flags();
		es << "Program exited with code " << dec << status << ".\n";
		es.flags(flags);
	}
}

#endif /* CUI_DEBUGGER_H */
