/* -*- C++ -*-
 *
 * <<< forwarder.h >>>
 *
 * --- Copyright (C) 1995-1999 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef FORWARDER_H
#define FORWARDER_H 1

#include <cstddef>
#include <iostream>
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include "freelist.h"

template<class T> class forward_elem;
template<class T> class forward_list;
template<class T> class forwarder;
template<class T> ostream& operator<<( ostream&, const forward_elem<T>& );

template<class T>
class forward_elem {
	forward_elem *next;
	T* elem;

	forward_elem( T* e = 0, forward_elem*n = 0 ) { elem = e; next = n; }
	void reset() {
		delete elem;
		if( next ) next->reset();
		delete next;
	}
	T* search( T& );

	bool del( void*, forward_elem** );

	static freelist<void*> free_forward_elem;
	void operator delete( void *elem ) { free_forward_elem.put(elem); }
	void* operator new( size_t size ) {
		void *n = free_forward_elem.get();
		if( n == 0 )
			n = ::new char[size];
		return n;
	}
public:
	void output(ostream&) const;

	friend class forward_list<T>;
	friend class debug;
};

template<class T>
class forward_list {
	forward_list *next;
	forward_elem<T> *list;

	forward_list( forward_list*p = 0 ) { next = p; list = 0; }
	void reset() {
		if( list )
			list->reset();
		delete list;
		if( next )
			next->reset();
		delete next;
	}

	T* search( T& );
	void* put( T* );
	bool del( void*, forward_list** );

	bool is_empty() const { return !list; }

	static freelist<void*> free_forward_list;
	void operator delete( void *elem ) { free_forward_list.put(elem); }
	void* operator new( size_t size ) {
		void *n = free_forward_list.get();
		if( n == 0 ) n = ::new char[size];
		return n; }
public:
	void output(ostream&) const;

	friend class forwarder<T>;
	friend class debug;
};

template<class T>
class forwarder {
	forward_list<T> *list;
public:
	forwarder() { list = 0; }
	void reset() { if( list ) list->reset(); delete list; }

	void* put( T* );
	T* get( T& );
	void del( void* );
	void step();
	void output(ostream&) const;

	friend class debug;
};

// --------------------------------------------------------
template<class T>
void* forwarder<T>::put( T* elem )
{
	if( list == 0 )
		list = new forward_list<T>;
	return list->put( elem );
}

template<class T>
void forwarder<T>::step()
{
	if( list && !list->is_empty() ) {
		forward_list<T>*n = new forward_list<T>( list );
		list = n;
	}
}

template<class T>
T* forwarder<T>::get( T& comp )
{
	if( list == 0 )
		return 0;
	return list->search( comp );
}

template<class T>
void forwarder<T>::del( void* id )
{
	forward_list<T>* lst;
	if( list ) {
		if( list->del( id, &lst ) ) {
			delete list;
			list = lst;
		}
	}
}

// --------------------------------------------------------
template<class T>
void* forward_list<T>::put( T* elem )
{
	forward_elem<T>*n = new forward_elem<T>( elem, list );

	list = n;

	return (void*)n;
}

template<class T>
T* forward_list<T>::search( T& comp )
{
	if( list ) {
		T* t = list->search( comp );

		if( t )
			return t;
	}
	if( next )
		return next->search( comp );

	return 0;
}

template<class T>
bool forward_list<T>::del( void* id, forward_list<T>**n )
{
	forward_elem<T>* l;
	if( list && list->del( id, &l ) ) {
		if( (list = l) == 0 ) {
			*n = next;
			return true;
		}
	}
	else if( next ) {
		forward_list<T>* lst;
		if( next->del( id, &lst )) {
			delete next;
			next = lst;
		}
	}

	return false;
}
// --------------------------------------------------------
template<class T>
T* forward_elem<T>::search( T& comp )
{
	if( *elem == comp )
		return elem;

	if( next )
		return next->search( comp );

	return 0;
}

template<class T>
bool forward_elem<T>::del( void *id, forward_elem<T>**n )
{
	if( (void*)this == id ) {
		*n = next;

		delete elem;
		delete this;

		return true;
	}
	else if( next ) {
		forward_elem<T>* l;
		if( next->del(id,&l) ) {
			next = l;
			*n = this;
			return true;
		}
	}

	return false;
}

// --------------------------------------------------------

template<class T>
void forward_elem<T>::output(ostream& os) const
{
	os << "{";
	for( const forward_elem<T>* l = this; l; l = l->next )
		os << *l->elem << ",";
	os << "}";
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
}

template<class T>
ostream& operator<<( ostream& os, const forward_elem<T>& a )
{
	if (os) a.output(os);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
	return os;
}

template<class T>
void forward_list<T>::output(ostream& os) const
{
	os << "{";
	for( const forward_list<T>* l = this; l; l = l->next )
		os << *l->list << ",";
	os << "}";
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
}

template<class T>
ostream& operator<<( ostream& os, const forward_list<T>& a )
{
	if (os) a.output(os);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
	return os;
}

template<class T>
void forwarder<T>::output(ostream& os) const
{
	os << *(*this).list;
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
}

template<class T>
ostream& operator<<( ostream& os, const forwarder<T>& a )
{
	if (os) a.output(os);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
	return os;
}

#endif /* FORWARDER_H */
