/* -*- C++ -*-
 *
 * <<< r3000_memory_access_unit.h >>>
 *
 * --- R3000 memory access unit class 'r3000_memory_access_unit'
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3000_MEMORY_ACCESS_UNIT_H
#define R3000_MEMORY_ACCESS_UNIT_H 1

#include "r3000_bus_interface.h"
#include "r3000_directmap_cache.h"
#include "r3000_cp0_register_file.h"

class ostream;

class r3000_memory_access_unit
{
public:
	typedef r3000_bus_interface::address_type address_type;
	typedef r3000_bus_interface::data_type data_type;
	typedef r3000_bus_interface::transaction_id_type transaction_id_type;
private:
	typedef r3000_memory_access_unit thisclass;
	enum state_type {
		READY,
		SINGLE_INSTRUCTION_READ_REQ_OWNER,
		SINGLE_INSTRUCTION_READ_SEND,
		SINGLE_INSTRUCTION_READ_WAIT,
		SINGLE_INSTRUCTION_READ_GRANT,
		SINGLE_INSTRUCTION_READ_ACK,
		SINGLE_INSTRUCTION_READ_NACK,
		SINGLE_INSTRUCTION_READ_TRANS,
		MULTI_INSTRUCTION_READ_REQ_OWNER,
		MULTI_INSTRUCTION_READ_SEND,
		MULTI_INSTRUCTION_READ_WAIT,
		MULTI_INSTRUCTION_READ_GRANT,
		MULTI_INSTRUCTION_READ_ACK,
		MULTI_INSTRUCTION_READ_NACK,
		MULTI_INSTRUCTION_READ_TRANS,
		MULTI_INSTRUCTION_READ_TRANS_LAST,
		SINGLE_DATA_READ_REQ_OWNER,
		SINGLE_DATA_READ_SEND,
		SINGLE_DATA_READ_WAIT,
		SINGLE_DATA_READ_GRANT,
		SINGLE_DATA_READ_ACK,
		SINGLE_DATA_READ_NACK,
		SINGLE_DATA_READ_TRANS,
		MULTI_DATA_READ_REQ_OWNER,
		MULTI_DATA_READ_SEND,
		MULTI_DATA_READ_WAIT,
		MULTI_DATA_READ_GRANT,
		MULTI_DATA_READ_ACK,
		MULTI_DATA_READ_NACK,
		MULTI_DATA_READ_TRANS,
		MULTI_DATA_READ_TRANS_LAST,
		DATA_WRITE_REQ_OWNER,
		DATA_WRITE_SEND,
		DATA_WRITE_WAIT,
		DATA_WRITE_GRANT,
		DATA_WRITE_ACK,
		DATA_WRITE_NACK,
		DATA_WRITE_TRANS
	};
	class request_buffer_base
	{
	private:
		enum request_type { SINGLE, MULTI };
		address_type address_;
		request_type type_;
		bool flag_;
	public:
		request_buffer_base(void);
		bool is_requested(void) const { return flag_; }
		address_type address(void) const { return address_; }
		bool is_single(void) const { return type_ == SINGLE; }
		bool is_multi(void) const { return type_ == MULTI; }
		void set_single(address_type a)
			{ address_ = a, type_ = SINGLE, flag_ = true; }
		void set_multi(address_type a)
			{ address_ = a, type_ = MULTI, flag_ = true; }
		void clear(void) { flag_ = false; }
	};
	class read_request_buffer : public request_buffer_base
	{
	public:
		read_request_buffer(void);
	};
	class write_request_buffer : public request_buffer_base
	{
	private:
		data_type data_;
	public:
		write_request_buffer(void);
		data_type data(void) const { return data_; }
		void set_single(address_type a, data_type b)
			{ request_buffer_base::set_single(a), data_ = b; }
		void set_multi(address_type a, data_type b)
			{ request_buffer_base::set_multi(a), data_ = b; }
	};
	enum trans_state_type { T_READY, T_WAIT, T_ACK, T_TRANS, T_LAST };
	r3000_bus_interface& bus_if;
	r3000_cp0_register_file& cp0_rf;
	state_type state;
	read_request_buffer inst_read_buf;
	read_request_buffer data_read_buf;
	write_request_buffer data_write_buf;
	trans_state_type inst_read_state, data_read_state, data_write_state;
	address_type inst_read_adr_, data_read_adr_, data_write_adr_;
	data_type inst_read_data_, data_read_data_, data_write_data_;
	transaction_id_type trans_id;
	unsigned int trans_count;
	r3000_directmap_cache* inst_cache_;
	r3000_directmap_cache* data_cache_;
	const r3000_directmap_cache& inst_cache(void) const
		{ return cp0_rf.is_swap_caches() ? *data_cache_ : *inst_cache_; }
	r3000_directmap_cache& inst_cache(void)
		{ return cp0_rf.is_swap_caches() ? *data_cache_ : *inst_cache_; }
	const r3000_directmap_cache& data_cache(void) const
		{ return cp0_rf.is_swap_caches() ? *inst_cache_ : *data_cache_; }
	r3000_directmap_cache& data_cache(void)
		{ return cp0_rf.is_swap_caches() ? *inst_cache_ : *data_cache_; }
	void send_request(void);
public:
	r3000_memory_access_unit(r3000_bus_interface&, r3000_cp0_register_file&);
	r3000_memory_access_unit(const thisclass&, r3000_bus_interface&,
							 r3000_cp0_register_file&);
	~r3000_memory_access_unit();
	r3000_memory_access_unit& operator=(const thisclass&);
	void clock_in(void);
	void clock_out(void);
	void reset(void);
	void output(ostream&) const;
	bool is_connected_to_instruction_cache(void) const
	  { return (cp0_rf.is_swap_caches() ? data_cache_ : inst_cache_) != NULL; }
	bool is_connected_to_data_cache(void) const
	  { return (cp0_rf.is_swap_caches() ? inst_cache_ : data_cache_) != NULL; }
	void connect_instruction_cache(r3000_directmap_cache&);
	void connect_data_cache(r3000_directmap_cache&);
	void disconnect_instruction_cache(void);
	void disconnect_data_cache(void);
	address_type instruction_read_address(void) const { return inst_read_adr_; }
	data_type instruction_read_data(void) const { return inst_read_data_; }
	address_type data_read_address(void) const { return data_read_adr_; }
	data_type data_read_data(void) const { return data_read_data_; }
	address_type data_write_address(void) const { return data_write_adr_; }
	data_type data_write_data(void) const { return data_write_data_; }
	// state of transfer
	bool instruction_read_is_requested(void) const
		{ return inst_read_buf.is_requested(); }
	bool instruction_read_is_ready(void) const
		{ return inst_read_state == T_READY; }
	bool instruction_read_is_wait(void) const
		{ return inst_read_state == T_WAIT; }
	bool instruction_read_is_ack(void) const
		{ return inst_read_state == T_ACK; }
	bool instruction_read_is_trans(void) const
		{ return inst_read_state == T_TRANS; }
	bool instruction_read_is_trans_last(void) const
		{ return inst_read_state == T_LAST; }
	bool data_read_is_requested(void) const
		{ return data_read_buf.is_requested(); }
	bool data_read_is_ready(void) const
		{ return data_read_state == T_READY; }
	bool data_read_is_wait(void) const
		{ return data_read_state == T_WAIT; }
	bool data_read_is_ack(void) const
		{ return data_read_state == T_ACK; }
	bool data_read_is_trans(void) const
		{ return data_read_state == T_TRANS; }
	bool data_read_is_trans_last(void) const
		{ return data_read_state == T_LAST; }
	bool data_write_is_requested(void) const
		{ return data_write_buf.is_requested(); }
	bool data_write_is_ready(void) const
		{ return data_write_state == T_READY; }
	bool data_write_is_wait(void) const
		{ return data_write_state == T_WAIT; }
	bool data_write_is_ack(void) const
		{ return data_write_state == T_ACK; }
	bool data_write_is_trans(void) const
		{ return data_write_state == T_TRANS; }
	bool data_write_is_trans_last(void) const
		{ return data_write_state == T_LAST; }
	// send request
	void send_single_instruction_read_request(address_type a)
		{ inst_read_buf.set_single(a); }
	void send_multi_instruction_read_request(address_type a)
		{ inst_read_buf.set_multi(inst_cache().tag_address(a)); }
	void send_single_data_read_request(address_type a)
		{ data_read_buf.set_single(a); }
	void send_multi_data_read_request(address_type a)
		{ data_read_buf.set_multi(data_cache().tag_address(a)); }
	void send_data_write_request(address_type a, data_type b)
		{ data_write_buf.set_single(a, b); }
	// others
	inline bool is_instruction_reading(void) const;
	inline bool is_data_reading(void) const;
	inline bool is_reading(void) const;
	inline bool is_writing(void) const;
	inline bool have_read_request(void) const;
	inline bool is_cacheable(address_type) const;
	inline bool is_cacheable_instruction(address_type) const;
	inline bool is_cacheable_data(address_type) const;
	inline bool is_instruction_cache_hit(address_type) const;
	inline bool is_data_cache_hit(address_type) const;
	inline data_type instruction_cache_read(address_type) const;
	inline data_type data_cache_read(address_type) const;
	inline void data_cache_write(address_type, data_type);
	inline void data_cache_invalidate(address_type);
};

ostream& operator<<(ostream&, const r3000_memory_access_unit&);

inline bool r3000_memory_access_unit::is_instruction_reading(void) const
{
	switch (state) {
	case SINGLE_INSTRUCTION_READ_WAIT:
	case SINGLE_INSTRUCTION_READ_ACK:
	case SINGLE_INSTRUCTION_READ_TRANS:
	case MULTI_INSTRUCTION_READ_WAIT:
	case MULTI_INSTRUCTION_READ_ACK:
	case MULTI_INSTRUCTION_READ_TRANS:
	case MULTI_INSTRUCTION_READ_TRANS_LAST:
		return true;
	default:
		return false;
	}
}

inline bool r3000_memory_access_unit::is_data_reading(void) const
{
	switch (state) {
	case SINGLE_DATA_READ_WAIT:
	case SINGLE_DATA_READ_ACK:
	case SINGLE_DATA_READ_TRANS:
	case MULTI_DATA_READ_WAIT:
	case MULTI_DATA_READ_ACK:
	case MULTI_DATA_READ_TRANS:
	case MULTI_DATA_READ_TRANS_LAST:
		return true;
	default:
		return false;
	}
}

inline bool r3000_memory_access_unit::is_reading(void) const
{
	switch (state) {
	case SINGLE_INSTRUCTION_READ_WAIT:
	case SINGLE_INSTRUCTION_READ_ACK:
	case SINGLE_INSTRUCTION_READ_TRANS:
	case MULTI_INSTRUCTION_READ_WAIT:
	case MULTI_INSTRUCTION_READ_ACK:
	case MULTI_INSTRUCTION_READ_TRANS:
	case MULTI_INSTRUCTION_READ_TRANS_LAST:
	case SINGLE_DATA_READ_WAIT:
	case SINGLE_DATA_READ_ACK:
	case SINGLE_DATA_READ_TRANS:
	case MULTI_DATA_READ_WAIT:
	case MULTI_DATA_READ_ACK:
	case MULTI_DATA_READ_TRANS:
	case MULTI_DATA_READ_TRANS_LAST:
		return true;
	default:
		return false;
	}
}

inline bool r3000_memory_access_unit::is_writing(void) const
{
	switch (state) {
	case DATA_WRITE_WAIT:
	case DATA_WRITE_ACK:
	case DATA_WRITE_TRANS:
		return true;
	default:
		return false;
	}
}

inline bool r3000_memory_access_unit::have_read_request(void) const
{
	return is_reading() ||
		   inst_read_buf.is_requested() ||
		   data_read_buf.is_requested();
}

inline bool r3000_memory_access_unit::is_cacheable
	(r3000_memory_access_unit::address_type a) const
{
	return a < 0xa0000000UL || a >= 0xc0000000UL;
}

inline bool r3000_memory_access_unit::is_cacheable_instruction
	(r3000_memory_access_unit::address_type a) const
{
	return is_connected_to_instruction_cache() && is_cacheable(a);
}

inline bool r3000_memory_access_unit::is_cacheable_data
	(r3000_memory_access_unit::address_type a) const
{
	return is_connected_to_data_cache() && is_cacheable(a);
}

inline bool r3000_memory_access_unit::is_instruction_cache_hit
	(r3000_memory_access_unit::address_type a) const
{
	return inst_cache().is_hit(inst_cache().tag_address(a));
}

inline bool r3000_memory_access_unit::is_data_cache_hit
	(r3000_memory_access_unit::address_type a) const
{
	return data_cache().is_hit(data_cache().tag_address(a));
}

inline r3000_memory_access_unit::data_type
	r3000_memory_access_unit::instruction_cache_read
	(r3000_memory_access_unit::address_type a) const
{
	return inst_cache().read(a);
}

inline r3000_memory_access_unit::data_type
	r3000_memory_access_unit::data_cache_read
	(r3000_memory_access_unit::address_type a) const
{
	return data_cache().read(a);
}

inline void r3000_memory_access_unit::data_cache_write
	(r3000_memory_access_unit::address_type a,
	 r3000_memory_access_unit::data_type b)
{
	data_cache().write(a, b);
}

inline void r3000_memory_access_unit::data_cache_invalidate
	(r3000_memory_access_unit::address_type a)
{
	data_cache().invalidate(a);
}

#endif /* R3000_MEMORY_ACCESS_UNIT_H */
