/*
 * <<< r3000_integer_unit.cc >>>
 *
 * --- R3000 integer unit class 'r3000_integer_unit'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cassert>
#include <cstring> // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include "r3000_integer_unit.h"
#include "r3000_cp0.h"
#include "r3000_memory_access_unit.h"
#include "r3000_write_buffer_unit.h"
#include "r3010.h"

r3000_integer_unit::r3000_integer_unit(r3000_memory_access_unit& a,
	r3000_write_buffer_unit& b, r3000_cp0& c)
	: ma_unit(a),
	  wb_unit(b),
	  cp0(c),
	  cp1(NULL),
	  pc_(r3000_reset_exception_vector),
	  streaming_flag(0)
{}

r3000_integer_unit::r3000_integer_unit(const r3000_integer_unit& a,
	r3000_memory_access_unit& b, r3000_write_buffer_unit& c, r3000_cp0& d)
	: inst_buf(a.inst_buf),
	  gpr(a.gpr),
	  mul_unit(a.mul_unit),
	  stall_buf(a.stall_buf),
	  ma_unit(b),
	  wb_unit(c),
	  cp0(d),
	  cp1(NULL),
	  pc_(a.pc_),
	  streaming_flag(a.streaming_flag)
{}

r3000_integer_unit::~r3000_integer_unit()
{}

r3000_integer_unit& r3000_integer_unit::operator=(const r3000_integer_unit& a)
{
	if (this != &a) {
		inst_buf = a.inst_buf;
		gpr = a.gpr;
		mul_unit = a.mul_unit;
		stall_buf = a.stall_buf;
	}
	return *this;
}

inline void r3000_integer_unit::instruction_fetch(void)
{
	const word_type adr = pc();
	if (ma_unit.is_cacheable_instruction(adr)) {
		if (ma_unit.is_instruction_cache_hit(adr)) {
			inst_buf.if_stage().instruction_fetch(adr,
				ma_unit.instruction_cache_read(adr));
		} else {
			ma_unit.send_multi_instruction_read_request(adr);
			stall_buf.push(r3000_stall::INSTRUCTION_CACHE_MISS_STALL);
		}
	} else {
		ma_unit.send_single_instruction_read_request(adr);
		stall_buf.push(r3000_stall::UNCACHEABLE_INSTRUCTION_STALL);
	}
}

inline void r3000_integer_unit::register_fetch(void)
{
	r3000_pipeline_stage& rd_stage(inst_buf.rd_stage());
	rd_stage.register_fetch(gpr);
	rd_stage.forwarding(inst_buf.mm_stage());
	rd_stage.forwarding(inst_buf.ex_stage());
	if (is_connected_to_coprocessor()) {
		if (cp1->instruction_fetch_p()) {
			cp1->instruction_fetch(rd_stage.image());
		} else {
			stall_buf.push(r3000_stall::COPROCESSOR_IFETCH_BUSY_STALL);
		}
	}
}

inline void r3000_integer_unit::execute(void)
{
	r3000_pipeline_stage& ex_stage(inst_buf.ex_stage());
	if (is_connected_to_coprocessor() && ex_stage.coprocessor_number() == 1) {
		if (ex_stage.opcode() == r3000_instruction::BCT ||
			ex_stage.opcode() == r3000_instruction::BCF) {
			ex_stage.coprocessor_condition_fetch(cp1->fpcond() ? 1 : 0);
		}
	}
	ex_stage.execute();
	switch (ex_stage.opcode()) {
	case r3000_instruction::J:
	case r3000_instruction::JAL:
	case r3000_instruction::JR:
	case r3000_instruction::JALR:
	case r3000_instruction::BEQ:
	case r3000_instruction::BNE:
	case r3000_instruction::BLEZ:
	case r3000_instruction::BGTZ:
	case r3000_instruction::BLTZ:
	case r3000_instruction::BGEZ:
	case r3000_instruction::BLTZAL:
	case r3000_instruction::BGEZAL:
	case r3000_instruction::BCT:
	case r3000_instruction::BCF:
		pc() = ex_stage.destination_pc();
		break;
	case r3000_instruction::MULT:
		mul_unit.mult(ex_stage.source1(), ex_stage.source2());
		break;
	case r3000_instruction::MULTU:
		mul_unit.multu(ex_stage.source1(), ex_stage.source2());
		break;
	case r3000_instruction::DIV:
		mul_unit.div(ex_stage.source1(), ex_stage.source2());
		break;
	case r3000_instruction::DIVU:
		mul_unit.divu(ex_stage.source1(), ex_stage.source2());
		break;
	case r3000_instruction::MFHI:
		if (mul_unit.ready()) {
			ex_stage.mf_fetch(mul_unit.hi());
			inst_buf.rd_stage().forwarding(ex_stage);
		} else {
			stall_buf.push(r3000_stall::MULTIPLY_UNIT_BUSY_STALL);
		}
		break;
	case r3000_instruction::MFLO:
		if (mul_unit.ready()) {
			ex_stage.mf_fetch(mul_unit.lo());
			inst_buf.rd_stage().forwarding(ex_stage);
		} else {
			stall_buf.push(r3000_stall::MULTIPLY_UNIT_BUSY_STALL);
		}
		break;
	case r3000_instruction::MTHI:
		mul_unit.hi() = ex_stage.source1();
		break;
	case r3000_instruction::MTLO:
		mul_unit.lo() = ex_stage.source1();
		break;
	default:
		break;
	}
}

inline void r3000_integer_unit::memory_access(void)
{
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	switch (mm_stage.opcode()) {
	case r3000_instruction::LB:
	case r3000_instruction::LBU:
	case r3000_instruction::LH:
	case r3000_instruction::LHU:
	case r3000_instruction::LW:
	case r3000_instruction::LWL:
	case r3000_instruction::LWR:
		// load
		{
			const word_type adr = mm_stage.word_aligned_access_address();
			if (cp0.register_file().is_isolate_caches()) {
				// isolated cache read
				if (ma_unit.is_connected_to_data_cache()) {
					mm_stage.load_fetch_for_big_endian(
						ma_unit.data_cache_read(adr));
				} else {
					cp0.register_file().set_cache_miss_error();
				}
			} else if (ma_unit.is_cacheable_data(adr)) {
				// cache read
				if (!wb_unit.empty()) {
					// flush write buffer
					stall_buf.push(r3000_stall::DATA_CACHE_MISS_STALL_WFLUSH);
				} else if (ma_unit.is_data_cache_hit(adr)) {
					// cache hit
					mm_stage.load_fetch_for_big_endian(
						ma_unit.data_cache_read(adr));
				} else {
					// cache miss
					ma_unit.send_multi_data_read_request(adr);
					stall_buf.push(r3000_stall::DATA_CACHE_MISS_STALL);
				}
			} else {
				// uncacheable read
				if (!wb_unit.empty()) {
					// flush write buffer
					stall_buf.push(r3000_stall::UNCACHEABLE_DATA_STALL_WFLUSH);
				} else {
					ma_unit.send_single_data_read_request(adr);
					stall_buf.push(r3000_stall::UNCACHEABLE_DATA_STALL);
				}
			}
		}
		break;
	case r3000_instruction::LWC:
		// load word coprocessor
		if (is_connected_to_coprocessor() &&
			mm_stage.coprocessor_number() == 1) {
			const word_type adr = mm_stage.word_aligned_access_address();
			if (cp0.register_file().is_isolate_caches()) {
				// isolated cache read
				word_type dat;
				if (ma_unit.is_connected_to_data_cache()) {
					dat = ma_unit.data_cache_read(adr);
				} else {
					cp0.register_file().set_cache_miss_error();
					dat = word_type(0);
				}
				mm_stage.load_fetch_for_big_endian(dat);
				if (cp1->data_fetch_p()) {
					cp1->data_fetch(dat);
				} else {
					stall_buf.push(r3000_stall::COPROCESSOR_FETCH_BUSY_STALL);
				}
			} else if (ma_unit.is_cacheable_data(adr)) {
				// cache read
				if (!wb_unit.empty()) {
					// flush write buffer
					stall_buf.push
						(r3000_stall::DATA_CACHE_MISS_STALL_WFLUSH);
				} else if (ma_unit.is_data_cache_hit(adr)) {
					// cache hit
					const word_type dat = ma_unit.data_cache_read(adr);
					mm_stage.load_fetch_for_big_endian(dat);
					if (cp1->data_fetch_p()) {
						cp1->data_fetch(dat);
					} else {
						stall_buf.push
							(r3000_stall::COPROCESSOR_FETCH_BUSY_STALL);
					}
				} else {
					// cache miss
					ma_unit.send_multi_data_read_request(adr);
					stall_buf.push(r3000_stall::DATA_CACHE_MISS_STALL);
				}
			} else {
				// uncacheable read
				if (!wb_unit.empty()) {
					// flush write buffer
					stall_buf.push(r3000_stall::UNCACHEABLE_DATA_STALL_WFLUSH);
				} else {
					ma_unit.send_single_data_read_request(adr);
					stall_buf.push(r3000_stall::UNCACHEABLE_DATA_STALL);
				}
			}
		}
		break;
	case r3000_instruction::SW:
		// word write
		if (cp0.register_file().is_isolate_caches()) {
			if (ma_unit.is_connected_to_data_cache()) {
				ma_unit.data_cache_write(mm_stage.access_address(),
										 mm_stage.access_data());
			} else {
				cp0.register_file().set_cache_miss_error();
			}
		} else if (!wb_unit.full()) {
			wb_unit.push(mm_stage.access_address(), mm_stage.access_data());
		} else {
			stall_buf.push(r3000_stall::WRITE_BUSY_STALL);
		}
		break;
	case r3000_instruction::SB:
	case r3000_instruction::SH:
	case r3000_instruction::SWL:
	case r3000_instruction::SWR:
		{
			const word_type adr =
				inst_buf.mm_stage().word_aligned_access_address();
			// partial word write
			if (cp0.register_file().is_isolate_caches()) {
				// invalidate cache line, data is dummy
				if (ma_unit.is_connected_to_data_cache()) {
					ma_unit.data_cache_invalidate(adr);
				} else {
					cp0.register_file().set_cache_miss_error();
				}
				mm_stage.partial_word_store_fetch_for_big_endian(word_type(0));
			} else if (ma_unit.is_cacheable_data(adr)) {
				// cache read and merge
				if (wb_unit.buffered(adr)) {
					// write buffer hit
					mm_stage.partial_word_store_fetch_for_big_endian(
						wb_unit.read(adr));
					stall_buf.push(r3000_stall::PARTIAL_WORD_STORES_FIXUP);
				} else if (ma_unit.is_data_cache_hit(adr)) {
					// cache hit
					mm_stage.partial_word_store_fetch_for_big_endian(
						ma_unit.data_cache_read(adr));
					stall_buf.push(r3000_stall::PARTIAL_WORD_STORES_FIXUP);
				} else {
					// cache & write buffer miss
					ma_unit.send_multi_data_read_request(adr);
					stall_buf.push(r3000_stall::DATA_CACHE_MISS_STALL);
				}
			} else {
				// uncacheable read and merge
				if (wb_unit.buffered(adr)) {
					// write buffer hit
					mm_stage.partial_word_store_fetch_for_big_endian(
						wb_unit.read(adr));
					stall_buf.push(r3000_stall::PARTIAL_WORD_STORES_FIXUP);
				} else {
					// write buffer miss
					ma_unit.send_single_data_read_request(adr);
					stall_buf.push(r3000_stall::UNCACHEABLE_DATA_STALL);
				}
			}
		}
		break;
	case r3000_instruction::SWC:
		// store word coprocessor
		if (is_connected_to_coprocessor() &&
			mm_stage.coprocessor_number() == 1) {
			if (cp1->data_out_p()) {
				word_type dat = cp1->data_out();
				cp1->data_out_ok();
				mm_stage.coprocessor_store_fetch(dat);
				if (cp0.register_file().is_isolate_caches()) {
					if (ma_unit.is_connected_to_data_cache()) {
						ma_unit.data_cache_write(mm_stage.access_address(),
												 dat);
					} else {
						cp0.register_file().set_cache_miss_error();
					}
				} else if (!wb_unit.full()) {
					wb_unit.push(mm_stage.access_address(), dat);
				} else {
					stall_buf.push(r3000_stall::WRITE_BUSY_STALL);
				}
			} else {
				stall_buf.push(r3000_stall::COPROCESSOR_OUTPUT_BUSY_STALL);
			}
		}
		break;
	case r3000_instruction::MFC:
	case r3000_instruction::CFC:
		if (mm_stage.coprocessor_number() == 0) {
			mm_stage.mf_fetch(cp0.register_file()[mm_stage.source1_number()]);
		} else if (mm_stage.coprocessor_number() == 1 &&
			is_connected_to_coprocessor()) {
			if (cp1->data_out_p()) {
				mm_stage.mf_fetch(cp1->data_out());
				cp1->data_out_ok();
			} else {
				stall_buf.push(r3000_stall::COPROCESSOR_OUTPUT_BUSY_STALL);
			}
		}
		break;
	case r3000_instruction::MTC:
	case r3000_instruction::CTC:
		if (mm_stage.coprocessor_number() == 0) {
			cp0.register_file()[mm_stage.destination_number()]
				= mm_stage.source1();
		} else if (mm_stage.coprocessor_number() == 1 &&
			is_connected_to_coprocessor()) {
			if (cp1->data_fetch_p()) {
				cp1->data_fetch(mm_stage.source1());
			} else {
				stall_buf.push(r3000_stall::COPROCESSOR_FETCH_BUSY_STALL);
			}
		}
		break;
	default:
		break;
	}
}

inline void r3000_integer_unit::writeback(void)
{
	inst_buf.wb_stage().writeback(gpr);
}

inline void r3000_integer_unit::execute_instruction_cache_miss_stall(void)
{
	if (!ma_unit.instruction_read_is_requested() &&
		!ma_unit.instruction_read_is_wait()) {
		stall_buf.change(r3000_stall::INSTRUCTION_CACHE_MISS_REFILL);
	}
}

inline void r3000_integer_unit::execute_instruction_cache_miss_refill(void)
{
	if (ma_unit.instruction_read_is_trans_last() ||
		ma_unit.instruction_read_is_ready()) {
		stall_buf.change(r3000_stall::INSTRUCTION_CACHE_MISS_FIXUP);
	} else if (ma_unit.instruction_read_address() == pc() &&
			   !stall_buf.have_next_stall()) {
		stall_buf.change
			(r3000_stall::INSTRUCTION_CACHE_MISS_FIXUP_TO_STREAMING);
	}
}

inline void r3000_integer_unit::execute_instruction_cache_miss_fixup(void)
{
	const word_type adr = pc();
	inst_buf.if_stage().instruction_fetch(adr,
		ma_unit.instruction_cache_read(adr));
	stall_buf.pop();
}

inline void
	r3000_integer_unit::execute_instruction_cache_miss_fixup_to_streaming(void)
{
	const word_type adr = pc();
	inst_buf.if_stage().instruction_fetch(adr,
		ma_unit.instruction_cache_read(adr));
	stall_buf.pop();
	if (!ma_unit.instruction_read_is_trans_last() &&
		!ma_unit.instruction_read_is_ready()) {
		streaming_flag = 1;
	}
}

inline void r3000_integer_unit::execute_uncacheable_instruction_stall(void)
{
	if (!ma_unit.instruction_read_is_requested() &&
		!ma_unit.instruction_read_is_wait()) {
		stall_buf.change(r3000_stall::UNCACHEABLE_INSTRUCTION_FIXUP);
	}
}

inline void r3000_integer_unit::execute_uncacheable_instruction_fixup(void)
{
	inst_buf.if_stage().instruction_fetch(pc(),
		ma_unit.instruction_read_data());
	stall_buf.pop();
}

inline void r3000_integer_unit::execute_data_cache_miss_stall_wflush(void)
{
	if (!wb_unit.empty()) return;
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	if (mm_stage.opcode() != r3000_instruction::LWC) {
		// load
		const word_type adr = mm_stage.word_aligned_access_address();
		if (ma_unit.is_data_cache_hit(adr)) {
			// cache hit
			mm_stage.load_fetch_for_big_endian(ma_unit.data_cache_read(adr));
			inst_buf.rd_stage().forwarding(mm_stage);
			stall_buf.pop();
		} else {
			// cache miss
			ma_unit.send_multi_data_read_request(adr);
			stall_buf.change(r3000_stall::DATA_CACHE_MISS_STALL);
		}
	} else {
		// load to coprocessor
		if (is_connected_to_coprocessor() &&
			mm_stage.coprocessor_number() == 1) {
			const word_type adr = mm_stage.word_aligned_access_address();
			if (ma_unit.is_data_cache_hit(adr)) {
				// cache hit
				const word_type dat = ma_unit.data_cache_read(adr);
				mm_stage.load_fetch_for_big_endian(dat);
				inst_buf.rd_stage().forwarding(mm_stage);
				if (cp1->data_fetch_p()) {
					cp1->data_fetch(dat);
					stall_buf.pop();
				} else {
					stall_buf.change(
						r3000_stall::COPROCESSOR_FETCH_BUSY_STALL);
				}
			} else {
				// cache miss
				ma_unit.send_multi_data_read_request(adr);
				stall_buf.change(r3000_stall::DATA_CACHE_MISS_STALL);
			}
		}
	}
}

inline void r3000_integer_unit::execute_data_cache_miss_stall(void)
{
	if (!ma_unit.data_read_is_requested() && !ma_unit.data_read_is_wait()) {
		stall_buf.change(r3000_stall::DATA_CACHE_MISS_REFILL);
	}
}

inline void r3000_integer_unit::execute_data_cache_miss_refill(void)
{
	if (ma_unit.data_read_is_trans_last() || ma_unit.data_read_is_ready()) {
		stall_buf.change(r3000_stall::DATA_CACHE_MISS_FIXUP);
	}
}

inline void r3000_integer_unit::execute_data_cache_miss_fixup(void)
{
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	switch (mm_stage.opcode()) {
	case r3000_instruction::LB:
	case r3000_instruction::LBU:
	case r3000_instruction::LH:
	case r3000_instruction::LHU:
	case r3000_instruction::LW:
	case r3000_instruction::LWL:
	case r3000_instruction::LWR:
		// load
		mm_stage.load_fetch_for_big_endian(ma_unit.data_cache_read(
			mm_stage.word_aligned_access_address()));
		inst_buf.rd_stage().forwarding(mm_stage);
		stall_buf.pop();
		break;
	case r3000_instruction::LWC:
		if (mm_stage.coprocessor_number() == 1 &&
			is_connected_to_coprocessor()) {
			word_type dat = ma_unit.data_cache_read(
								mm_stage.word_aligned_access_address());
			mm_stage.load_fetch_for_big_endian(dat);
			inst_buf.rd_stage().forwarding(mm_stage);
			stall_buf.pop();
			if (cp1->data_fetch_p()) cp1->data_fetch(dat);
		}
		break;
	case r3000_instruction::SB:
	case r3000_instruction::SH:
	case r3000_instruction::SWL:
	case r3000_instruction::SWR:
		// partial word store
		mm_stage.partial_word_store_fetch_for_big_endian(
			ma_unit.data_cache_read(mm_stage.word_aligned_access_address() &
				~word_type(sizeof_word_type - 1)));
		stall_buf.change(r3000_stall::PARTIAL_WORD_STORES_FIXUP);
		break;
	default:
		assert(0);
		break;
	}
}

inline void r3000_integer_unit::execute_uncacheable_data_stall_wflush(void)
{
	if (!wb_unit.empty()) return;
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	if (mm_stage.opcode() != r3000_instruction::LWC) {
		// load
		const word_type adr = mm_stage.word_aligned_access_address();
		ma_unit.send_single_data_read_request(adr);
		stall_buf.change(r3000_stall::UNCACHEABLE_DATA_STALL);
	} else {
		// load to coprocessor
		if (is_connected_to_coprocessor() &&
			mm_stage.coprocessor_number() == 1) {
			const word_type adr = mm_stage.word_aligned_access_address();
			ma_unit.send_single_data_read_request(adr);
			stall_buf.change(r3000_stall::UNCACHEABLE_DATA_STALL);
		} else {
			// nothing to do: exception is better
			stall_buf.pop();
		}
	}
}

inline void r3000_integer_unit::execute_uncacheable_data_stall(void)
{
	if (!ma_unit.data_read_is_requested() && !ma_unit.data_read_is_wait()) {
		stall_buf.change(r3000_stall::UNCACHEABLE_DATA_FIXUP);
	}
}

inline void r3000_integer_unit::execute_uncacheable_data_fixup(void)
{
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	switch (mm_stage.opcode()) {
	case r3000_instruction::LB:
	case r3000_instruction::LBU:
	case r3000_instruction::LH:
	case r3000_instruction::LHU:
	case r3000_instruction::LW:
	case r3000_instruction::LWL:
	case r3000_instruction::LWR:
		// load
		mm_stage.load_fetch_for_big_endian(ma_unit.data_read_data());
		inst_buf.rd_stage().forwarding(mm_stage);
		stall_buf.pop();
		break;
	case r3000_instruction::LWC:
		if (mm_stage.coprocessor_number() == 1 &&
			is_connected_to_coprocessor()) {
			word_type dat = ma_unit.data_read_data();
			mm_stage.load_fetch_for_big_endian(dat);
			inst_buf.rd_stage().forwarding(mm_stage);
			stall_buf.pop();
			if (cp1->data_fetch_p()) cp1->data_fetch(dat);
		}
		break;
	case r3000_instruction::SB:
	case r3000_instruction::SH:
	case r3000_instruction::SWL:
	case r3000_instruction::SWR:
		// partial word store
		mm_stage.partial_word_store_fetch_for_big_endian
			(ma_unit.data_read_data());
		stall_buf.change(r3000_stall::PARTIAL_WORD_STORES_FIXUP);
		break;
	default:
		assert(0);
		break;
	}
}

inline void r3000_integer_unit::execute_partial_word_stores_fixup(void)
{
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	if (!wb_unit.full()) {
		wb_unit.push(mm_stage.word_aligned_access_address(),
			mm_stage.access_data());
		stall_buf.pop();
	} else {
		stall_buf.change(r3000_stall::WRITE_BUSY_STALL);
	}
}

inline void r3000_integer_unit::execute_write_busy_stall(void)
{
	if (!wb_unit.full()) {
		stall_buf.change(r3000_stall::WRITE_BUSY_FIXUP);
	}
}

inline void r3000_integer_unit::execute_write_busy_fixup(void)
{
	r3000_pipeline_stage& mm_stage(inst_buf.mm_stage());
	wb_unit.push(mm_stage.word_aligned_access_address(),
				 mm_stage.access_data());
	stall_buf.pop();
}

inline void r3000_integer_unit::execute_multiply_unit_busy_stall(void)
{
	if (mul_unit.ready()) {
		stall_buf.change(r3000_stall::MULTIPLY_UNIT_BUSY_FIXUP);
	}
}

inline void r3000_integer_unit::execute_multiply_unit_busy_fixup(void)
{
	r3000_pipeline_stage& ex_stage(inst_buf.ex_stage());
	switch (ex_stage.opcode()) {
	case r3000_instruction::MFHI:
		ex_stage.mf_fetch(mul_unit.hi());
		inst_buf.rd_stage().forwarding(ex_stage);
		break;
	case r3000_instruction::MFLO:
		ex_stage.mf_fetch(mul_unit.lo());
		inst_buf.rd_stage().forwarding(ex_stage);
		break;
	default:
		break;
	}
	stall_buf.pop();
}

inline void r3000_integer_unit::execute_coprocessor_ifetch_busy_stall(void)
{
	if (cp1->instruction_fetch_p()) {
		stall_buf.change(r3000_stall::COPROCESSOR_IFETCH_BUSY_FIXUP);
	}
}

inline void r3000_integer_unit::execute_coprocessor_ifetch_busy_fixup(void)
{
	cp1->instruction_fetch(inst_buf.rd_stage().image());
	stall_buf.pop();
}

inline void r3000_integer_unit::execute_coprocessor_fetch_busy_stall(void)
{
	if (cp1->data_fetch_p()) {
		stall_buf.change(r3000_stall::COPROCESSOR_FETCH_BUSY_FIXUP);
	}
}

inline void r3000_integer_unit::execute_coprocessor_fetch_busy_fixup(void)
{
	cp1->data_fetch(inst_buf.mm_stage().destination());
	stall_buf.pop();
}

inline void r3000_integer_unit::execute_coprocessor_output_busy_stall(void)
{
	if (cp1->data_out_p()) {
		stall_buf.change(r3000_stall::COPROCESSOR_OUTPUT_BUSY_FIXUP);
	}
}

inline void r3000_integer_unit::execute_coprocessor_output_busy_fixup(void)
{
	r3000_pipeline_stage& mm_stage = inst_buf.mm_stage();
	if (inst_buf.mm_stage().opcode() == r3000_instruction::SWC) {
		word_type dat = cp1->data_out();
		mm_stage.coprocessor_store_fetch(dat);
		if (!wb_unit.full()) {
			wb_unit.push(mm_stage.access_address(), dat);
		} else {
			stall_buf.push(r3000_stall::WRITE_BUSY_STALL);
		}
	} else {
		mm_stage.mf_fetch(cp1->data_out());
	}
	cp1->data_out_ok();
	stall_buf.pop();
}

void r3000_integer_unit::clock(void)
{
	mul_unit.clock();
	if (is_streaming()) {
		// streaming
		const word_type old_pc = pc();
		writeback();
		memory_access();
		execute();
		register_fetch();
		if (pc() != old_pc) {
			streaming_flag = 0;
			if (!ma_unit.instruction_read_is_trans_last()) {
				stall_buf.streaming_to_instruction_cache_miss_refill();
			}
			instruction_fetch();
			stall_buf.fetch();
		} else if (stall_buf.is_buffered()) {
			streaming_flag = 0;
			if (!ma_unit.instruction_read_is_trans_last()) {
				stall_buf.streaming_to_instruction_cache_miss_refill();
			} else {
				inst_buf.if_stage().instruction_fetch(pc(),
					ma_unit.instruction_cache_read(pc()));
			}
			stall_buf.fetch();
		} else {
			if (ma_unit.instruction_read_is_trans_last()) {
				streaming_flag = 0;
			}
			inst_buf.if_stage().instruction_fetch(pc(),
				ma_unit.instruction_cache_read(pc()));
		}
	} else if (is_run_cycle()) {
		// run cycle
		writeback();
		memory_access();
		execute();
		register_fetch();
		instruction_fetch();
		stall_buf.fetch();
	} else {
		// stall cycle
		switch (stall_buf.type()) {
		case r3000_stall::INSTRUCTION_CACHE_MISS_STALL:
			execute_instruction_cache_miss_stall();
			break;
		case r3000_stall::INSTRUCTION_CACHE_MISS_REFILL:
			execute_instruction_cache_miss_refill();
			break;
		case r3000_stall::INSTRUCTION_CACHE_MISS_FIXUP:
			execute_instruction_cache_miss_fixup();
			break;
		case r3000_stall::INSTRUCTION_CACHE_MISS_FIXUP_TO_STREAMING:
			execute_instruction_cache_miss_fixup_to_streaming();
			break;
		case r3000_stall::UNCACHEABLE_INSTRUCTION_STALL:
			execute_uncacheable_instruction_stall();
			break;
		case r3000_stall::UNCACHEABLE_INSTRUCTION_FIXUP:
			execute_uncacheable_instruction_fixup();
			break;
		case r3000_stall::DATA_CACHE_MISS_STALL_WFLUSH:
			execute_data_cache_miss_stall_wflush();
			break;
		case r3000_stall::DATA_CACHE_MISS_STALL:
			execute_data_cache_miss_stall();
			break;
		case r3000_stall::DATA_CACHE_MISS_REFILL:
			execute_data_cache_miss_refill();
			break;
		case r3000_stall::DATA_CACHE_MISS_FIXUP:
			execute_data_cache_miss_fixup();
			break;
		case r3000_stall::UNCACHEABLE_DATA_STALL_WFLUSH:
			execute_uncacheable_data_stall_wflush();
			break;
		case r3000_stall::UNCACHEABLE_DATA_STALL:
			execute_uncacheable_data_stall();
			break;
		case r3000_stall::UNCACHEABLE_DATA_FIXUP:
			execute_uncacheable_data_fixup();
			break;
		case r3000_stall::PARTIAL_WORD_STORES_FIXUP:
			execute_partial_word_stores_fixup();
			break;
		case r3000_stall::WRITE_BUSY_STALL:
			execute_write_busy_stall();
			break;
		case r3000_stall::WRITE_BUSY_FIXUP:
			execute_write_busy_fixup();
			break;
		case r3000_stall::MULTIPLY_UNIT_BUSY_STALL:
			execute_multiply_unit_busy_stall();
			break;
		case r3000_stall::MULTIPLY_UNIT_BUSY_FIXUP:
			execute_multiply_unit_busy_fixup();
			break;
		case r3000_stall::COPROCESSOR_IFETCH_BUSY_STALL:
			execute_coprocessor_ifetch_busy_stall();
			break;
		case r3000_stall::COPROCESSOR_IFETCH_BUSY_FIXUP:
			execute_coprocessor_ifetch_busy_fixup();
			break;
		case r3000_stall::COPROCESSOR_FETCH_BUSY_STALL:
			execute_coprocessor_fetch_busy_stall();
			break;
		case r3000_stall::COPROCESSOR_FETCH_BUSY_FIXUP:
			execute_coprocessor_fetch_busy_fixup();
			break;
		case r3000_stall::COPROCESSOR_OUTPUT_BUSY_STALL:
			execute_coprocessor_output_busy_stall();
			break;
		case r3000_stall::COPROCESSOR_OUTPUT_BUSY_FIXUP:
			execute_coprocessor_output_busy_fixup();
			break;
		default:
			break;
		}
	}
	if (is_run_cycle()) {
		pc() += 4;
		inst_buf.step();
	}
	if (is_connected_to_coprocessor()) cp1->clock();
}

void r3000_integer_unit::reset(void)
{
	inst_buf.clear();
	gpr.clear();
	mul_unit.clear();
	stall_buf.clear();
	if (is_connected_to_coprocessor()) cp1->reset();
	pc() = r3000_reset_exception_vector;
}

void r3000_integer_unit::connect_coprocessor(r3010& a)
{
	cp1 = &a;
}

void r3000_integer_unit::disconnect_coprocessor(void)
{
	cp1 = NULL;
}

void r3000_integer_unit::output(ostream& os) const
{
	const long flags = os.flags();
	const char fill = os.fill();
	os << hex << setfill('0');
	os << gpr << '\n'
	   <<  "pc:" << setw(8) << pc();
	if (mul_unit.ready()) {
		os << " hi:" << setw(8) << hi()
		   << " lo:" << setw(8) << lo();
	} else {
		os << " hi:-locked- lo:-locked-";
	}
	os << '\n'
	   << "rd: " << inst_buf.rd_stage() << '\n'
	   << "ex: " << inst_buf.ex_stage() << '\n'
	   << "mm: " << inst_buf.mm_stage() << '\n'
	   << "wb: " << inst_buf.wb_stage();
	if (is_stall_cycle()) {
		os << '\n' << stall_buf;
	} else if (is_streaming()) {
		os << '\n' << "streaming";
	}
	os.flags(flags);
	os.fill(fill);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
}

bool r3000_integer_unit::output(ostream& os, const string& s) const
{
	if (s == "instruction_buffer" || s == "ibuf") {
		os << inst_buf;
		return true;
	} else if (s == "gpr") {
		os << gpr;
		return true;
	} else if (s == "stall") {
		os << stall_buf;
		return true;
	} else if (s == "pc") {
		const int flags = os.flags();
		os << "0x" << hex << pc_;
		os.flags(flags);
		return true;
	}
	return false;
}

ostream& operator<<(ostream& os, const r3000_integer_unit& a)
{
	if (os) a.output(os);
#	ifdef DEBUG
		os.flush();
#	endif // DEBUG
	return os;
}
