/*
 * <<< fileio_map.cc >>>
 *
 * --- Pseudo file I/O map class 'fileio_map'
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cassert>
#include <fstream>
#include "fileio_map.h"
#include "io_protocol.h"

fileio_map::entry::entry(void)
	: fp(NULL),
	  flags_(0)
{}

fileio_map::entry::~entry()
{
	clear();
}

bool fileio_map::entry::open(const char* a, int mode)
{
	if (is_opened()) return false;
	ios::open_mode ios_mode = ios::open_mode(0);
	if (mode & IO_FILE_OPEN_RDONLY) {
		ios_mode = ios::open_mode(ios_mode | ios::in);
	}
	if (mode & IO_FILE_OPEN_WRONLY) {
		ios_mode = ios::open_mode(ios_mode | ios::out);
	}
	if (mode & IO_FILE_OPEN_RDWR) {
		ios_mode = ios::open_mode(ios_mode | ios::in | ios::out);
	}
	if (mode & IO_FILE_OPEN_APPEND) {
		ios_mode = ios::open_mode(ios_mode | ios::app);
	}
	if (ios_mode & ios::out) {
		if (!(mode & IO_FILE_OPEN_CREAT)) {
			ios_mode = ios::open_mode(ios_mode | ios::nocreate);
		}
		if (mode & IO_FILE_OPEN_TRUNC) {
			ios_mode = ios::open_mode(ios_mode | ios::trunc);
		}
		if (mode & IO_FILE_OPEN_EXCL) {
			ios_mode = ios::open_mode(ios_mode | ios::noreplace);
		}
	}
	fstream* file = new fstream(a, ios_mode);
	if (!file->bad()) {
		fp = file;
		flags_ = opened_mask;
		if (mode & ios::in) flags_ |= readable_mask;
		if (mode & ios::out) flags_ |= writable_mask;
		return true;
	} else {
		file->close();
		delete file;
		return false;
	}
}

bool fileio_map::entry::close(void)
{
	if (!is_opened()) return false;
	delete (fstream*)(fp);
	fp = NULL;
	flags_ = 0;
	return true;
}

long fileio_map::entry::seek(long offset, int whence)
{
	if (!is_opened()) return -1;
	ios::seek_dir stream_whence = ios::beg;
	switch (whence) {
	case IO_FILE_SEEK_SET:
		stream_whence = ios::beg;
		break;
	case IO_FILE_SEEK_CUR:
		stream_whence = ios::cur;
		break;
	case IO_FILE_SEEK_END:
		stream_whence = ios::end;
		break;
	default:
		assert(0); // fileio protocol error
		break;
	}
	long result = -1;
	if (is_readable()) {
		input_stream().seekg(offset, stream_whence);
		result = input_stream().tellg();
	}
	if (is_writable()) {
		output_stream().seekp(offset, stream_whence);
		result = output_stream().tellp();
	}
	return result;
}

void fileio_map::entry::clear(void)
{
	if (is_opened()) close();
	fp = NULL;
	flags_ = 0;
}

fileio_map::fileio_map(void)
	: buf(0, entry())
{}

fileio_map::fileio_map(size_t a)
	: buf(a, entry())
{}

fileio_map::fileio_map(const fileio_map& a)
	: buf(a.buf)
{
	for (size_t i = 0; i < buf.size(); i++) {
		buf[i].force_clear_open_flag();
	}
}

fileio_map::~fileio_map()
{}

void fileio_map::set_standard_io(void)
{
	if (size() < 2 || buf[0].is_opened() || buf[1].is_opened() ||
		buf[2].is_opened()) return;
	set_input_stream(0, cin);
	set_output_stream(1, cout);
	set_output_stream(2, cerr);
}
