/* -*- C++ -*-
 *
 * <<< directmap_cache.h >>>
 *
 * --- Directmap cache class 'directmap_cache'
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef DIRECTMAP_CACHE_H
#define DIRECTMAP_CACHE_H 1

#include <cstddef>
#include <iostream>
#include <iomanip>
#include <vector>
#include "root_object.h"

template <class L>
class directmap_cache : public root_object
{
public:
	typedef typename L::address_type address_type;
	typedef typename L::data_type data_type;
	typedef L line_type;
private:
	typedef directmap_cache<L> thisclass;
	typedef root_object inherited;
	vector<line_type> buf;
	address_type line_size_, cache_size_;
	size_t upper_shift, lower_shift;
	address_type upper_mask, lower_mask, tag_mask;
	size_t index_address(const address_type& adr) const
		{ return size_t((adr & upper_mask) >> upper_shift); }
	size_t line_address(const address_type& adr) const
		{ return size_t((adr & lower_mask) >> lower_shift); }
public:
	directmap_cache(void);
	explicit directmap_cache(const address_type&, const address_type&);
	directmap_cache(const thisclass&);
	virtual ~directmap_cache();
	virtual void output(ostream&) const;
	const address_type& cache_size(void) const
		{ return cache_size_; }
	const address_type& line_size(void) const
		{ return line_size_; }
	size_t line_size_in_word(void) const
		{ return (line_size_ >> lower_shift); }
	address_type tag_address(const address_type& adr) const
		{ return adr & tag_mask; }
	bool is_hit(const address_type& adr) const
		{ return buf[index_address(adr)].is_hit(tag_address(adr)); }
	bool is_valid(const address_type& adr) const
		{ return buf[index_address(adr)].is_valid(); }
	const address_type& read_address(const address_type& adr) const
		{ return buf[index_address(adr)].tag(); }
	const data_type& read(const address_type& adr) const
		{ return buf[index_address(adr)][line_address(adr)]; }
	inline void write(const address_type&, const data_type&);
	void invalidate(const address_type& adr)
		{ return buf[index_address(adr)].set_invalid(); }
	void clear(void);
	void resize(const address_type&, const address_type&);
};

template <class L>
inline void directmap_cache<L>::write
	(const directmap_cache<L>::address_type& adr,
	 const directmap_cache<L>::data_type& dat)
{
	line_type& cl(buf[index_address(adr)]);
	address_type a_ = tag_address(adr);
	if (!cl.is_hit(a_)) {
		cl.tag() = a_;
		cl.set_valid();
	}
	cl[line_address(adr)] = dat;
}

template <class L>
directmap_cache<L>::directmap_cache(void)
{
	resize(0, 0);
}

template <class L>
directmap_cache<L>::directmap_cache
	(const directmap_cache<L>::address_type& a,
	 const directmap_cache<L>::address_type& b)
{
	resize(a, b);
}

template <class L>
directmap_cache<L>::directmap_cache(const directmap_cache<L>& a)
	: buf(a.buf),
	  line_size_(a.line_size_),
	  cache_size_(a.cache_size_),
	  upper_shift(a.upper_shift),
	  lower_shift(a.lower_shift),
	  upper_mask(a.upper_mask),
	  lower_mask(a.lower_mask),
	  tag_mask(a.tag_mask)
{}

template <class L>
directmap_cache<L>::~directmap_cache()
{}

template <class L>
void directmap_cache<L>::resize
	(const directmap_cache<L>::address_type& cache_sz,
	 const directmap_cache<L>::address_type& line_sz)
{
	if ((!(cache_sz > 0 || cache_sz == 0)) || !(line_sz > 0 || line_sz == 0))
		return;
	{
		size_t tmp = line_sz - 1;
		int i = 1;
		while (tmp > 1) tmp >>= 1, i++;
		line_size_ = (1 << i);
		upper_shift = i;
	}
	if (line_size_ == 0) {
		cache_size_ = 0;
		buf.resize(0);
		return;
	}
	{
		size_t tmp = (cache_sz - 1) / line_size_;
		int i = 1;
		while (tmp > 1) tmp >>= 1, i++;
		cache_size_ = (1 << i) * line_size_;
		upper_mask = (~(~address_type(0) << i) << upper_shift);
	}
	{
		size_t tmp = sizeof(data_type) - 1;
		int i = 1;
		while (tmp > 1) tmp >>= 1, i++;
		lower_shift = i;
		lower_mask = (~(~address_type(0) << (upper_shift - lower_shift))
					  << lower_shift);
	}
	tag_mask = (~address_type(0) << upper_shift);
	buf.resize(cache_size_ / line_size_);
	for (size_t i = 0; i < buf.size(); i++) {
		buf[i].resize(line_size_ >> lower_shift);
	}
	clear();
}

template <class L>
void directmap_cache<L>::clear(void)
{
	for (size_t i = 0; i < buf.size(); i++) buf[i].set_invalid();
}

template <class L>
void directmap_cache<L>::output(ostream& os) const
{
	if (cache_size() == 0 || line_size() == 0) return;
	size_t width;
	{
		size_t tmp = buf.size() - 1;
		int i = 1;
		while (tmp >= 16) tmp >>= 4, i++;
		width = i;
	}
	const char fill = os.fill();
	const long flags = os.flags();
	os << hex << setfill('0');
	for (size_t i = 0; i < buf.size(); i++) {
		os << '[' << setw(width) << i << "] " << buf[i];
		if (i < buf.size() - 1) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* DIRECTMAP_CACHE_H */
