/*
 * <<< sh_queen.c >>>
 *
 * --- Sample application for isis 'n-queen' - for multiprocessor
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include "osiris.h"
#include "shlib.h"

#define DEFAULT_SIZE	8
#define	MAXITER			1000
#define	MAXSIZE			64

typedef int Type;
typedef Type vector[MAXSIZE];
typedef Type vector_2[MAXSIZE * 2];
typedef Type matrix[MAXSIZE][MAXSIZE];

#define rnd() ((unsigned)((rnd_seed = 1566083941UL * rnd_seed + 1) >> 16))
unsigned long rnd_seed = 0;

static int queen(matrix*, int, int);
static void show(const matrix*, int);
#if 0
static void show_potential(const matrix*, int);
static void show_vector(const vector*, int);
#endif

static int debug_msg_flag = 0;

int main(int argc, char *argv[])
{
	int seed_table[] = {  0,  0,  0,  6, 49, 67, 98, 41, 66, 59,
						 73, 38, 30, 10, 57, 26, 16, 46, 38, 83,
						 34, 45, 49, 71, 24, 81, 40,  1, 54, 46,
						 15, 60, 55, 36, 69, 72, 58, 81, 30, 51,
						 72, 63, 32,  4, 82, 58, 91, 11, 87, 33,
						 71, 76, 27, 51, 77, 90, 84, 42, 76, 17,
						 56, 68, 96,  0 };
	matrix *ans = (matrix*)shared_malloc(sizeof(matrix));
	int punum = get_punum(), puid = get_puid();
	int verbose_flag = 0, size, seed, val;
	while (*++argv != NULL && **argv == '-') {
		argc--;
		switch (*++*argv) {
		case 'v':
			verbose_flag = 1;
			break;
		case 'z':
			debug_msg_flag = 1;
			break;
		default:
			break;
		}
	}
	if (*argv != NULL) {
		size = atoi(*argv++);
		size = (size < 0) ? DEFAULT_SIZE : ((size > MAXSIZE) ? MAXSIZE : size);
	} else {
		size = DEFAULT_SIZE;
	}
	seed = (*argv != NULL) ? atoi(*argv++) : seed_table[size - 1];
	if (verbose_flag && puid == 0) {
		printf("size:%d seed:%d\n", size, seed);
	}
	{
		int i, j;
		for (i = puid; i < size; i += punum) {
			for (j = 0; j < size; j++) {
				(*ans)[i][j] = 0;
			}
		}
	}
	barrier();
	val = queen(ans, size, seed);
	barrier();
	if (verbose_flag && puid == 0) {
		int result = (val > 0);
		printf("%s. (iter:%d)\n", (result ? "success" : "failed"), val);
		show((const matrix*)ans, size);
	}
	return (val > 0) ? 0 : 1;
}

int queen(matrix *ans, int size, int seed)
{
	matrix *potential = (matrix*)shared_malloc(sizeof(matrix));
	vector *row_ = (vector*)shared_malloc(sizeof(vector));
	vector *col_ = (vector*)shared_malloc(sizeof(vector));
	vector_2 *diag1_ = (vector_2*)shared_malloc(sizeof(vector_2));
	vector_2 *diag2_ = (vector_2*)shared_malloc(sizeof(vector_2));
	int *count_ = (int*)shared_malloc(sizeof(int));
	int *flag_ = (int*)shared_malloc(sizeof(int));
#	define U (*potential)
#	define V (*ans)
#	define row (*row_)
#	define col (*col_)
#	define diag1 (*diag1_)
#	define diag2 (*diag2_)
#	define count (*count_)
#	define flag (*flag_)
	static matrix T; /* "static" -- because of poor stack size */
	int punum = get_punum(), puid = get_puid();
	Type C = 1;
	count = 0;
	if (puid == 0) {
		int i, j;
		rnd_seed = seed;
		for (i = 0; i < size; i++) {
			for (j = 0; j < size; j++) {
				Type tmp = rnd() & 0xf;
				tmp = (tmp >= 0) ? -tmp : tmp;
				U[i][j] = tmp;
			}
		}
	}
	barrier();
	while (1) {
		int x, y;
		if (puid == 0 && debug_msg_flag) {
			printf("iter%d:\n", count);
			show((const matrix*)ans, size);
		}
		if (puid == 0) flag = 0;
		for (y = 0; y < size; y++) {
			for (x = 0; x < size; x++) {
				T[y][x] = V[y][x];
			}
		}
		for (y = puid; y < size; y += punum) {
			Type tmp = 0;
			for (x = 0; x < size; x++) tmp += T[y][x];
			row[y] = tmp;
		}
		for (x = puid; x < size; x += punum) {
			Type tmp = 0;
			for (y = 0; y < size; y++) tmp += T[y][x];
			col[x] = tmp;
		}
		for (x = puid; x < size; x += punum) {
			Type tmp = 0;
			for (y = 0; y <= x; y++) tmp += T[y][x - y];
			diag1[x] = tmp;
		}
		for (x = puid + 1; x < size; x += punum) {
			Type tmp = 0;
			for (y = 0; y < size - x; y++) tmp += T[size - 1 - y][x + y];
			diag1[x + size - 1] = tmp;
		}
		for (y = puid; y < size; y += punum) {
			Type tmp = 0;
			for (x = 0; x <= y; x++) tmp += T[y - x][size - 1 - x];
			diag2[y] = tmp;
		}
		for (y = puid + 1; y < size; y += punum) {
			Type tmp = 0;
			for (x = 0; x < size - y; x++) tmp += T[y + x][x];
			diag2[y + size - 1] = tmp;
		}
		barrier();
		for (y = puid; y < size; y += punum) {
			for (x = 0; x < size; x++) {
				Type h, u, v = T[y][x];
				Type ro = row[y], co = col[x];
				Type d1 = diag1[x + y] - v,
					 d2 = diag2[size - 1 - x + y] - v;
				h = (ro == 0) + (co == 0);
				u = (U[y][x] - (ro + co - 2) - (d1 + d2) + C * h);
				U[y][x] = (u > 15) ? 15 : ((u < -20) ? -20 : u);
				if ((co + ro != 2) || (d1 > 1) || (d2 > 1)) flag = 1;
			}
		}
		barrier();
		if (count > MAXITER) return -1;
		if (flag == 0) return count;
		C = (count % 20 < 5) ? 4 : 1;
		for (y = puid; y < size; y += punum) {
			for (x = 0; x < size; x++) {
				V[y][x] = (U[y][x] > 3);
			}
		}
		if (puid == 0) count++;
	}
#	undef U
#	undef V
#	undef row
#	undef col
#	undef diag1
#	undef diag2
#	undef count
#	undef flag
}

void show(const matrix *a, int size)
{
	int x, y;
	for (y = 0; y < size; y++) {
		for (x = 0; x < size; x++) {
			fputs((*a)[y][x] ? "* " : "- ", stdout);
		}
		putchar('\n');
	}
}

#if 0
void show_potential(const matrix *a, int size)
{
	int x, y;
	for (y = 0; y < size; y++) {
		for (x = 0; x < size; x++) {
			printf("%4d", (*a)[y][x]);
		}
		putchar('\n');
	}
}

void show_vector(const vector *a, int size)
{
	int i;
	for (i = 0; i < size; i++) {
		printf("%3d", (*a)[i]);
	}
	putchar('\n');
}
#endif
