/*
 * <<< radix.c >>>
 *
 * --- Sample application for isis 'radix sort' - for uniprocessor
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#define MAXRADIXBIT			16
#define VALUEBIT			16
#define VALUEMASK			(~(~0 << VALUEBIT))

#define DEFAULT_SIZE		16
#define DEFAULT_RADIXBIT	8
#define MAX_PRINT_SIZE		256

typedef unsigned data_t;

#define rnd() ((unsigned)((seed = 1566083941UL * seed + 1) >> 16))
static unsigned long seed = 1;

static void init(data_t*, int);
static void show(data_t*, int);
static int check(data_t*, int);
static void radix_sort(data_t*, data_t*, int, int);

void init(data_t *a, int size)
{
	int	i;
	for (i = 0; i < size; i++) {
		unsigned long rnd_num = ((unsigned long)rnd() << 16) | rnd();
		a[i] = ((rnd_num >> (32 - VALUEBIT)) & VALUEMASK);
	}
}

void show(data_t *a, int size)
{
	int i;
	for (i = 0; ; i++) {
		printf("%0*x", ((VALUEBIT + 3) / 4), a[i]);
		if (i == size - 1) break;
		putchar((i % 16 != 15) ? '.' : '\n');
	}
	putchar('\n');
}

int check(data_t *a, int size)
{
	int i;
	for (i = 1; i < size; i++) {
		if (a[i] < a[i - 1]) return 0;
	}
	return 1;
}

int main(int argc, char **argv)
{
	data_t *a;
	data_t *w;
	int size = DEFAULT_SIZE, radixbit = DEFAULT_RADIXBIT,
		verbose_flag = 0, check_flag = 0, result;

	/* read arguments */
	while (*++argv != NULL) {
		if (**argv == '-') {
			switch (*++*argv) {
			case 'r':
				radixbit = (isdigit((int)*++(*argv))) ?
						   atoi(*argv) : atoi(*++argv);
				radixbit = (radixbit < 1) ? DEFAULT_RADIXBIT :
						   (radixbit > MAXRADIXBIT) ? MAXRADIXBIT : radixbit;
				break;
			case 't':
				check_flag = 1;
				break;
			case 'v':
				verbose_flag = 1;
				break;
			default:
				break;
			}
		} else if (isdigit((int)**argv)) {
			size = atoi(*argv);
			size = (size < 0) ? DEFAULT_SIZE : size;
		}
	}
	if (verbose_flag) {
		printf("size:%d value:0-%#x radix:%#x\n", size, VALUEMASK,
				~(~0 << radixbit));
	}

	/* initialize */
	a = (data_t*)malloc(size * sizeof(data_t));
	w = (data_t*)malloc(size * sizeof(data_t));
	if (a == NULL || w == NULL) {
		fputs("Out of memory.\n", stderr);
		exit(1);
	}
	init(a, size);
	if (verbose_flag) {
		if (size <= MAX_PRINT_SIZE) {
			puts("source:");
			show(a, size);
		}
	}

	/* calculate */
	radix_sort(a, w, size, radixbit);

	/* check */
	result = ((check_flag) ? check(a, size) : 1);
	if (verbose_flag) {
		if (check_flag) {
			if (result) {
				puts("success.");
			} else {
				puts("failed.");
			}
		}
		if (size <= MAX_PRINT_SIZE) {
			puts("destination:");
			show(a, size);
		}
	}
	return result ? 0 : 1;
}

void radix_sort(data_t *a, data_t *w, const int size, const int radixbit)
{
	int shiftbit, flag;
	flag = 0;
	for (shiftbit = 0; shiftbit < VALUEBIT; shiftbit += radixbit) {
		static int c[1 << MAXRADIXBIT];
		data_t *const src = (!flag) ? a : w, *dst = (!flag) ? w : a;
		const int radix = (1 << radixbit), radixmask = radix - 1;
		int i, j;
		for (i = 0; i < radix; i++) c[i] = 0;
		for (i = 0; i < size; i++) c[(src[i] >> shiftbit) & radixmask]++;
		for (j = c[0], i = 1; i < radix; i++) j = (c[i] += j);
		for (i = size - 1; i >= 0; i--)
			dst[--c[(src[i] >> shiftbit) & radixmask]] = src[i];
		flag = !flag;
	}
	if (flag) {
		int i;
		for (i = 0; i < size; i++) a[i] = w[i];
	}
}
