/*
 * <<< comm_io_funcs.c >>>
 *
 * --- Communication routines with isis::comm_io_unit 'comm_io_funcs'
 *     Copyright (C) 1996-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <sys/types.h>
#include "comm_io_funcs.h"
#include "io_protocol.h"

static ssize_t send_to_host_primitive(const void*, size_t);
static ssize_t receive_from_host_primitive(void*, size_t);

static volatile int *comm_header_adr;
static volatile int *comm_size_adr;
static volatile char *comm_packet_adr;
static int comm_body_size;

ssize_t send_to_host_primitive(const void *buf, size_t size)
{
	volatile int *const header = comm_header_adr;
	volatile int *packet_size;
	volatile char *packet_body;
	size_t i, ret_value;
	if (*header != COMM_IO_READY) return 0;
	packet_size = comm_size_adr;
	packet_body = comm_packet_adr;
	*packet_size = size;
	for (i = 0; i < size; i++) packet_body[i] = ((const char*)buf)[i];
	*header = COMM_IO_SEND_REQUEST;
	while (*header != COMM_IO_ACK && *header != COMM_IO_NACK);
	ret_value = (*header == COMM_IO_ACK) ? *packet_size : 0;
	*header = COMM_IO_READY;
	return (ssize_t)ret_value;
}

ssize_t receive_from_host_primitive(void *buf, size_t size)
{
	volatile int *const header = comm_header_adr;
	volatile int *packet_size;
	volatile const char *packet_body;
	size_t i;
	if (*header != COMM_IO_READY) return 0;
	packet_size = comm_size_adr;
	packet_body = comm_packet_adr;
	*packet_size = size;
	*header = COMM_IO_RECEIVE_REQUEST;
	while (*header != COMM_IO_ACK) {
		if (*header == COMM_IO_NACK) {
			*header = COMM_IO_READY;
			return 0;
		}
	}
	size = *packet_size;
	for (i = 0; i < size; i++) ((char*)buf)[i] = packet_body[i];
	*header = COMM_IO_READY;
	return (ssize_t)size;
}

void __comm_io_init(void *adr)
{
	comm_body_size = *(volatile int*)adr;
	comm_header_adr = (volatile int*)adr + 1;
	comm_size_adr = comm_header_adr + 1;
	comm_packet_adr = (char*)(comm_size_adr + 1);
}

ssize_t __comm_io_send_to_host(const void *buf, size_t size)
{
	const char *p = (const char*)buf;
	int total_size = 0;
	while (1) {
		int s = (size > comm_body_size) ? comm_body_size : size;
		int result;
		size -= s;
		result = send_to_host_primitive(p, s);
		p += s;
		total_size += result;
		if (result < s || size == 0) break;
	}
	return total_size;
}

ssize_t __comm_io_receive_from_host(void *buf, size_t size)
{
	char *p = (char*)buf;
	int total_size = 0;
	while (1) {
		int s = (size > comm_body_size) ? comm_body_size : size;
		int result;
		size -= s;
		result = receive_from_host_primitive(p, s);
		p += s;
		total_size += result;
		if (result < s || size == 0) break;
	}
	return total_size;
}
