/*
 * <<< sh_matrix.c >>>
 *
 * --- Sample application for isis 'matrix multiply' - for multiprocessor
 *     Copyright (C) 1997-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include "osiris.h"
#include "shlib.h"

#define DEFAULT_SIZE	4
#define MAXSIZE			32

typedef int Type;
typedef Type matrix[MAXSIZE][MAXSIZE];

#define rnd() ((unsigned)((seed = 1566083941UL * seed + 1) >> 16))
unsigned long seed = 1;

static void init(matrix*, int);
static void copy(const matrix*, matrix*, int);
static void mult(const matrix*, const matrix*, matrix*, int);
static void show(const matrix*, int);

int main(int argc, char **argv)
{
	matrix *shared_a = shared_malloc(sizeof(matrix));
	matrix *shared_b = shared_malloc(sizeof(matrix));
	matrix *shared_c = shared_malloc(sizeof(matrix));
	static matrix a, b; /* "static" -- because of poor stack size */
	int verbose_flag = 0, puid = get_puid();
	int size;
	if (shared_a == NULL || shared_b == NULL || shared_c == NULL) {
		fputs("Out of memory.\n", stderr);
		exit(1);
	}
	while (*++argv != NULL && **argv == '-') {
		switch (*++*argv) {
		case 'v':
			verbose_flag = 1;
			break;
		default:
			break;
		}
	}
	if (*argv != NULL) {
		size = atoi(*argv);
		size = (size < 0) ? DEFAULT_SIZE : ((size > MAXSIZE) ? MAXSIZE : size);
	} else {
		size = DEFAULT_SIZE;
	}
	if (puid == 0) {
		init(shared_a, size);
		init(shared_b, size);
	}
	barrier();
	copy((const matrix*)shared_a, &a, size);
	copy((const matrix*)shared_b, &b, size);
	if (verbose_flag && puid == 0) {
		printf("size : %d\n", size);
		puts("A[][] =");
		show((const matrix*)(&a), size);
		puts("\nB[][] =");
		show((const matrix*)(&b), size);
		putchar('\n');
	}
	mult((const matrix*)(&a), (const matrix*)(&b), shared_c, size);
	barrier();
	if (verbose_flag && puid == 0) {
		puts("C[][] =");
		show((const matrix*)shared_c, size);
	}
	return 0;
}

void init(matrix *a, int size)
{
	int	i, j;
	for (i = 0; i < size; i++) {
		for (j = 0; j < size; j++) {
			(*a)[i][j] = ((rnd() >> 12) & 0x3) + 1;
		}
	}
}

void copy(const matrix *src, matrix *dst, int size)
{
	int	i, j;
	for (i = 0; i < size; i++) {
		for (j = 0; j < size; j++) {
			(*dst)[i][j] = (*src)[i][j];
		}
	}
}

void mult(const matrix *a, const matrix *b, matrix *c, int size)
{
	int punum = get_punum(), puid = get_puid();
	int i, j, k;
	for (i = puid; i < size; i += punum) {
		for (j = 0; j < size; j++) {
			int sum = 0;
			for (k = 0; k < size; k++) sum += (*a)[i][k] * (*b)[k][j];
			(*c)[i][j] = sum;
		}
	}
}

void show(const matrix *a, int size)
{
	int i, j;
	for (i = 0; i < size; i++) {
		for (j = 0; j < size; j++) {
			printf(" %3d", (*a)[i][j]);
		}
		putchar('\n');
	}
}
