/*
 * <<< fft.c >>>
 *
 * --- Sample application for isis 'fast fourier transform' - for uniprocessor
 *     Copyright (C) 1995-2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define PI 3.14159265358979323846

#define DEFAULT_SIZE	64
#define MAXSIZE			256
#define DIFF_LIMIT		1e-7

typedef double data_t;
typedef struct { data_t re, im; } complex;

static void fft(complex*, int); 
static void ufft(complex*, int); 
static void make_rotate_table(complex*, int, int);
static void make_data(complex*, int);
static void copy_data(const complex*, complex*, int);
static int check(const complex*, const complex*, int);
#if 0
static void show(const complex*, int);
#endif

int main(int argc, char **argv)
{
	complex data1[MAXSIZE], data2[MAXSIZE], data3[MAXSIZE];
	int n = DEFAULT_SIZE, check_flag = 0, verbose_flag = 0, result;
	while (*++argv != NULL) {
		if (**argv == '-') {
			switch (*++*argv) {
			case 't':
				check_flag = 1;
				break;
			case 'v':
				verbose_flag = 1;
				break;
			default:
				break;
			}
		} else if (isdigit((int)**argv)) {
			n = atoi(*argv);
			n = (n < 0) ? DEFAULT_SIZE : ((n > MAXSIZE) ? MAXSIZE : n);
		}
	}
	make_data(data1, n);
	copy_data(data1, data2, n);
	fft(data2, n);
	copy_data(data2, data3, n);
	ufft(data3, n);
	result = (check_flag) ? check(data1, data3, n) : 1;
	if (verbose_flag) {
		if (n <= 64) {
			int i;
			printf("idx --- "
				   "source:           "
				   "fft:              "
				   "ufft:             "
				   "diff:\n");
			for (i = 0; i < n; i++) {
				double dx = data1[i].re - data3[i].re;
				double dy = data1[i].im - data3[i].im;
				double d = sqrt(dx * dx + dy * dy);
				printf("%3d --- ", i);
				printf("(% 5.3f, % 5.3f)  ", data1[i].re, data1[i].im);
				printf("(% 5.3f, % 5.3f)  ", data2[i].re, data2[i].im);
				printf("(% 5.3f, % 5.3f)  ", data3[i].re, data3[i].im);
				printf("%9.3e\n", d);
			}
		} else {
			int i;
			puts("\"source-re\"");
			for (i = 0; i < n; i++) printf("%3d % 5.3f\n", i, data1[i].re);
			puts("\n\"fft-re\"");
			for (i = 0; i < n; i++) printf("%3d % 5.3f\n", i, data2[i].re);
			puts("\n\"fft-im\"");
			for (i = 0; i < n; i++) printf("%3d % 5.3f\n", i, data2[i].im);
			puts("\n\"ufft-re\"");
			for (i = 0; i < n; i++) printf("%3d % 5.3f\n", i, data3[i].re);
		}
		if (check_flag) puts(result ? "success." : "failed.");
	}
	return result ? 0 : 1;
}

void fft(complex *z, int n)
{
	static complex rotate_table[MAXSIZE];
	int i;
	make_rotate_table(rotate_table, n, 0);
	for (i = n; i >= 2; i /= 2) {
		int j;
		for (j = 0; j < n; j += i) {
			int k;
			for (k = 0; k < (i / 2); k++) {
				complex *z0_p = z + j + k;
				complex *z1_p = z0_p + (i / 2);
				complex tmp0, tmp1;
				complex *rot_p = &rotate_table[k * n / i];
				tmp0.re = z0_p->re, tmp0.im = z0_p->im;
				tmp1.re = z1_p->re, tmp1.im = z1_p->im;
				z0_p->re += tmp1.re, z0_p->im += tmp1.im;
				tmp1.re = tmp0.re - tmp1.re, tmp1.im = tmp0.im - tmp1.im;
				z1_p->re = tmp1.re * rot_p->re - tmp1.im * rot_p->im;
				z1_p->im = tmp1.re * rot_p->im + tmp1.im * rot_p->re;
			}
		}
	}
	for (i = 0; i < n; i++) z[i].re /= n, z[i].im /= n;
}

void ufft(complex *z, int n)
{
	static complex rotate_table[MAXSIZE];
	int i;
	make_rotate_table(rotate_table, n, 1);
	for (i = 2; i <= n; i <<= 1) {
		int j;
		for (j = 0; j < n; j += i) {
			int k;
			for (k = 0; k < (i / 2); k++) {
				complex tmp0, tmp1;
				complex *z0_p = &z[j + k];
				complex *z1_p = &z[j + k + (i / 2)];
				tmp0.re = z0_p->re, tmp0.im = z0_p->im;
				tmp1.re = z1_p->re, tmp1.im = z1_p->im;
				if (i > 2) {
					complex tmp;
					complex *rot_p = &rotate_table[k * n / i];
					tmp.re = tmp1.re * rot_p->re - tmp1.im * rot_p->im;
					tmp.im = tmp1.re * rot_p->im + tmp1.im * rot_p->re;
					tmp1 = tmp;
				}
				z0_p->re = tmp0.re + tmp1.re, z0_p->im = tmp0.im + tmp1.im;
				z1_p->re = tmp0.re - tmp1.re, z1_p->im = tmp0.im - tmp1.im;
			}
		}
	}
}

void make_rotate_table(complex *table, int n, int inverse_flag)
{
	int i;
	for (i = 0; i < n / 2; i++) {
		table[i].re = cos(2 * PI * i / n);
		table[i].im = sin(2 * PI * i / n);
		if (!inverse_flag) table[i].im = -table[i].im;
	}
}

void make_data(complex* z, int n)
{
	int i;
	for (i = 0; i < n; i++) {
		/* z[i].re = (i == 1); */
		/* z[i].re = sin(3 * PI * i / n); */
		/* z[i].re = (i % (n / 8) == 0); */
		z[i].re = 3 * cos(3 * 2 * PI * i / n)
				+ 2 * sin(9 * 2 * PI * i / n)
				- 1;
		z[i].im = 0;
	}
}

void copy_data(const complex* src, complex* dst, int n)
{
	int i;
	for (i = 0; i < n; i++) dst[i] = src[i];
}

int check(const complex *z1, const complex *z2, int n)
{
	double diff_sigma = 0;
	int i;
	for (i = 0; i < n; i++) {
		double dx = z1[i].re - z2[i].re;
		double dy = z1[i].im - z2[i].im;
		diff_sigma += sqrt(dx * dx + dy * dy);
	}
	return diff_sigma < DIFF_LIMIT;
}

#if 0
void show(const complex *z, int n)
{
	int i;
	for (i = 0; i < n; i++) {
		printf("%3d: % 5.3f, % 5.3f\n", i, z[i].re, z[i].im);
	}
	printf("\n");
}
#endif
