###############################################
# library                       
###############################################

source ./env.tcl
source ./com.tcl

###############################################
# make library                       
###############################################

set mw_ref_libs ""
foreach lib $libs {
	lappend mw_ref_libs $mw_path/$lib
}

create_mw_lib $top_design \
	-technology  $tech_file \
	-mw_reference_library $mw_ref_libs \
	-open

set_tlu_plus_files \
	-max_tluplus $tlup_max \
	-min_tluplus $tlup_min \
	-tech2itf_map  $tlup_map 
check_tlu_plus_files

set mw_logic0_net "VSS"
set mw_logic1_net "VDD"

import_designs -format verilog {../../net/Ring/MULT_UNIT.v} \
	-cel $top_design \
	-top mult_unit

read_sdc $sdc_file
read_sdf $sdf_file

report_timing_requirements

###############################################
# floorplan                       
###############################################

initialize_floorplan \
				-control_type row_number \
				-core_utilization 0.65 \
				-number_row 156 \
				-start_first_row \
				-flip_first_row \
				-left_io2core 2 \
				-bottom_io2core 2 \
				-right_io2core 2 \
				-top_io2core 2


add_end_cap -lib_cell {SC23YUZCUBAS321} -mode bottom_left

set_ignored_layers    -min_routing_layer MET1 -max_routing_layer MET6

initialize_pg

add_tap_cell_array -master_cell_name {SC23YUZTAP021} \
				-offset -13.6 \
                   -distance 30 \
                   -left_boundary_extra_tap no_insert \
                   -right_boundary_extra_tap no_insert \
                   -left_macro_blockage_extra_tap no_insert \
                   -right_macro_blockage_extra_tap no_insert \
                   -connect_power_name {VDD} \
                   -connect_ground_name {VSS} \
				-ignore_soft_blockage true

report_fp_placement_strategy
create_fp_placement
legalize_placement

set_preroute_drc_strategy  -report_fail  -min_layer MET1  -max_layer METG2

create_power_straps \
	 -direction vertical  \
	 -start_at 5.0 -num_placement_strap 7 -increment_x_or_y 30 -nets  {VSS VDD} \
	 -layer METS1 -width 0.8 \
	 -pitch_within_group 15.0 \
	 -extend_low_ends force_to_boundary_and_generate_pins \
	 -extend_high_ends force_to_boundary_and_generate_pins

insert_stdcell_filler \
	 -cell_without_metal $filler_without_metal_list \
	 -connect_to_power {VDD}  -connect_to_ground {VSS}

preroute_standard_cells \
	-nets  {VDD VSS} \
	-connect horizontal -extend_to_boundaries_and_generate_pins  \
	-port_filter_mode off -cell_master_filter_mode off \
	-cell_instance_filter_mode off -voltage_area_filter_mode off \
	-do_not_route_over_macros

verify_pg_nets  -macro_pin_connection all -pad_pin_connection all -error_cell ${top_design}_pg
open_mw_cel -not_as_current -readonly ${top_design}_pg

analyze_fp_rail  -nets {VDD VSS} -voltage_supply 1.2 -use_pins_as_pa

remove_stdcell_filler  -stdcell

set_pnet_options -complete {MET6}
report_pnet_options
create_fp_placement  -timing_driven -incremental all

extract_rc
report_timing_preroute

save_mw_cel 
save_mw_cel -as planned

###############################################
# placement
###############################################

report_clock
report_clock -skew

check_physical_design -for_placement
check_physical_constraints -v

create_placement
legalize_placement

report_timing_preroute

create_placement -timing_driven -congestion
legalize_placement

report_ahfs_options

report_timing_preroute

insert_stdcell_filler \
         -cell_without_metal $filler_without_metal_list \
         -connect_to_power {VDD}  -connect_to_ground {VSS}

save_mw_cel -as placed

###############################################
# routing
###############################################

report_timing
report_timing -delay min

check_routeability -error_cell ${top_design}.err

report_route_opt_strategy

route_opt

save_mw_cel
verify_route
verify_drc -read_cell_view -check_via_size -check_enclosure -check_end_of_line -check_via_farm -check_fat_poly_contact -check_blockage

set_route_options \
	 -groute_timing_driven true \
      -groute_skew_control false \
	 -groute_congestion_weight 4 \
	 -groute_clock_routing balanced \
	 -groute_incremental true \
	 -track_assign_timing_driven true \
	 -droute_connect_tie_off true \
	 -droute_connect_open_nets true \
	 -droute_reroute_user_wires false \
	 -droute_CTS_nets minor_change_only \
      -droute_single_row_column_via_array center \
	 -droute_stack_via_less_than_min_area add_metal_stub \
	 -droute_stack_via_less_than_min_area_cost 0 \
	 -poly_pin_access auto \
	 -drc_distance diagonal \
	 -same_net_notch check_and_fix \
	 -merge_fat_wire_on signal_routing_too \
	 -fat_wire_check merge_then_check \
	 -fat_blockage_as fat_wire \
	 -wire_contact_eol_rule check_and_fix

route_search_repair -loop 10

save_mw_cel
initialize_pg
verify_drc -read_cell_view -check_via_size -check_enclosure \
	-check_end_of_line -check_via_farm -check_fat_poly_contact -check_blockage
verify_lvs


set_si_options \
	-route_xtalk_prevention ture \
	-delta_delay true \
	-static_noise true 

report_constraint -all
report_timing -crosstalk_delta

route_opt -incremental -xtalk_reduction
report_constraint -all


save_mw_cel 
save_mw_cel -as routed
initialize_pg
verify_drc -read_cell_view -check_via_size -check_enclosure -check_end_of_line -check_via_farm -check_fat_poly_contact -check_blockage
verify_lvs

###############################################
# finishing
###############################################

route_search_repair -loop 100 -rerun_drc -trim_antenna_of_user_wires
route_global
route_spreadwires -timing_driven -search_repair_loop 5
route_search_repair -loop 100 -rerun_drc -trim_antenna_of_user_wires

set_parameter -module droute -name  doAntennaConx  -value 4
source  -e -v ../lib/antenna_rule_icc.sc
report_antenna_rules
report_antenna_ratio
route_search_repair -rerun_drc -loop 1
insert_diode
initialize_pg
#sizeof [get_cells -hierarchical antenna_diode*]

verify_route -antenna

insert_stdcell_filler \
	  -cell_without_metal $filler_without_metal_list \
	  -connect_to_power {VDD}  -connect_to_ground {VSS}

initialize_pg
save_mw_cel 
verify_drc -read_cell_view \
	-check_via_size \
	-check_enclosure \
	-check_end_of_line \
	-check_via_farm \
	-check_fat_poly_contact \
	-check_blockage
verify_lvs
verify_pg_nets  -macro_pin_connection all \
 -pad_pin_connection all -error_cell ${top_design}_pg

set_delay_calculation -arnoldi
report_constraints
report_timing

###############################################
# output
###############################################

change_names  -hierarchy -rules  verilog -v
save_mw_cel -as ${top_design}
write_verilog \
	-force_output_references $filler_list \
	 -output_net_name_for_pg \
	 -diode_ports -pg_ports -split_bus \
	 ./outputs/${top_design}_lvs.v

write -format verilog -hierarchy -output ${top_design}.v

set_write_stream_options -map_layer ../lib/gdsout.map \
	 -output_pin {text geometry} -child_depth 10000
#set_write_stream_options -map_layer ../lib/gdsout.map \
#	-skip_ref_lib_cells -output_pin {text geometry}
write_stream  -lib_name ${top_design} -cells ${top_design} \
	 -format gds ./outputs/${top_design}.gds

write_sdf -version 3.0 ./outputs/${top_design}.sdf
extract_rc
write_parasitics -output ./outputs/${top_design}.spef

create_macro_fram -library_name ${top_design} -cell_name ${top_design} \
	-routing_blockage_output_layer metBlk \
	-treat_metal_blockage_as_thin { poly m1 m2 m3 m4 m5 m6 m7 m8 m9 m10 m11 m12 m13 m14 m15} \
	-extract_blockage_by_block_core_with_margin "m1 m2 m3 m4 m5 m6 m7 m8 m9 m10 m11 m12 m13 m14 m15"  \
	-extract_pin_connectivity_through { v1 v2 v3 v4 v5 v6 v7 v8 v9 v10 v11 v12 v13 v14} 

extract_hier_antenna_property  -cell_name  $top_design

create_ilm \
	-include_side_load boundary \
	-latch_level 1 \
	-verbose \
	-keep_boundary_cells \
	-keep_full_clock_tree \
	-keep_parasitics \
	-keep_macro \
	-include_xtalk

write_sdc ./outputs/${top_design}.sdc
write  -format ddc   -output ./outputs/${top_design}.ddc
