/* loader.h */

/* SimpleScalar(TM) Tool Suite
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 * All Rights Reserved. 
 * 
 * THIS IS A LEGAL DOCUMENT, BY USING SIMPLESCALAR,
 * YOU ARE AGREEING TO THESE TERMS AND CONDITIONS.
 * 
 * No portion of this work may be used by any commercial entity, or for any
 * commercial purpose, without the prior, written permission of SimpleScalar,
 * LLC (info@simplescalar.com). Nonprofit and noncommercial use is permitted
 * as described below.
 * 
 * 1. SimpleScalar is provided AS IS, with no warranty of any kind, express
 * or implied. The user of the program accepts full responsibility for the
 * application of the program and the use of any results.
 * 
 * 2. Nonprofit and noncommercial use is encouraged. SimpleScalar may be
 * downloaded, compiled, executed, copied, and modified solely for nonprofit,
 * educational, noncommercial research, and noncommercial scholarship
 * purposes provided that this notice in its entirety accompanies all copies.
 * Copies of the modified software can be delivered to persons who use it
 * solely for nonprofit, educational, noncommercial research, and
 * noncommercial scholarship purposes provided that this notice in its
 * entirety accompanies all copies.
 * 
 * 3. ALL COMMERCIAL USE, AND ALL USE BY FOR PROFIT ENTITIES, IS EXPRESSLY
 * PROHIBITED WITHOUT A LICENSE FROM SIMPLESCALAR, LLC (info@simplescalar.com).
 * 
 * 4. No nonprofit user may place any restrictions on the use of this software,
 * including as modified by the user, by any other authorized user.
 * 
 * 5. Noncommercial and nonprofit users may distribute copies of SimpleScalar
 * in compiled or executable form as set forth in Section 2, provided that
 * either: (A) it is accompanied by the corresponding machine-readable source
 * code, or (B) it is accompanied by a written offer, with no time limit, to
 * give anyone a machine-readable copy of the corresponding source code in
 * return for reimbursement of the cost of distribution. This written offer
 * must permit verbatim duplication by anyone, or (C) it is distributed by
 * someone who received only the executable form, and is accompanied by a
 * copy of the written offer of source code.
 * 
 * 6. SimpleScalar was developed by Todd M. Austin, Ph.D. The tool suite is
 * currently maintained by SimpleScalar LLC (info@simplescalar.com). US Mail:
 * 2395 Timbercrest Court, Ann Arbor, MI 48105.
 * 
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 */
/*
 * This module implements program loading.  The program text (code) and
 * initialized data are first read from the program executable.  Next, the
 * program uninitialized data segment is initialized to all zero's.  Finally,
 * the program stack is initialized with command line arguments and
 * environment variables.  The format of the top of stack when the program
 * starts execution is as follows:
 *
 * 0x7fffffff    +----------+
 *               | unused   |
 * 0x7fffc000    +----------+
 *               | envp     |
 *               | strings  |
 *               +----------+
 *               | argv     |
 *               | strings  |
 *               +----------+
 *               | envp     |
 *               | array    |
 *               +----------+
 *               | argv     |
 *               | array    |
 *               +----------+
 *               | argc     |
 * regs_R[29]    +----------+
 * (stack ptr)
 *
 * NOTE: the start of envp is computed in crt0.o (C startup code) using the
 * value of argc and the computed size of the argv array, the envp array size
 * is not specified, but rather it is NULL terminated, so the startup code
 * has to scan memory for the end of the string.
 */

#ifndef LOADER_SIMPLE_H
#define LOADER_SIMPLE_H

#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#ifdef _MSC_VER
#include <io.h>
#include <time.h>
#else /* !_MSC_VER */
#include <unistd.h>
#include <sys/time.h>
#endif

#include <isis/host.h>
#include <isis/misc.h>
#include <isis/machine.h>
#include <isis/memory_s.h>
#include <isis/endian_s.h>
#include <isis/libexo.h>
#include <isis/regs_head.h>

/* EIO file formats */
#define EIO_PISA_FORMAT			1
#define EIO_ALPHA_FORMAT		2
/* EIO file version */
#define EIO_FILE_VERSION		3
#ifdef _MSC_VER
#define write		_write
#endif
#define EIO_FILE_HEADER							\
  "/* This is a SimpleScalar EIO file - DO NOT MOVE OR EDIT THIS LINE! */\n"
#define BLOB_TAIL_SIZE		256

#ifdef BFD_LOADER
#include <bfd.h>
#else /* !BFD_LOADER */
#include <isis/ecoff.h>
#endif /* BFD_LOADER */

#define TEXT_TAIL_PADDING 128

class loader_simple
{
 private:
  typedef memory_simple memory_simple_type;
  typedef stats_simple stats_simple_type;  
  memory_simple_type *mem_simple;
  stats_simple_type *st_simple;
  endian_simple endian;
  md_addr_t ld_environ_base;
  counter_t eio_trans_icnt;
  typedef libexo libexo_type;
  libexo_type lib;
/* open(2) flags translation table for SimpleScalar target */
  struct {
	int ss_flag;
	int local_flag;
  } ss_flag_table[10];
  
#define SS_NFLAGS	(sizeof(ss_flag_table)/sizeof(ss_flag_table[0]))


 public:
  FILE *eio_create(char *fname);  
  FILE *eio_open(char *fname);
  /* returns non-zero if file FNAME has a valid EIO header */
  int eio_valid(char *fname);
  void eio_close(FILE *fd);
  /* check point current architected state to stream FD, returns
	 EIO transaction count (an EIO file pointer) */
  counter_t eio_write_chkpt(struct regs_t *regs,		/* regs to dump */
							struct mem_t *mem,		/* memory to dump */
							FILE *fd);			/* stream to write to */
  /* read check point of architected state from stream FD, returns
	 EIO transaction count (an EIO file pointer) */
  counter_t eio_read_chkpt(struct regs_t *regs,		/* regs to dump */
						   struct mem_t *mem,		/* memory to dump */
						   FILE *fd);			/* stream to read */
  
  /* syscall proxy handler from an EIO trace, architect registers
	 and memory are assumed to be precise when this function is called,
	 register and memory are updated with the results of the sustem call */
  void eio_read_trace(FILE *eio_fd,			/* EIO stream file desc */
					  counter_t icnt,			/* instruction count */
					  struct regs_t *regs,		/* registers to update */
					  struct mem_t *mem,		/* memory to update */
					  md_inst_t inst);			/* system call inst */
  
  /* fast forward EIO trace EIO_FD to the transaction just after ICNT */
  void eio_fast_forward(FILE *eio_fd, counter_t icnt);
  void sys_syscall(struct regs_t *regs,	/* registers to access */
				   struct mem_t *mem,	/* memory space to access */
				   md_inst_t inst,		/* system call inst */
				   int traceable,
				   size_t,		/* traceable system call? */  
				   counter_t);

#define MD_MISC_REGS_TO_EXO(REGS, SIM_NUM_INSN)					 \
  lib.exo_new(ec_list,							\
	  /*icnt*/lib.exo_new(ec_integer, (exo_integer_t)SIM_NUM_INSN),	\
	  /*PC*/lib.exo_new(ec_address, (exo_integer_t)(REGS)->regs_PC),	\
	  /*NPC*/lib.exo_new(ec_address, (exo_integer_t)(REGS)->regs_NPC),	\
	  /*HI*/lib.exo_new(ec_integer, (exo_integer_t)(REGS)->regs_C.hi),	\
	  /*LO*/lib.exo_new(ec_integer, (exo_integer_t)(REGS)->regs_C.lo),	\
	  /*FCC*/lib.exo_new(ec_integer, (exo_integer_t)(REGS)->regs_C.fcc),\
	  NULL)

#define MD_IREG_TO_EXO(REGS, IDX)					\
  lib.exo_new(ec_address, (exo_integer_t)(REGS)->regs_R[IDX])

#define MD_FREG_TO_EXO(REGS, IDX)					\
  lib.exo_new(ec_address, (exo_integer_t)(REGS)->regs_F.l[IDX])

  void connect_eval(jmp_buf* buf, FILE* file, char* a, char* b) {
	sim_exit_buf = buf;
	sim_progfd = file;
	sim_chkpt_fname = a;
	sim_eio_fname = b;
  }

 public:
  jmp_buf *sim_exit_buf;
  FILE *sim_progfd;
  FILE *sim_eio_fd;
  char *sim_chkpt_fname;
  char *sim_eio_fname;
  
  md_addr_t ld_text_base;
  unsigned int ld_text_size;
  md_addr_t ld_data_base;
  unsigned int ld_data_size;
  md_addr_t ld_brk_point;
  md_addr_t ld_stack_base;
  unsigned int ld_stack_size;
  md_addr_t ld_stack_min;
  char *ld_prog_fname;
  md_addr_t ld_prog_entry;
  int ld_target_big_endian;

 public:
  loader_simple();

  void connect_mem(memory_simple_type& a){ mem_simple = &a; }
  void connect_stats(stats_simple_type& a){ st_simple = &a; }

  void regs_init(struct regs_t *regs){	/* register file to initialize */
	memset(regs, 0, sizeof(*regs));
  }


  /* register simulator-specific statistics */
  void
	ld_reg_stats(struct stat_sdb_t *sdb);	/* stats data base */
  
  /* load program text and initialized data into simulated virtual memory
	 space and initialize program segment range variables */
  void
	ld_load_prog(char *fname,		/* program to load */
				 int argc, char **argv,	/* simulated program cmd line args */
				 char **envp,		/* simulated program environment */
				 struct regs_t *regs,	/* registers to initialize for load */
				 struct mem_t *mem,		/* memory space to load prog into */
				 int zero_bss_segs);	/* zero uninit data segment? */
#ifndef HOST_ONLY
  /* probe target (simulated program) byte endian format, only
	 valid after program has been loaded */
  enum endian_t endian_target_byte_order(void);
  
  /* probe target (simulated program) double word endian format,
	 only valid after program has been loaded */
  enum endian_t endian_target_word_order(void);

#endif /* HOST_ONLY */
};

#endif /* LOADER_H */
