/* stats.h - statistical package interfaces */

/* SimpleScalar(TM) Tool Suite
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 * All Rights Reserved. 
 * 
 * THIS IS A LEGAL DOCUMENT, BY USING SIMPLESCALAR,
 * YOU ARE AGREEING TO THESE TERMS AND CONDITIONS.
 * 
 * No portion of this work may be used by any commercial entity, or for any
 * commercial purpose, without the prior, written permission of SimpleScalar,
 * LLC (info@simplescalar.com). Nonprofit and noncommercial use is permitted
 * as described below.
 * 
 * 1. SimpleScalar is provided AS IS, with no warranty of any kind, express
 * or implied. The user of the program accepts full responsibility for the
 * application of the program and the use of any results.
 * 
 * 2. Nonprofit and noncommercial use is encouraged. SimpleScalar may be
 * downloaded, compiled, executed, copied, and modified solely for nonprofit,
 * educational, noncommercial research, and noncommercial scholarship
 * purposes provided that this notice in its entirety accompanies all copies.
 * Copies of the modified software can be delivered to persons who use it
 * solely for nonprofit, educational, noncommercial research, and
 * noncommercial scholarship purposes provided that this notice in its
 * entirety accompanies all copies.
 * 
 * 3. ALL COMMERCIAL USE, AND ALL USE BY FOR PROFIT ENTITIES, IS EXPRESSLY
 * PROHIBITED WITHOUT A LICENSE FROM SIMPLESCALAR, LLC (info@simplescalar.com).
 * 
 * 4. No nonprofit user may place any restrictions on the use of this software,
 * including as modified by the user, by any other authorized user.
 * 
 * 5. Noncommercial and nonprofit users may distribute copies of SimpleScalar
 * in compiled or executable form as set forth in Section 2, provided that
 * either: (A) it is accompanied by the corresponding machine-readable source
 * code, or (B) it is accompanied by a written offer, with no time limit, to
 * give anyone a machine-readable copy of the corresponding source code in
 * return for reimbursement of the cost of distribution. This written offer
 * must permit verbatim duplication by anyone, or (C) it is distributed by
 * someone who received only the executable form, and is accompanied by a
 * copy of the written offer of source code.
 * 
 * 6. SimpleScalar was developed by Todd M. Austin, Ph.D. The tool suite is
 * currently maintained by SimpleScalar LLC (info@simplescalar.com). US Mail:
 * 2395 Timbercrest Court, Ann Arbor, MI 48105.
 * 
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 */


#ifndef STAT_H
#define STAT_H

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <math.h>
#include <ctype.h>
#include <errno.h>

#include <isis/host.h>
#include <isis/machine.h>
#include <isis/misc.h>
#include <isis/eval.h>

/* stat variable classes */
enum stat_class_t {
  sc_int = 0,			/* integer stat */
  sc_uint,			/* unsigned integer stat */
#ifdef HOST_HAS_QWORD
  sc_qword,			/* qword integer stat */
  sc_sqword,			/* signed qword integer stat */
#endif /* HOST_HAS_QWORD */
  sc_float,			/* single-precision FP stat */
  sc_double,			/* double-precision FP stat */
  sc_dist,			/* array distribution stat */
  sc_sdist,			/* sparse array distribution stat */
  sc_formula,			/* stat expression formula */
  sc_NUM,
  sc_comment,
  sc_comment2
};
/* sparse array distributions are implemented with a hash table */
#define HTAB_SZ			1024
#define HTAB_HASH(I)		((((I) >> 8) ^ (I)) & (HTAB_SZ - 1))
/* hash table bucket definition */
struct bucket_t {
  struct bucket_t *next;	/* pointer to the next bucket */
  md_addr_t index;		/* bucket index - as large as an addr */
  unsigned int count;		/* bucket count */
};
/* forward declaration */
struct stat_stat_t;
/* enable distribution components:  index, count, probability, cumulative */
#define PF_COUNT		0x0001
#define PF_PDF			0x0002
#define PF_CDF			0x0004
#define PF_ALL			(PF_COUNT|PF_PDF|PF_CDF)
/* statistical variable definition */
struct stat_stat_t {
  struct stat_stat_t *next;	/* pointer to next stat in database list */
  char *name;			/* stat name */
  char *desc;			/* stat description */
  const char *format;			/* stat output print format */
  enum stat_class_t sc;		/* stat class */
  int *num;
  union stat_variant_t {
    /* sc == sc_int */
    struct stat_for_int_t {
      int *var;			/* integer stat variable */
      int init_val;		/* initial integer value */
    } for_int;
    /* sc == sc_uint */
    struct stat_for_uint_t {
      unsigned int *var;	/* unsigned integer stat variable */
      unsigned int init_val;	/* initial unsigned integer value */
    } for_uint;
#ifdef HOST_HAS_QWORD
    /* sc == sc_qword */
    struct stat_for_qword_t {
      qword_t *var;		/* qword integer stat variable */
      qword_t init_val;		/* qword integer value */
    } for_qword;
    /* sc == sc_sqword */
    struct stat_for_sqword_t {
      sqword_t *var;		/* signed qword integer stat variable */
      sqword_t init_val;	/* signed qword integer value */
    } for_sqword;
#endif /* HOST_HAS_QWORD */
    /* sc == sc_float */
    struct stat_for_float_t {
      float *var;		/* float stat variable */
      float init_val;		/* initial float value */
    } for_float;
    /* sc == sc_double */
    struct stat_for_double_t {
      double *var;		/* double stat variable */
      double init_val;		/* initial double value */
    } for_double;
    /* sc == sc_dist */
    struct stat_for_dist_t {
      unsigned int init_val;	/* initial dist value */
      unsigned int *arr;	/* non-sparse array pointer */
      unsigned int arr_sz;	/* array size */
      unsigned int bucket_sz;	/* array bucket size */
      int pf;			/* printables */
      char **imap;		/* index -> string map */
#if 0
      print_fn_t print_fn;	/* optional user-specified print fn */
#endif
      unsigned int overflows;	/* total overflows in stat_add_samples() */
    } for_dist;
    /* sc == sc_sdist */
    struct stat_for_sdist_t {
      unsigned int init_val;	/* initial dist value */
      struct bucket_t **sarr;	/* sparse array pointer */
      int pf;			/* printables */
#if 0
      print_fn_t print_fn;	/* optional user-specified print fn */
#endif
    } for_sdist;
    /* sc == sc_formula */
    struct stat_for_formula_t {
      char *formula;		/* stat formula, see eval.h for format */
    } for_formula;
  } variant;
};
/* statistical database */
struct stat_sdb_t {
  struct stat_stat_t *stats;		/* list of stats in database */
  struct eval_state_t *evaluator;	/* an expression evaluator */
};

/*
 * The stats package is a uniform module for handling statistical variables,
 * including counters, distributions, and expressions.  The user must first
 * create a stats database using stat_new(), then statical counters are added
 * to the database using the *_reg_*() functions.  Interfaces are included to
 * allocate and manipulate distributions (histograms) and general expression
 * of other statistical variables constants.  Statistical variables can be
 * located by name using stat_find_stat().  And, statistics can be print in
 * a highly standardized and stylized fashion using stat_print_stats().
 */

class stats_simple
{
 public:
  stats_simple();
  struct eval_value_t stat_eval_ident_fn(struct eval_state_t *es);

  struct stat_stat_t * stat_reg_comment(struct stat_sdb_t *sdb,
										char *comment1,
										char *comment2,
										int *init_val);
  struct stat_stat_t * stat_reg_comment2(struct stat_sdb_t *sdb,
										 char *comment);
  struct stat_stat_t * stat_reg_int(struct stat_sdb_t *sdb,	/* stat database */
									char *name,	/* stat variable name */
									char *desc,	/* stat variable description */
									int *var,		/* stat variable */
									/*stat variable initial value*/
									int init_val,
									/* optional variable output format */
									const char *format);
  /* register a signed qword integer statistical variable */
  struct stat_stat_t * stat_reg_sqword(struct stat_sdb_t *sdb,/*stat database*/
									   char *name,		/* stat variable name*/
									   char *desc,/*stat variable description*/
									   sqword_t *var,		/* stat variable */
									   /* stat variable initial value */
									   sqword_t init_val,	
									   /* optional variable output format */
									   const char *format); 

  struct stat_stat_t * stat_reg_formula(struct stat_sdb_t *sdb,
										char *name,	/* stat variable name */
										/*stat variable description*/
										char *desc,
										char *formula, /* formula expression */
										/* optional variable output format */
										const char *format); 

  /* find a stat variable, returns NULL if it is not found */
  struct stat_stat_t * stat_find_stat(struct stat_sdb_t *sdb, /*stat database*/
									  char *stat_name);	/* stat name */

  void stat_add_samples(struct stat_stat_t *stat, /* stat database */
						md_addr_t index,	/* distribution index of samples */
						int nsamples);	/* number of samples to add to dist */
  
  struct stat_stat_t * stat_reg_sdist(struct stat_sdb_t *sdb,/*stat database*/
									  char *name,	/* stat variable name */
									  char *desc,/*stat variable description*/
									  /* dist initial value */
									  unsigned int init_val,
									  int pf,/* print format, use PF_* defs */
									  /* optional variable output format */
									  const char *format);


  /* create a new stats database */
  struct stat_sdb_t *stat_new(void);

  /* print the value of all stat variables in stat database SDB */
  void stat_print_stats(struct stat_sdb_t *sdb,/* stat database */
						FILE *fd);		/* output stream */
  
  /* register an unsigned integer statistical variable */
  struct stat_stat_t * stat_reg_uint(struct stat_sdb_t *sdb,/* stat database */
									 char *name,	/* stat variable name */
									 char *desc, /*stat variable description */
									 unsigned int *var,	/* stat variable */
									 /* stat variable initial value */
									 unsigned int init_val,
									 /* optional variable output format */
									 const char *format);

 private:
  void add_stat(struct stat_sdb_t *sdb,	/* stat database */
				struct stat_stat_t *stat);	/* stat variable */
  
#if 0
  void print_dist(struct stat_stat_t *stat,	/* stat variable */
				  FILE *fd);			/* output stream */
  void print_sdist(struct stat_stat_t *stat,	/* stat variable */
				   FILE *fd);			/* output stream */  
#endif
  
  /* delete a stats database */
  void stat_delete(struct stat_sdb_t *sdb);	/* stats database */
  
#ifdef HOST_HAS_QWORD
  /* register a qword integer statistical variable */
  struct stat_stat_t * stat_reg_qword(struct stat_sdb_t *sdb,/*stat database */
									  char *name,	/* stat variable name */
									  char *desc, /*stat variable description*/
									  qword_t *var,		/* stat variable */
									  /* stat variable initial value */
									  qword_t init_val,	
									  /* optional variable output format */
									  const char *format);	
#endif /* HOST_HAS_QWORD */
  
  /* register a float statistical variable */
  struct stat_stat_t * stat_reg_float(struct stat_sdb_t *sdb, /*stat database*/
									  char *name,	/* stat variable name */
									  char *desc, /*stat variable description*/
									  float *var,		/* stat variable */
									  /* stat variable initial value */
									  float init_val,		
									  /* optional variable output format */
									  const char *format);	
  
  /* register a double statistical variable */
  struct stat_stat_t * stat_reg_double(struct stat_sdb_t *sdb,/*stat database*/
									   char *name,	/* stat variable name */
									   char *desc,/*stat variable description*/
									   double *var,		/* stat variable */
									   /* stat variable initial value */
									   double init_val,	
									   /* optional variable output format */
									   const char *format); 
  
  /* add NSAMPLES to array or sparse array distribution STAT */
  /* add a single sample to array or sparse array distribution STAT */
  void stat_add_sample(struct stat_stat_t *stat,/* stat variable */
					   md_addr_t index);	/* index of sample */

  /* print the value of stat variable STAT */
  void stat_print_stat(struct stat_sdb_t *sdb,	/* stat database */
					   struct stat_stat_t *stat,/* stat variable */
					   FILE *fd);		/* output stream */



  // eval

 public:
  enum eval_err_t eval_error;
  char *eval_err_str[ERR_NUM];
  char *eval_type_str[et_NUM];
  
 private:
  struct eval_value_t err_value;
  /* *first* token character -> enum eval_token_t map */
  enum eval_token_t tok_map[256];
  int tok_map_initialized;
  void init_tok_map(void);
  enum eval_token_t		/* token parsed */
	get_next_token(struct eval_state_t *es);	/* expression evaluator */  
  enum eval_token_t		 /* next token in expression */
	peek_next_token(struct eval_state_t *es); /* expression evalutor */
  enum eval_type_t			/* type of expression result */
	result_type(enum eval_type_t t1,	/* left operand type */
				enum eval_type_t t2);	/* right operand type */
  struct eval_value_t
	f_add(struct eval_value_t val1, struct eval_value_t val2);
  struct eval_value_t
	f_sub(struct eval_value_t val1, struct eval_value_t val2);
  struct eval_value_t
	f_mult(struct eval_value_t val1, struct eval_value_t val2);
  struct eval_value_t
	f_div(struct eval_value_t val1, struct eval_value_t val2);
  struct eval_value_t f_neg(struct eval_value_t val1);
  int f_eq_zero(struct eval_value_t val1);
  struct eval_value_t constant(struct eval_state_t *es);
  struct eval_value_t factor(struct eval_state_t *es);
  struct eval_value_t term(struct eval_state_t *es);
  struct eval_value_t expr(struct eval_state_t *es);
  void eval_print(FILE *stream,		/* output stream */
				  struct eval_value_t val);	/* expression value to print */
  
  
 public:
  //struct eval_state_t * eval_new(eval_ident_t f_eval_ident, void *user_ptr);
  struct eval_state_t * eval_new(void *user_ptr);
  struct eval_value_t			/* value of the expression */
	eval_expr(struct eval_state_t *es,	/* expression evaluator */
			  char *p,		 /* ptr to expression string */
			  char **endp);	   /* returns ptr to 1st unused char */
  void eval_delete(struct eval_state_t *es);
  /* eval_value_t (any numeric type) -> double */
  double eval_as_double(struct eval_value_t val);
  
  /* eval_value_t (any numeric type) -> float */
  float eval_as_float(struct eval_value_t val);

#ifdef HOST_HAS_QWORD
  /* eval_value_t (any numeric type) -> qword_t */
  qword_t eval_as_qword(struct eval_value_t val);

  /* eval_value_t (any numeric type) -> sqword_t */
  sqword_t eval_as_sqword(struct eval_value_t val);
#endif /* HOST_HAS_QWORD */

  /* eval_value_t (any numeric type) -> md_addr_t */
  md_addr_t eval_as_addr(struct eval_value_t val);

  /* eval_value_t (any numeric type) -> unsigned int */
  unsigned int eval_as_uint(struct eval_value_t val);

  /* eval_value_t (any numeric type) -> int */
  int eval_as_int(struct eval_value_t val);


};


	       
#endif /* STAT_H */
