/* loader.c - program loader routines */

/* SimpleScalar(TM) Tool Suite
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 * All Rights Reserved. 
 * 
 * THIS IS A LEGAL DOCUMENT, BY USING SIMPLESCALAR,
 * YOU ARE AGREEING TO THESE TERMS AND CONDITIONS.
 * 
 * No portion of this work may be used by any commercial entity, or for any
 * commercial purpose, without the prior, written permission of SimpleScalar,
 * LLC (info@simplescalar.com). Nonprofit and noncommercial use is permitted
 * as described below.
 * 
 * 1. SimpleScalar is provided AS IS, with no warranty of any kind, express
 * or implied. The user of the program accepts full responsibility for the
 * application of the program and the use of any results.
 * 
 * 2. Nonprofit and noncommercial use is encouraged. SimpleScalar may be
 * downloaded, compiled, executed, copied, and modified solely for nonprofit,
 * educational, noncommercial research, and noncommercial scholarship
 * purposes provided that this notice in its entirety accompanies all copies.
 * Copies of the modified software can be delivered to persons who use it
 * solely for nonprofit, educational, noncommercial research, and
 * noncommercial scholarship purposes provided that this notice in its
 * entirety accompanies all copies.
 * 
 * 3. ALL COMMERCIAL USE, AND ALL USE BY FOR PROFIT ENTITIES, IS EXPRESSLY
 * PROHIBITED WITHOUT A LICENSE FROM SIMPLESCALAR, LLC (info@simplescalar.com).
 * 
 * 4. No nonprofit user may place any restrictions on the use of this software,
 * including as modified by the user, by any other authorized user.
 * 
 * 5. Noncommercial and nonprofit users may distribute copies of SimpleScalar
 * in compiled or executable form as set forth in Section 2, provided that
 * either: (A) it is accompanied by the corresponding machine-readable source
 * code, or (B) it is accompanied by a written offer, with no time limit, to
 * give anyone a machine-readable copy of the corresponding source code in
 * return for reimbursement of the cost of distribution. This written offer
 * must permit verbatim duplication by anyone, or (C) it is distributed by
 * someone who received only the executable form, and is accompanied by a
 * copy of the written offer of source code.
 * 
 * 6. SimpleScalar was developed by Todd M. Austin, Ph.D. The tool suite is
 * currently maintained by SimpleScalar LLC (info@simplescalar.com). US Mail:
 * 2395 Timbercrest Court, Ann Arbor, MI 48105.
 * 
 * Copyright (C) 1994-2003 by Todd M. Austin, Ph.D. and SimpleScalar, LLC.
 */

#include "loader_s.h"
#include "only_syscall.h"

loader_simple::loader_simple()
{
  eio_trans_icnt=-1;
  ld_text_base = 0;
  ld_text_size = 0;
  ld_data_base = 0;
  ld_data_size = 0;
  ld_brk_point = 0;
  ld_stack_base = MD_STACK_BASE;
  ld_stack_size = 0;
  ld_stack_min = (md_addr_t)-1;
  ld_prog_fname = NULL;
  ld_prog_entry = 0;
  ld_environ_base = 0;
  mem_simple = NULL;
  st_simple = NULL;
  sim_eio_fd = NULL;

  ss_flag_table[0].ss_flag = SS_O_RDONLY;
  ss_flag_table[0].local_flag = O_RDONLY;
  ss_flag_table[1].ss_flag = SS_O_WRONLY;
  ss_flag_table[1].local_flag = O_WRONLY;
  ss_flag_table[2].ss_flag = SS_O_RDWR;
  ss_flag_table[2].local_flag = O_RDWR;
  ss_flag_table[3].ss_flag = SS_O_APPEND;
  ss_flag_table[3].local_flag = O_APPEND;
  ss_flag_table[4].ss_flag = SS_O_CREAT;
  ss_flag_table[4].local_flag = O_CREAT;
  ss_flag_table[5].ss_flag = SS_O_TRUNC;
  ss_flag_table[5].local_flag = O_TRUNC;
  ss_flag_table[6].ss_flag = SS_O_EXCL;
  ss_flag_table[6].local_flag = O_EXCL;
  ss_flag_table[7].ss_flag = SS_O_NONBLOCK;
  ss_flag_table[7].local_flag = O_NONBLOCK;
  ss_flag_table[8].ss_flag = SS_O_NOCTTY;
  ss_flag_table[8].local_flag = O_NOCTTY;
  ss_flag_table[9].ss_flag = SS_O_SYNC;
  ss_flag_table[9].local_flag = O_SYNC;
}

#ifndef HOST_ONLY
/* probe target (simulated program) byte endian format, only
   valid after program has been loaded */
enum endian_t
loader_simple::endian_target_byte_order(void)
{
  return ld_target_big_endian ? endian_big : endian_little;
}

/* probe target (simulated program) double word endian format,
   only valid after program has been loaded */
enum endian_t
loader_simple::endian_target_word_order(void)
{
  /* same as byte sex for SimpleScalar target */
  return endian_target_byte_order();
}
#endif /* !HOST_ONLY */


/* register simulator-specific statistics */
void
loader_simple::ld_reg_stats(struct stat_sdb_t *sdb)	/* stats data base */
{
  st_simple->stat_reg_addr(sdb, "ld_text_base",
						   "program text (code) segment base",
						   &ld_text_base, ld_text_base, "  0x%08p");
  st_simple->stat_reg_uint(sdb, "ld_text_size",
						   "program text (code) size in bytes",
						   &ld_text_size, ld_text_size, NULL);
  st_simple->stat_reg_addr(sdb, "ld_data_base",
						   "program initialized data segment base",
						   &ld_data_base, ld_data_base, "  0x%08p");
  st_simple->stat_reg_uint(sdb, "ld_data_size",
						   "program init'ed `.data' and uninit'ed `.bss' size in bytes",
						   &ld_data_size, ld_data_size, NULL);
  st_simple->stat_reg_addr(sdb, "ld_stack_base",
						   "program stack segment base (highest address in stack)",
						   &ld_stack_base, ld_stack_base, "  0x%08p");
  st_simple->stat_reg_uint(sdb, "ld_stack_size",
						   "program initial stack size",
						   &ld_stack_size, ld_stack_size, NULL);
#if 0 /* FIXME: broken... */
  stat_reg_addr(sdb, "ld_stack_min",
				"program stack segment lowest address",
				&ld_stack_min, ld_stack_min, "  0x%08p");
#endif
  st_simple->stat_reg_addr(sdb, "ld_prog_entry",
						   "program entry point (initial PC)",
						   &ld_prog_entry, ld_prog_entry, "  0x%08p");
  st_simple->stat_reg_addr(sdb, "ld_environ_base",
						   "program environment base address address",
						   &ld_environ_base, ld_environ_base, "  0x%08p");
  st_simple->stat_reg_int(sdb, "ld_target_big_endian",
						  "target executable endian-ness, non-zero if big endian",
						  &ld_target_big_endian, ld_target_big_endian, NULL);
}


/* load program text and initialized data into simulated virtual memory
   space and initialize program segment range variables */
void
loader_simple::ld_load_prog(char *fname,		/* program to load */
							int argc, 
							char **argv, /* simulated program cmd line args */
							char **envp,	/* simulated program environment */
							struct regs_t *regs, /* registers to 
													initialize for load */
							struct mem_t *mem, /* memory space 
												  to load prog into */
							int zero_bss_segs)	/* zero uninit data segment? */
							//							mem_type *mem_isis)
{
  int i;
  word_t temp;
  md_addr_t sp, data_break = 0, null_ptr = 0, argv_addr, envp_addr;

  if (eio_valid(fname))
    {
      if (argc != 1)
		{
		  fprintf(stderr, "error: EIO file has arguments\n");
		  exit(1);
		}

      fprintf(stderr, "sim: loading EIO file: %s\n", fname);

      sim_eio_fname = mystrdup(fname);

      /* open the EIO file stream */
      sim_eio_fd = eio_open(fname);

      /* load initial state checkpoint */
      if (eio_read_chkpt(regs, mem, sim_eio_fd) != -1)
		fatal("bad initial checkpoint in EIO file");

      /* load checkpoint? */
      if (sim_chkpt_fname != NULL)
		{
		  counter_t restore_icnt;

		  FILE *chkpt_fd;

		  fprintf(stderr, "sim: loading checkpoint file: %s\n",
				  sim_chkpt_fname);

		  if (!eio_valid(sim_chkpt_fname))
			fatal("file `%s' does not appear to be a checkpoint file",
				  sim_chkpt_fname);

		  /* open the checkpoint file */
		  chkpt_fd = eio_open(sim_chkpt_fname);

		  /* load the state image */
		  restore_icnt = eio_read_chkpt(regs, mem, chkpt_fd);

		  /* fast forward the baseline EIO trace to checkpoint location */
		  myfprintf(stderr, "sim: fast forwarding to instruction %n\n",
					restore_icnt);
		  eio_fast_forward(sim_eio_fd, restore_icnt);
		}

      /* computed state... */
      ld_environ_base = regs->regs_R[MD_REG_SP];
      ld_prog_entry = regs->regs_PC;

      /* fini... */
      return;
    }
#ifdef MD_CROSS_ENDIAN
  else
    {
      fatal("SimpleScalar/PISA only supports binary execution on\n"
			"       same-endian hosts, use EIO files on cross-endian hosts");
    }
#endif /* MD_CROSS_ENDIAN */


  if (sim_chkpt_fname != NULL)
    fatal("checkpoints only supported while EIO tracing");

#ifdef BFD_LOADER

  {
    bfd *abfd;
    asection *sect;

    /* set up a local stack pointer, this is where the argv and envp
       data is written into program memory */
    ld_stack_base = MD_STACK_BASE;
    sp = ROUND_DOWN(MD_STACK_BASE - MD_MAX_ENVIRON, sizeof(dfloat_t));
    ld_stack_size = ld_stack_base - sp;

    /* initial stack pointer value */
    ld_environ_base = sp;

    /* load the program into memory, try both endians */
    if (!(abfd = bfd_openr(argv[0], "ss-coff-big")))
      if (!(abfd = bfd_openr(argv[0], "ss-coff-little")))
		fatal("cannot open executable `%s'", argv[0]);

    /* this call is mainly for its side effect of reading in the sections.
       we follow the traditional behavior of `strings' in that we don't
       complain if we don't recognize a file to be an object file.  */
    if (!bfd_check_format(abfd, bfd_object))
      {
		bfd_close(abfd);
		fatal("cannot open executable `%s'", argv[0]);
      }

    /* record profile file name */
    ld_prog_fname = argv[0];

    /* record endian of target */
    ld_target_big_endian = abfd->xvec->byteorder_big_p;

    debug("processing %d sections in `%s'...",
		  bfd_count_sections(abfd), argv[0]);

    /* read all sections in file */
    for (sect=abfd->sections; sect; sect=sect->next)
      {
		char *p;

		debug("processing section `%s', %d bytes @ 0x%08x...",
			  bfd_section_name(abfd, sect), bfd_section_size(abfd, sect),
			  bfd_section_vma(abfd, sect));

		/* read the section data, if allocated and loadable and non-NULL */
		if ((bfd_get_section_flags(abfd, sect) & SEC_ALLOC)
			&& (bfd_get_section_flags(abfd, sect) & SEC_LOAD)
			&& bfd_section_vma(abfd, sect)
			&& bfd_section_size(abfd, sect))
		  {
			/* allocate a section buffer */
			p = calloc(bfd_section_size(abfd, sect), sizeof(char));
			if (!p)
			  fatal("cannot allocate %d bytes for section `%s'",
					bfd_section_size(abfd, sect),
					bfd_section_name(abfd, sect));

			if (!bfd_get_section_contents(abfd, sect, p, (file_ptr)0,
										  bfd_section_size(abfd, sect)))
			  fatal("could not read entire `%s' section from executable",
					bfd_section_name(abfd, sect));

			/* copy program section it into simulator target memory */
			mem_simple->mem_bcopy(mem_access, mem, Write, bfd_section_vma(abfd, sect),
								  p, bfd_section_size(abfd, sect));

			/* release the section buffer */
			free(p);
		  }
		/* zero out the section if it is loadable but not allocated in exec */
		else if (zero_bss_segs
				 && (bfd_get_section_flags(abfd, sect) & SEC_LOAD)
				 && bfd_section_vma(abfd, sect)
				 && bfd_section_size(abfd, sect))
		  {
			/* zero out the section region */
			mem_simple->mem_bzero(mem_access, mem,
								  bfd_section_vma(abfd, sect),
								  bfd_section_size(abfd, sect));
		  }
		else
		  {
			/* else do nothing with this section, it's probably debug data */
			debug("ignoring section `%s' during load...",
				  bfd_section_name(abfd, sect));
		  }

		/* expected text section */
		if (!strcmp(bfd_section_name(abfd, sect), ".text"))
		  {
			/* .text section processing */
			ld_text_size =
			  ((bfd_section_vma(abfd, sect) + bfd_section_size(abfd, sect))
			   - MD_TEXT_BASE)
			  + /* for speculative fetches/decodes */TEXT_TAIL_PADDING;

			/* create tail padding and copy into simulator target memory */
			mem_simple->mem_bzero(mem_access, mem,
								  bfd_section_vma(abfd, sect)
								  + bfd_section_size(abfd, sect),
								  TEXT_TAIL_PADDING);
		  }
		/* expected data sections */
		else if (!strcmp(bfd_section_name(abfd, sect), ".rdata")
				 || !strcmp(bfd_section_name(abfd, sect), ".data")
				 || !strcmp(bfd_section_name(abfd, sect), ".sdata")
				 || !strcmp(bfd_section_name(abfd, sect), ".bss")
				 || !strcmp(bfd_section_name(abfd, sect), ".sbss"))
		  {
			/* data section processing */
			if (bfd_section_vma(abfd, sect) + bfd_section_size(abfd, sect) >
				data_break)
			  data_break = (bfd_section_vma(abfd, sect) +
							bfd_section_size(abfd, sect));
		  }
		else
		  {
			/* what is this section??? */
			fatal("encountered unknown section `%s', %d bytes @ 0x%08x",
				  bfd_section_name(abfd, sect), bfd_section_size(abfd, sect),
				  bfd_section_vma(abfd, sect));
		  }
      }

    /* compute data segment size from data break point */
    ld_text_base = MD_TEXT_BASE;
    ld_data_base = MD_DATA_BASE;
    ld_prog_entry = bfd_get_start_address(abfd);
    ld_data_size = data_break - ld_data_base;

    /* done with the executable, close it */
    if (!bfd_close(abfd))
      fatal("could not close executable `%s'", argv[0]);
  }

#else /* !BFD_LOADER, i.e., standalone loader */

  {
    FILE *fobj;
    long floc;
    struct ecoff_filehdr fhdr;
    struct ecoff_aouthdr ahdr;
    struct ecoff_scnhdr shdr;

    /* set up a local stack pointer, this is where the argv and envp
       data is written into program memory */
    ld_stack_base = MD_STACK_BASE;
    sp = ROUND_DOWN(MD_STACK_BASE - MD_MAX_ENVIRON, sizeof(dfloat_t));
    ld_stack_size = ld_stack_base - sp;

    /* initial stack pointer value */
    ld_environ_base = sp;

    /* record profile file name */
    ld_prog_fname = argv[0];

    /* load the program into memory, try both endians */
#if defined(__CYGWIN32__) || defined(_MSC_VER)
    fobj = fopen(argv[0], "rb");
#else
    fobj = fopen(argv[0], "r");
#endif
    if (!fobj)
      fatal("cannot open executable `%s'", argv[0]);

    if (fread(&fhdr, sizeof(struct ecoff_filehdr), 1, fobj) < 1)
      fatal("cannot read header from executable `%s'", argv[0]);

    /* record endian of target */
    if (fhdr.f_magic == ECOFF_EB_MAGIC)
      ld_target_big_endian = TRUE;
    else if (fhdr.f_magic == ECOFF_EL_MAGIC)
      ld_target_big_endian = FALSE;
    else if (fhdr.f_magic == ECOFF_EB_OTHER || fhdr.f_magic == ECOFF_EL_OTHER)
      fatal("PISA binary `%s' has wrong endian format", argv[0]);
    else if (fhdr.f_magic == ECOFF_ALPHAMAGIC)
      fatal("PISA simulator cannot run Alpha binary `%s'", argv[0]);
    else
      fatal("bad magic number in executable `%s' (not an executable?)",
			argv[0]);

    if (fread(&ahdr, sizeof(struct ecoff_aouthdr), 1, fobj) < 1)
      fatal("cannot read AOUT header from executable `%s'", argv[0]);

    data_break = MD_DATA_BASE + ahdr.dsize + ahdr.bsize;

#if 0
    Data_start = ahdr.data_start;
    Data_size = ahdr.dsize;
    Bss_size = ahdr.bsize;
    Bss_start = ahdr.bss_start;
    Gp_value = ahdr.gp_value;
    Text_entry = ahdr.entry;
#endif

    /* seek to the beginning of the first section header, the file header comes
       first, followed by the optional header (this is the aouthdr), the size
       of the aouthdr is given in Fdhr.f_opthdr */
    fseek(fobj, sizeof(struct ecoff_filehdr) + fhdr.f_opthdr, 0);

    debug("processing %d sections in `%s'...", fhdr.f_nscns, argv[0]);

    /* loop through the section headers */
    floc = ftell(fobj);
    for (i = 0; i < fhdr.f_nscns; i++)
      {
		char *p;

		if (fseek(fobj, floc, 0) == -1)
		  fatal("could not reset location in executable");
		if (fread(&shdr, sizeof(struct ecoff_scnhdr), 1, fobj) < 1)
		  fatal("could not read section %d from executable", i);
		floc = ftell(fobj);

		switch (shdr.s_flags)
		  {
		  case ECOFF_STYP_TEXT:
			ld_text_size = ((shdr.s_vaddr + shdr.s_size) - MD_TEXT_BASE) 
			  + TEXT_TAIL_PADDING;

			p = (char *)calloc(shdr.s_size, sizeof(char));
			if (!p)
			  fatal("out of virtual memory");

			if (fseek(fobj, shdr.s_scnptr, 0) == -1)
			  fatal("could not read `.text' from executable", i);
			if (fread(p, shdr.s_size, 1, fobj) < 1)
			  fatal("could not read text section from executable");

			/* copy program section into simulator target memory */
			mem_simple->mem_bcopy(mem, Write, shdr.s_vaddr, p, shdr.s_size);

			/* create tail padding and copy into simulator target memory */
			mem_simple->mem_bzero(mem,
								  shdr.s_vaddr + shdr.s_size, TEXT_TAIL_PADDING);
  
			/* release the section buffer */
			free(p);

#if 0
			Text_seek = shdr.s_scnptr;
			Text_start = shdr.s_vaddr;
			Text_size = shdr.s_size / 4;
			/* there is a null routine after the supposed end of text */
			Text_size += 10;
			Text_end = Text_start + Text_size * 4;
			/* create_text_reloc(shdr.s_relptr, shdr.s_nreloc); */
#endif
			break;

		  case ECOFF_STYP_RDATA:
			/* The .rdata section is sometimes placed before the text
			 * section instead of being contiguous with the .data section.
			 */
#if 0
			Rdata_start = shdr.s_vaddr;
			Rdata_size = shdr.s_size;
			Rdata_seek = shdr.s_scnptr;
#endif
			/* fall through */
		  case ECOFF_STYP_DATA:
#if 0
			Data_seek = shdr.s_scnptr;
#endif
			/* fall through */
		  case ECOFF_STYP_SDATA:
#if 0
			Sdata_seek = shdr.s_scnptr;
#endif

			p = (char *)calloc(shdr.s_size, sizeof(char));
			if (!p)
			  fatal("out of virtual memory");

			if (fseek(fobj, shdr.s_scnptr, 0) == -1)
			  fatal("could not read `.text' from executable", i);
			if (fread(p, shdr.s_size, 1, fobj) < 1)
			  fatal("could not read text section from executable");

			/* copy program section it into simulator target memory */
			mem_simple->mem_bcopy(mem, Write, shdr.s_vaddr, p, shdr.s_size);

			/* release the section buffer */
			free(p);

			break;

		  case ECOFF_STYP_BSS:
			break;

		  case ECOFF_STYP_SBSS:
			break;
		  }
      }

    /* compute data segment size from data break point */
    ld_text_base = MD_TEXT_BASE;
    ld_data_base = MD_DATA_BASE;
    ld_prog_entry = ahdr.entry;
    ld_data_size = data_break - ld_data_base;

    /* done with the executable, close it */
    if (fclose(fobj))
      fatal("could not close executable `%s'", argv[0]);
  }
#endif /* BFD_LOADER */

  /* perform sanity checks on segment ranges */
  if (!ld_text_base || !ld_text_size)
    fatal("executable is missing a `.text' section");
  if (!ld_data_base || !ld_data_size)
    fatal("executable is missing a `.data' section");
  if (!ld_prog_entry)
    fatal("program entry point not specified");

  /* determine byte/words swapping required to execute on this host */
  int sim_swap_bytes = (endian.endian_host_byte_order() 
						!= endian_target_byte_order());
  if (sim_swap_bytes)
    {
#if 0 /* FIXME: disabled until further notice... */
      /* cross-endian is never reliable, why this is so is beyond the scope
		 of this comment, e-mail me for details... */
      fprintf(stderr, "sim: *WARNING*: swapping bytes to match host...\n");
      fprintf(stderr, "sim: *WARNING*: swapping may break your program!\n");
#else
      fatal("binary endian does not match host endian");
#endif
    }
  int sim_swap_words = (endian.endian_host_word_order() 
						!= endian_target_word_order());
  if (sim_swap_words)
    {
#if 0 /* FIXME: disabled until further notice... */
      /* cross-endian is never reliable, why this is so is beyond the scope
		 of this comment, e-mail me for details... */
      fprintf(stderr, "sim: *WARNING*: swapping words to match host...\n");
      fprintf(stderr, "sim: *WARNING*: swapping may break your program!\n");
#else
      fatal("binary endian does not match host endian");
#endif
    }

  /* write [argc] to stack */
  temp = MD_SWAPW(argc);
  mem_simple->mem_access(mem, Write, sp, &temp, sizeof(word_t), 1);
  sp += sizeof(word_t);

  /* skip past argv array and NULL */
  argv_addr = sp;
  sp = sp + (argc + 1) * sizeof(md_addr_t);

  /* save space for envp array and NULL */
  envp_addr = sp;
  for (i=0; envp[i]; i++)
    sp += sizeof(md_addr_t);
  sp += sizeof(md_addr_t);

  /* fill in the argv pointer array and data */
  for (i=0; i<argc; i++)
    {
      /* write the argv pointer array entry */
      temp = MD_SWAPW(sp);
      mem_simple->mem_access(mem, Write, argv_addr + i*sizeof(md_addr_t),
							 &temp, sizeof(md_addr_t), 1);
	  
      /* and the data */
      mem_simple->mem_strcpy(mem, Write, sp, argv[i]);

      sp += strlen(argv[i]) + 1;
    }
  /* terminate argv array with a NULL */
  mem_simple->mem_access(mem, Write, argv_addr + i*sizeof(md_addr_t),
						 &null_ptr, sizeof(md_addr_t), 1);


  /* write envp pointer array and data to stack */
  for (i = 0; envp[i]; i++)
    {
      /* write the envp pointer array entry */
      temp = MD_SWAPW(sp);
      mem_simple->mem_access(mem, Write, envp_addr + i*sizeof(md_addr_t),
							 &temp, sizeof(md_addr_t), 1);

      /* and the data */
      mem_simple->mem_strcpy(mem, Write, sp, envp[i]);
      sp += strlen(envp[i]) + 1;
    }
  /* terminate the envp array with a NULL */
  mem_simple->mem_access(mem, Write, envp_addr + i*sizeof(md_addr_t),
						 &null_ptr, sizeof(md_addr_t), 1);

  /* did we tromp off the stop of the stack? */
  if (sp > ld_stack_base)
    {
      /* we did, indicate to the user that MD_MAX_ENVIRON must be increased,
		 alternatively, you can use a smaller environment, or fewer
		 command line arguments */
      fatal("environment overflow, increase MD_MAX_ENVIRON in ss.h");
    }

  /* initialize the bottom of heap to top of data segment */
  ld_brk_point = ROUND_UP(ld_data_base + ld_data_size, MD_PAGE_SIZE);

  /* set initial minimum stack pointer value to initial stack value */
  ld_stack_min = regs->regs_R[MD_REG_SP];

  /* set up initial register state */
  regs->regs_R[MD_REG_SP] = ld_environ_base;
  regs->regs_PC = ld_prog_entry;

  debug("ld_text_base: 0x%08x  ld_text_size: 0x%08x",
		ld_text_base, ld_text_size);
  debug("ld_data_base: 0x%08x  ld_data_size: 0x%08x",
		ld_data_base, ld_data_size);
  debug("ld_stack_base: 0x%08x  ld_stack_size: 0x%08x",
		ld_stack_base, ld_stack_size);
  debug("ld_prog_entry: 0x%08x", ld_prog_entry);

  /* finally, predecode the text segment... */
  {
    md_addr_t addr;
    md_inst_t inst;
    enum md_fault_type fault;

    if (OP_MAX > 255)
      fatal("cannot perform fast decoding, too many opcodes");

    debug("sim: decoding text segment...");
    for (addr=ld_text_base;
		 addr < (ld_text_base+ld_text_size);
		 addr += sizeof(md_inst_t))
      {
		fault = mem_simple->mem_access(mem, Read, addr, &inst, sizeof(inst), 1);
		if (fault != md_fault_none)
		  fatal("could not read instruction memory");
		inst.a = (inst.a & ~0xff) | (word_t)MD_OP_ENUM(MD_OPFIELD(inst));
		fault = mem_simple->mem_access(mem, Write, addr,
									   &inst, sizeof(inst), 1);

		if (fault != md_fault_none)
		  fatal("could not write instruction memory");
      }
  }
}


FILE *
loader_simple::eio_create(char *fname)
{
  FILE *fd;
  struct exo_term_t *exo;
  int target_big_endian;

  target_big_endian = (endian.endian_host_byte_order() == endian_big);

  fd = gzopen(fname, "w");
  if (!fd)
    fatal("unable to create EIO file `%s'", fname);

  /* emit EIO file header */
  fprintf(fd, "%s\n", EIO_FILE_HEADER);
  fprintf(fd, "/* file_format: %d, file_version: %d, big_endian: %d */\n", 
		  MD_EIO_FILE_FORMAT, EIO_FILE_VERSION, ld_target_big_endian);
  exo = lib.exo_new(ec_list,
					lib.exo_new(ec_integer, (exo_integer_t)MD_EIO_FILE_FORMAT),
					lib.exo_new(ec_integer, (exo_integer_t)EIO_FILE_VERSION),
					lib.exo_new(ec_integer, (exo_integer_t)target_big_endian),
					NULL);
  lib.exo_print(exo, fd);
  lib.exo_delete(exo);
  fprintf(fd, "\n\n");

  return fd;
}

FILE *
loader_simple::eio_open(char *fname)
{
  FILE *fd;
  struct exo_term_t *exo;
  int file_format, file_version, big_endian, target_big_endian;

  target_big_endian = (endian.endian_host_byte_order() == endian_big);

  fd = gzopen(fname, "r");
  if (!fd)
    fatal("unable to open EIO file `%s'", fname);

  /* read and check EIO file header */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list
      || !exo->as_list.head
      || exo->as_list.head->ec != ec_integer
      || !exo->as_list.head->next
      || exo->as_list.head->next->ec != ec_integer
      || !exo->as_list.head->next->next
      || exo->as_list.head->next->next->ec != ec_integer
      || exo->as_list.head->next->next->next != NULL)
    fatal("could not read EIO file header");

  file_format = exo->as_list.head->as_integer.val;
  file_version = exo->as_list.head->next->as_integer.val;
  big_endian = exo->as_list.head->next->next->as_integer.val;
  lib.exo_delete(exo);

  if (file_format != MD_EIO_FILE_FORMAT)
    fatal("EIO file `%s' has incompatible format", fname);

  if (file_version != EIO_FILE_VERSION)
    fatal("EIO file `%s' has incompatible version", fname);

  return fd;
}

/* returns non-zero if file FNAME has a valid EIO header */
int 
loader_simple::eio_valid(char *fname)
{
  FILE *fd;
  char buf[512];

  /* open possible EIO file */
  fd = gzopen(fname, "r");
  if (!fd)
    return FALSE;

  /* read and check EIO file header */
  fgets(buf, 512, fd);

  /* check the header */
  if (strcmp(buf, EIO_FILE_HEADER))
    return FALSE;

  /* all done, close up file */
  gzclose(fd);

  /* else, has a valid header, go with it... */
  return TRUE;
}

void 
loader_simple::eio_close(FILE *fd)
{
  gzclose(fd);
}

/* check point current architected state to stream FD, returns
   EIO transaction count (an EIO file pointer) */
counter_t 
loader_simple::eio_write_chkpt(struct regs_t *regs,	/* regs to dump */
							   struct mem_t *mem,/* memory to dump */
							   FILE *fd)	/* stream to write to */
{
  int i;
  struct exo_term_t *exo;
  struct mem_pte_t *pte;

  myfprintf(fd, "/* ** start checkpoint @ %n... */\n\n", eio_trans_icnt);

  myfprintf(fd, "/* EIO file pointer: %n... */\n", eio_trans_icnt);
  exo = lib.exo_new(ec_integer, (exo_integer_t)eio_trans_icnt);
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  /* dump misc regs: icnt, PC, NPC, etc... */
  fprintf(fd, "/* misc regs icnt, PC, NPC, etc... */\n");
  counter_t a = 0;
  exo = MD_MISC_REGS_TO_EXO(regs, a);
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  /* dump integer registers */
  fprintf(fd, "/* integer regs */\n");
  exo = lib.exo_new(ec_list, NULL);
  for (i=0; i < MD_NUM_IREGS; i++)
    exo->as_list.head = lib.exo_chain(exo->as_list.head, MD_IREG_TO_EXO(regs, i));
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  /* dump FP registers */
  fprintf(fd, "/* FP regs (integer format) */\n");
  exo = lib.exo_new(ec_list, NULL);
  for (i=0; i < MD_NUM_FREGS; i++)
    exo->as_list.head = lib.exo_chain(exo->as_list.head, MD_FREG_TO_EXO(regs, i));
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  fprintf(fd, "/* writing `%d' memory pages... */\n", (int)mem->page_count);
  exo = lib.exo_new(ec_list,
				lib.exo_new(ec_integer, (exo_integer_t)mem->page_count),
				lib.exo_new(ec_address, (exo_integer_t)ld_brk_point),
				lib.exo_new(ec_address, (exo_integer_t)ld_stack_min),
				NULL);
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  fprintf(fd, "/* text segment specifiers (base & size) */\n");
  exo = lib.exo_new(ec_list,
				lib.exo_new(ec_address, (exo_integer_t)ld_text_base),
				lib.exo_new(ec_integer, (exo_integer_t)ld_text_size),
				NULL);
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  fprintf(fd, "/* data segment specifiers (base & size) */\n");
  exo = lib.exo_new(ec_list,
				lib.exo_new(ec_address, (exo_integer_t)ld_data_base),
				lib.exo_new(ec_integer, (exo_integer_t)ld_data_size),
				NULL);
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  fprintf(fd, "/* stack segment specifiers (base & size) */\n");
  exo = lib.exo_new(ec_list,
				lib.exo_new(ec_address, (exo_integer_t)ld_stack_base),
				lib.exo_new(ec_integer, (exo_integer_t)ld_stack_size),
				NULL);
  lib.exo_print(exo, fd);
  fprintf(fd, "\n\n");
  lib.exo_delete(exo);

  /* visit all active memory pages, and dump them to the checkpoint file */
  //MEM_FORALL(mem, i, pte)
  for (i = 0; i < MEM_PTAB_SIZE; i++){	 
	for (pte = mem->ptab[i]; pte != NULL; pte = pte->next) {
	  
	  exo_integer_t tmp_addr;
	  tmp_addr = ((pte->tag << (MD_LOG_PAGE_SIZE + MEM_LOG_PTAB_SIZE))
				  | ( i << MD_LOG_PAGE_SIZE));
	  
	  /* dump this page... */
	  exo = lib.exo_new(ec_list,
					lib.exo_new(ec_address, tmp_addr),
					lib.exo_new(ec_blob, MD_PAGE_SIZE, pte->page),
					NULL);
	  lib.exo_print(exo, fd);
	  fprintf(fd, "\n\n");
	  lib.exo_delete(exo);
	}
  }

  myfprintf(fd, "/* ** end checkpoint @ %n... */\n\n", eio_trans_icnt);

  return eio_trans_icnt;
}

/* read check point of architected state from stream FD, returns
   EIO transaction count (an EIO file pointer) */
counter_t 
loader_simple::eio_read_chkpt(struct regs_t *regs,	/* regs to dump */
							  struct mem_t *mem,	/* memory to dump */
							  FILE *fd)	/* stream to read */
{
  int i, page_count;
  counter_t trans_icnt;
  struct exo_term_t *exo, *elt;

  /* read the EIO file pointer */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_integer)
    fatal("could not read EIO file pointer");
  trans_icnt = exo->as_integer.val;
  lib.exo_delete(exo);

  /* read misc regs: icnt, PC, NPC, HI, LO, FCC */
  exo = lib.exo_read(fd);
  counter_t a = 0;
  MD_EXO_TO_MISC_REGS(exo, a, regs);
  lib.exo_delete(exo);

  /* read integer registers */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list)
    fatal("could not read EIO integer regs");
  elt = exo->as_list.head;
  for (i=0; i < MD_NUM_IREGS; i++) {
	if (!elt)
	  fatal("could not read EIO integer regs (too few)");

	if (elt->ec != ec_address)
	  fatal("could not read EIO integer regs (bad value)");

	MD_EXO_TO_IREG(elt, regs, i);
	elt = elt->next;
  }
  if (elt != NULL)
    fatal("could not read EIO integer regs (too many)");
  lib.exo_delete(exo);

  /* read FP registers */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list)
    fatal("could not read EIO FP regs");
  elt = exo->as_list.head;
  for (i=0; i < MD_NUM_FREGS; i++) {
	if (!elt)
	  fatal("could not read EIO FP regs (too few)");

	if (elt->ec != ec_address)
	  fatal("could not read EIO FP regs (bad value)");

	MD_EXO_TO_FREG(elt, regs, i);
	elt = elt->next;
  }
  if (elt != NULL)
    fatal("could not read EIO FP regs (too many)");
  lib.exo_delete(exo);

  /* read the number of page defs, and memory config */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list
      || !exo->as_list.head
      || exo->as_list.head->ec != ec_integer
      || !exo->as_list.head->next
      || exo->as_list.head->next->ec != ec_address
      || !exo->as_list.head->next->next
      || exo->as_list.head->next->next->ec != ec_address
      || exo->as_list.head->next->next->next != NULL)
    fatal("could not read EIO memory page count");
  page_count = exo->as_list.head->as_integer.val;
  ld_brk_point = (md_addr_t)exo->as_list.head->next->as_address.val;
  ld_stack_min = (md_addr_t)exo->as_list.head->next->next->as_address.val;
  lib.exo_delete(exo);

  /* read text segment specifiers */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list
      || !exo->as_list.head
      || exo->as_list.head->ec != ec_address
      || !exo->as_list.head->next
      || exo->as_list.head->next->ec != ec_integer
      || exo->as_list.head->next->next != NULL)
    fatal("count not read EIO text segment specifiers");
  ld_text_base = (md_addr_t)exo->as_list.head->as_address.val;
  ld_text_size = (unsigned int)exo->as_list.head->next->as_integer.val;
  lib.exo_delete(exo);

  /* read data segment specifiers */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list
      || !exo->as_list.head
      || exo->as_list.head->ec != ec_address
      || !exo->as_list.head->next
      || exo->as_list.head->next->ec != ec_integer
      || exo->as_list.head->next->next != NULL)
    fatal("count not read EIO data segment specifiers");
  ld_data_base = (md_addr_t)exo->as_list.head->as_address.val;
  ld_data_size = (unsigned int)exo->as_list.head->next->as_integer.val;
  lib.exo_delete(exo);

  /* read stack segment specifiers */
  exo = lib.exo_read(fd);
  if (!exo
      || exo->ec != ec_list
      || !exo->as_list.head
      || exo->as_list.head->ec != ec_address
      || !exo->as_list.head->next
      || exo->as_list.head->next->ec != ec_integer
      || exo->as_list.head->next->next != NULL)
    fatal("count not read EIO stack segment specifiers");
  ld_stack_base = (md_addr_t)exo->as_list.head->as_address.val;
  ld_stack_size = (unsigned int)exo->as_list.head->next->as_integer.val;
  lib.exo_delete(exo);

  for (i=0; i < page_count; i++) {
	int j;
	md_addr_t page_addr;
	struct exo_term_t *blob;

	/* read the page */
	exo = lib.exo_read(fd);
	if (!exo
		|| exo->ec != ec_list
		|| !exo->as_list.head
		|| exo->as_list.head->ec != ec_address
		|| !exo->as_list.head->next
		|| exo->as_list.head->next->ec != ec_blob
		|| exo->as_list.head->next->next != NULL)
	  fatal("could not read EIO memory page");
	page_addr = (md_addr_t)exo->as_list.head->as_address.val;
	blob = exo->as_list.head->next;

	/* write data to simulator memory */
	for (j=0; j < blob->as_blob.size; j++) {
	  byte_t val;

	  val = blob->as_blob.data[j];
	  /* unchecked access... */
	  mem_simple->mem_write_byte(mem, page_addr, val);
	  page_addr++;
	}
	lib.exo_delete(exo);
  }

  return trans_icnt;
}

/* syscall proxy handler from an EIO trace, architect registers
   and memory are assumed to be precise when this function is called,
   register and memory are updated with the results of the sustem call */
void 
loader_simple::eio_read_trace(FILE *eio_fd,	/* EIO stream file desc */
							  counter_t icnt,		/* instruction count */
							  struct regs_t *regs,/* registers to update */
							  struct mem_t *mem,		/* memory to update */
							  md_inst_t inst)			/* system call inst */
{
  int i;
  struct exo_term_t *exo, *exo_icnt, *exo_pc;
  struct exo_term_t *exo_inregs, *exo_inmem, *exo_outregs, *exo_outmem;
  struct exo_term_t *brkrec, *regrec, *memrec;

  /* exit() system calls get executed for real... */
  if (MD_EXIT_SYSCALL(regs)) {
	sys_syscall(regs, mem, inst, FALSE, 1, icnt);
	panic("returned from exit() system call");
  }

  /* else, read the external I/O (EIO) transaction */
  exo = lib.exo_read(eio_fd);

  /* one more transaction processed */
  eio_trans_icnt = icnt;

  /* pull apart the EIO transaction (EXO format) */
  if (!exo
      || exo->ec != ec_list
      || !(exo_icnt = exo->as_list.head)
      || exo_icnt->ec != ec_integer
      || !(exo_pc = exo_icnt->next)
      || exo_pc->ec != ec_address
      || !(exo_inregs = exo_pc->next)
      || exo_inregs->ec != ec_list
      || !(exo_inmem = exo_inregs->next)
      || exo_inmem->ec != ec_list
      || !(exo_outregs = exo_inmem->next)
      || exo_outregs->ec != ec_list
      || !(exo_outmem = exo_outregs->next)
      || exo_outmem->ec != ec_list
      || exo_outmem->next != NULL)
    fatal("cannot read EIO transaction");

  /*
   * check the system call inputs
   */

  /* check ICNT input */
  if (icnt != (counter_t)exo_icnt->as_integer.val)
    fatal("EIO trace inconsistency: ICNT mismatch");

  /* check PC input */
  if (regs->regs_PC != (md_addr_t)exo_pc->as_integer.val)
    fatal("EIO trace inconsistency: PC mismatch");

  /* check integer register inputs */
  for (i=MD_FIRST_IN_REG, regrec=exo_inregs->as_list.head;
       i <= MD_LAST_IN_REG; i++, regrec=regrec->next) {
	if (!regrec || regrec->ec != ec_address)
	  fatal("EIO trace inconsistency: missing input reg");

	if (MD_EXO_CMP_IREG(regrec, regs, i))
	  fatal("EIO trace inconsistency: R[%d] input mismatch", i);
#ifdef VERBOSE
	fprintf(stderr, "** R[%d] checks out...\n", i);
#endif /* VERBOSE */
  }
  if (regrec != NULL)
    fatal("EIO trace inconsistency: too many input regs");

  /* check memory inputs */
  for (memrec=exo_inmem->as_list.head; memrec != NULL; memrec=memrec->next) {
	md_addr_t loc;
	struct exo_term_t *addr, *blob;

	/* check the mem transaction format */
	if (!memrec
		|| memrec->ec != ec_list
		|| !(addr = memrec->as_list.head)
		|| addr->ec != ec_address
		|| !(blob = addr->next)
		|| blob->ec != ec_blob
		|| blob->next != NULL)
	  fatal("EIO trace inconsistency: bad memory transaction");

	for (loc=addr->as_integer.val, i=0; i < blob->as_blob.size; loc++,i++) {
	  unsigned char val;

	  /* was: val = MEM_READ_BYTE(loc); */
	  mem_simple->mem_access(mem, Read, loc, &val, sizeof(unsigned char), 1);

	  if (val != blob->as_blob.data[i])
		fatal("EIO trace inconsistency: addr 0x%08p input mismatch", loc);

#ifdef VERBOSE
	  myfprintf(stderr, "** 0x%08p checks out...\n", loc);
#endif /* VERBOSE */
	}

	/* simulate view'able I/O */
	if (MD_OUTPUT_SYSCALL(regs)) {
	  if (sim_progfd) {
		/* redirect program output to file */
		fwrite(blob->as_blob.data, 1, blob->as_blob.size, sim_progfd);
	  } else {
		/* write the output to stdout/stderr */
		write(MD_STREAM_FILENO(regs),
			  blob->as_blob.data, blob->as_blob.size);
	  }
	}
  }

  /*
   * write system call outputs
   */

  /* adjust breakpoint */
  brkrec = exo_outregs->as_list.head;
  if (!brkrec || brkrec->ec != ec_address)
    fatal("EIO trace inconsistency: missing memory breakpoint");
  ld_brk_point = (md_addr_t)brkrec->as_integer.val;

  /* write integer register outputs */
  for (i=MD_FIRST_OUT_REG, regrec=exo_outregs->as_list.head->next;
       i <= MD_LAST_OUT_REG; i++, regrec=regrec->next) {
	if (!regrec || regrec->ec != ec_address)
	  fatal("EIO trace inconsistency: missing output reg");

	MD_EXO_TO_IREG(regrec, regs, i);

#ifdef VERBOSE
	fprintf(stderr, "** R[%d] written...\n", i);
#endif /* VERBOSE */
  }
  if (regrec != NULL)
    fatal("EIO trace inconsistency: too many output regs");

  /* write memory outputs */
  for (memrec=exo_outmem->as_list.head; memrec != NULL; memrec=memrec->next) {
	md_addr_t loc;
	struct exo_term_t *addr, *blob;

	/* check the mem transaction format */
	if (!memrec
		|| memrec->ec != ec_list
		|| !(addr = memrec->as_list.head)
		|| addr->ec != ec_address
		|| !(blob = addr->next)
		|| blob->ec != ec_blob
		|| blob->next != NULL)
	  fatal("EIO trace icnonsistency: bad memory transaction");

	for (loc=addr->as_integer.val, i=0; i < blob->as_blob.size; loc++,i++) {
	  /* was: MEM_WRITE_BYTE(loc, blob->as_blob.data[i]); */
	  mem_simple->mem_access(mem, Write,
							 loc, &blob->as_blob.data[i],
							 sizeof(unsigned char), 1);

#ifdef VERBOSE
	  fprintf(stderr, "** 0x%08p written...\n", loc);
#endif /* VERBOSE */
	}
  }

  /* release the EIO EXO node */
  lib.exo_delete(exo);
}

/* fast forward EIO trace EIO_FD to the transaction just after ICNT */
void 
loader_simple::eio_fast_forward(FILE *eio_fd, counter_t icnt)
{
  struct exo_term_t *exo, *exo_icnt;

  do {
	/* read the next external I/O (EIO) transaction */
	exo = lib.exo_read(eio_fd);

	if (!exo)
	  fatal("could not fast forward to EIO checkpoint");

	/* one more transaction processed */
	eio_trans_icnt = icnt;

	/* pull apart the EIO transaction (EXO format) */
	if (!exo
		|| exo->ec != ec_list
		|| !(exo_icnt = exo->as_list.head)
		|| exo_icnt->ec != ec_integer)
	  fatal("cannot read EIO transaction (during fast forward)");
  }
  while ((counter_t)exo_icnt->as_integer.val != icnt);

  /* found it! */
}


void
loader_simple::sys_syscall(struct regs_t *regs,/* registers to access */
						   struct mem_t *mem,/* memory space to access */
						   md_inst_t inst,   /* system call inst */
						   int traceable,	/* traceable system call? */
						   size_t puid,
						   counter_t sim_num_insn)
{
  word_t syscode = regs->regs_R[2];
 
  /* first, check if an EIO trace is being consumed... */
  if (traceable && sim_eio_fd != NULL) {
	eio_read_trace(sim_eio_fd, sim_num_insn, regs, mem, inst);
	/* fini... */
	return;
  }
#ifdef MD_CROSS_ENDIAN
  else if (syscode == SS_SYS_exi) {
	/* exit jumps to the target set in main() */
	longjmp(*sim_exit_buf, /* exitcode + fudge */regs->regs_R[4]+1);
  } else
    fatal("cannot execute PISA system call on cross-endian host");
  
#else /* !MD_CROSS_ENDIAN */

  /* no, OK execute the live system call... */
  switch (syscode)
    {
    case SS_SYS_exit:
	  if ( puid == 0 ) {
		/* exit jumps to the target set in main() */
		longjmp(*sim_exit_buf, /* exitcode + fudge */regs->regs_R[4]+1);
	  }		
	  break;  
    case SS_SYS_read:
      {
		char *buf;
		
		/* allocate same-sized input buffer in host memory */
		if (!(buf = (char *)calloc(/*nbytes*/regs->regs_R[6], sizeof(char))))
		  fatal("out of memory in SYS_read");
		
		/* read data from file */
		/*nread*/regs->regs_R[2] =
				   read(/*fd*/regs->regs_R[4], buf, /*nbytes*/regs->regs_R[6]);
		
		/* check for error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* copy results back into host memory */
		mem_simple->mem_bcopy(mem,
							  Write, /*buf*/regs->regs_R[5],
							  buf, /*nread*/regs->regs_R[2]);
		
		/* done with input buffer */
		free(buf);
      }
      break;	  
    case SS_SYS_write:
      {
		char *buf;
		
		/* allocate same-sized output buffer in host memory */
		if (!(buf = (char *)calloc(/*nbytes*/regs->regs_R[6], sizeof(char))))
		  fatal("out of memory in SYS_write");
		
		/* copy inputs into host memory */
		mem_simple->mem_bcopy(mem,
							  Read, /*buf*/regs->regs_R[5],
							  buf, /*nbytes*/regs->regs_R[6]);
		
		/* write data to file */
		if (sim_progfd && MD_OUTPUT_SYSCALL(regs)) {
		  /* redirect program output to file */
		  
		  /*nwritten*/regs->regs_R[2] =
						fwrite(buf, 1, /*nbytes*/regs->regs_R[6], sim_progfd);
		} else {
		  /* perform program output request */
		  
		  /*nwritten*/regs->regs_R[2] =
						write(/*fd*/regs->regs_R[4],
							  buf, /*nbytes*/regs->regs_R[6]);
		}
		
		/* check for an error condition */
		if (regs->regs_R[2] == regs->regs_R[6])
		  /*result*/regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
		
		/* done with output buffer */
		free(buf);
      }
      break;	  
    case SS_SYS_open:
      {
		char buf[MAXBUFSIZE];
		unsigned int i;
		int ss_flags = regs->regs_R[5], local_flags = 0;
		
		/* translate open(2) flags */
		for (i=0; i<SS_NFLAGS; i++) {
		  if (ss_flags & ss_flag_table[i].ss_flag) {
			ss_flags &= ~ss_flag_table[i].ss_flag;
			local_flags |= ss_flag_table[i].local_flag;
	      }
		}
		/* any target flags left? */
		if (ss_flags != 0)
		  fatal("syscall: open: cannot decode flags: 0x%08x", ss_flags);
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);
		
		/* open the file */
		/*fd*/regs->regs_R[2] =
				open(buf, local_flags, /*mode*/regs->regs_R[6]);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_close:
      /* don't close stdin, stdout, or stderr as this messes up sim logs */
      if (/*fd*/regs->regs_R[4] == 0
		  || /*fd*/regs->regs_R[4] == 1
		  || /*fd*/regs->regs_R[4] == 2) {
		regs->regs_R[7] = 0;
		break;
	  }
	  
      /* close the file */
      regs->regs_R[2] = close(/*fd*/regs->regs_R[4]);
	  
      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
      break;	  
    case SS_SYS_creat:
      {
		char buf[MAXBUFSIZE];
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);
		
		/* create the file */
		/*fd*/regs->regs_R[2] = creat(buf, /*mode*/regs->regs_R[5]);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_unlink:
      {
		char buf[MAXBUFSIZE];
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);
		
		/* delete the file */
		/*result*/regs->regs_R[2] = unlink(buf);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_chdir:
      {
		char buf[MAXBUFSIZE];
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);
		
		/* change the working directory */
		/*result*/regs->regs_R[2] = chdir(buf);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_chmod:
      {
		char buf[MAXBUFSIZE];
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);
		
		/* chmod the file */
		/*result*/regs->regs_R[2] = chmod(buf, /*mode*/regs->regs_R[5]);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_chown:
#ifdef _MSC_VER
      warn("syscall chown() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* !_MSC_VER */
      {
		char buf[MAXBUFSIZE];
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);
		
		/* chown the file */
		/*result*/regs->regs_R[2] = chown(buf, /*owner*/regs->regs_R[5],
										  /*group*/regs->regs_R[6]);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
#endif /* _MSC_VER */
      break;
    case SS_SYS_brk:
      {
		md_addr_t addr;
		
		/* round the new heap pointer to the its page boundary */
		addr = ROUND_UP(/*base*/regs->regs_R[4], MD_PAGE_SIZE);
		
		/* check whether heap area has merged with stack area */
		if (addr >= ld_brk_point && addr < (md_addr_t)regs->regs_R[29]) {
		  regs->regs_R[2] = 0;
		  regs->regs_R[7] = 0;
		  ld_brk_point = addr;
		} else {
		  /* out of address space, indicate error */
		  regs->regs_R[2] = ENOMEM;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_lseek:
      /* seek into file */
      regs->regs_R[2] =
		lseek(/*fd*/regs->regs_R[4],
			  /*off*/regs->regs_R[5], /*dir*/regs->regs_R[6]);
	  
      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
      break;
    case SS_SYS_getpid:
      /* get the simulator process id */
      /*result*/regs->regs_R[2] = getpid();
	
	/* check for an error condition */
	if (regs->regs_R[2] != -1)
	  regs->regs_R[7] = 0;
	else {
	  /* got an error, return details */
	  regs->regs_R[2] = errno;
	  regs->regs_R[7] = 1;
	}
	break;
	
    case SS_SYS_getuid:
#ifdef _MSC_VER
      warn("syscall getuid() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* !_MSC_VER */
      /* get current user id */
      /*first result*/regs->regs_R[2] = getuid();
	  
      /* get effective user id */
      /*second result*/regs->regs_R[3] = geteuid();
	  
      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
#endif /* _MSC_VER */
      break;
	  
    case SS_SYS_access:
      {
		char buf[MAXBUFSIZE];
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fName*/regs->regs_R[4], buf);
		
		/* check access on the file */
		/*result*/regs->regs_R[2] = access(buf, /*mode*/regs->regs_R[5]);
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
    case SS_SYS_stat:
    case SS_SYS_lstat:
      {
		char buf[MAXBUFSIZE];
		struct ss_statbuf ss_sbuf;
#ifdef _MSC_VER
		struct _stat sbuf;
#else /* !_MSC_VER */
		struct stat sbuf;
#endif /* _MSC_VER */
		
		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fName*/regs->regs_R[4], buf);
		
		/* stat() the file */
		if (syscode == SS_SYS_stat)
		  /*result*/regs->regs_R[2] = stat(buf, &sbuf);
		else /* syscode == SS_SYS_lstat */ {
#ifdef _MSC_VER
		  warn("syscall lstat() not yet implemented for MSC...");
		  regs->regs_R[7] = 0;
		  break;
#else /* !_MSC_VER */
		  /*result*/regs->regs_R[2] = lstat(buf, &sbuf);
#endif /* _MSC_VER */
		}
		
		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
		
		/* translate from host stat structure to target format */
		ss_sbuf.ss_st_dev = MD_SWAPH(sbuf.st_dev);
		ss_sbuf.ss_pad = 0;
		ss_sbuf.ss_st_ino = MD_SWAPW(sbuf.st_ino);
		ss_sbuf.ss_st_mode = MD_SWAPH(sbuf.st_mode);
		ss_sbuf.ss_st_nlink = MD_SWAPH(sbuf.st_nlink);
		ss_sbuf.ss_st_uid = MD_SWAPH(sbuf.st_uid);
		ss_sbuf.ss_st_gid = MD_SWAPH(sbuf.st_gid);
		ss_sbuf.ss_st_rdev = MD_SWAPH(sbuf.st_rdev);
		ss_sbuf.ss_pad1 = 0;
		ss_sbuf.ss_st_size = MD_SWAPW(sbuf.st_size);
		ss_sbuf.ss_st_atime = MD_SWAPW(sbuf.st_atime);
		ss_sbuf.ss_st_spare1 = 0;
		ss_sbuf.ss_st_mtime = MD_SWAPW(sbuf.st_mtime);
		ss_sbuf.ss_st_spare2 = 0;
		ss_sbuf.ss_st_ctime = MD_SWAPW(sbuf.st_ctime);
		ss_sbuf.ss_st_spare3 = 0;
#ifndef _MSC_VER
		ss_sbuf.ss_st_blksize = MD_SWAPW(sbuf.st_blksize);
		ss_sbuf.ss_st_blocks = MD_SWAPW(sbuf.st_blocks);
#endif /* !_MSC_VER */
		ss_sbuf.ss_st_gennum = 0;
		ss_sbuf.ss_st_spare4 = 0;
		
		/* copy stat() results to simulator memory */
		mem_simple->mem_bcopy(mem, Write, /*sbuf*/regs->regs_R[5],
							  &ss_sbuf, sizeof(struct ss_statbuf));
      }
      break;
	  
    case SS_SYS_dup:
      /* dup() the file descriptor */
      /*fd*/regs->regs_R[2] = dup(/*fd*/regs->regs_R[4]);
	
	/* check for an error condition */
	if (regs->regs_R[2] != -1)
	  regs->regs_R[7] = 0;
	else {
	  /* got an error, return details */
	  regs->regs_R[2] = errno;
	  regs->regs_R[7] = 1;
	}
	break;

#ifndef _MSC_VER
    case SS_SYS_pipe:
      {
		int fd[2];
		
		/* copy pipe descriptors to host memory */;
		mem_simple->mem_bcopy(mem, Read, regs->regs_R[4], fd, sizeof(fd));
		
		/* create a pipe */
		/*result*/regs->regs_R[7] = pipe(fd);
		
		/* copy descriptor results to result registers */
		/*pipe1*/regs->regs_R[2] = fd[0];
		/*pipe 2*/regs->regs_R[3] = fd[1];
		
		/* check for an error condition */
		if (regs->regs_R[7] == -1) {
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;
#endif
	  
    case SS_SYS_getgid:
#ifdef _MSC_VER
      warn("syscall getgid() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* !_MSC_VER */
      /* get current group id */
      /*first result*/regs->regs_R[2] = getgid();
	  
      /* get current effective group id */
      /*second result*/regs->regs_R[3] = getegid();
	  
	  /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
#endif /* _MSC_VER */
      break;
	  
    case SS_SYS_ioctl:
      {
		char buf[NUM_IOCTL_BYTES];
		int local_req = 0;
		
		/* convert target ioctl() request to host ioctl() request values */
		switch (/*req*/regs->regs_R[5]) {
#ifdef TCGETA
		case SS_IOCTL_TCGETA:
		  local_req = TCGETA;
		  break;
#endif
#ifdef TIOCGWINSZ
		case SS_IOCTL_TIOCGWINSZ:
		  local_req = TIOCGWINSZ;
		  break;
#endif
#ifdef TCSETAW
		case SS_IOCTL_TCSETAW:
		  local_req = TCSETAW;
		  break;
#endif
		}
		
#if !defined(TIOCGETP) && (defined(linux) || defined(__CYGWIN32__))
        if (!local_req && /*req*/regs->regs_R[5] == SS_IOCTL_TIOCGETP) {
		  struct termios lbuf;
		  struct ss_sgttyb buf;
		  
		  /* result */regs->regs_R[2] =
						tcgetattr(/* fd */(int)regs->regs_R[4], &lbuf);
		  
		  /* translate results */
		  buf.sg_ispeed = lbuf.c_ispeed;
		  buf.sg_ospeed = lbuf.c_ospeed;
		  buf.sg_erase = lbuf.c_cc[VERASE];
		  buf.sg_kill = lbuf.c_cc[VKILL];
		  buf.sg_flags = 0;   /* FIXME: this is wrong... */
		  
		  mem_simple->mem_bcopy(mem, Write,
								/* buf */regs->regs_R[6], &buf,
								sizeof(struct ss_sgttyb));
		  
		  if (regs->regs_R[2] != -1)
			regs->regs_R[7] = 0;
		  else /* probably not a typewriter, return details */ {
			regs->regs_R[2] = errno;
			regs->regs_R[7] = 1;
		  }
		} else
#endif
		  
		  if (!local_req) {
			/* FIXME: could not translate the ioctl() request, just warn user
			   and ignore the request */
			warn("syscall: ioctl: ioctl code not supported d=%d, req=%d",
				 regs->regs_R[4], regs->regs_R[5]);
			regs->regs_R[2] = 0;
			regs->regs_R[7] = 0;
		  } else {
#ifdef _MSC_VER
			warn("syscall getgid() not yet implemented for MSC...");
			regs->regs_R[7] = 0;
			break;
#else /* !_MSC_VER */
			  
			/* ioctl() code was successfully translated to a host code */
			  
			/* if arg ptr exists, copy NUM_IOCTL_BYTES bytes to host mem */
			if (/*argp*/regs->regs_R[6] != 0)
			  mem_simple->mem_bcopy(mem,
									Read, /*argp*/regs->regs_R[6],
									buf, NUM_IOCTL_BYTES);

			/* perform the ioctl() call */
			/*result*/regs->regs_R[2] =
						ioctl(/*fd*/regs->regs_R[4], local_req, buf);
			  
			/* if arg ptr exists, copy NUM_IOCTL_BYTES bytes from host mem */
			if (/*argp*/regs->regs_R[6] != 0)
			  mem_simple->mem_bcopy(mem, Write, regs->regs_R[6],
									buf, NUM_IOCTL_BYTES);
			  
			/* check for an error condition */
			if (regs->regs_R[2] != -1)
			  regs->regs_R[7] = 0;
			else {	
			  /* got an error, return details */
			  regs->regs_R[2] = errno;
			  regs->regs_R[7] = 1;
			}
#endif /* _MSC_VER */
		  }
      }
      break;
	  
    case SS_SYS_fstat:
      {
		struct ss_statbuf ss_sbuf;
#ifdef _MSC_VER
		struct _stat sbuf;
#else /* !_MSC_VER */
		struct stat sbuf;
#endif /* _MSC_VER */

		/* fstat() the file */
		/*result*/regs->regs_R[2] = fstat(/*fd*/regs->regs_R[4], &sbuf);

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* translate the stat structure to host format */
		ss_sbuf.ss_st_dev = MD_SWAPH(sbuf.st_dev);
		ss_sbuf.ss_pad = 0;
		ss_sbuf.ss_st_ino = MD_SWAPW(sbuf.st_ino);
		ss_sbuf.ss_st_mode = MD_SWAPH(sbuf.st_mode);
		ss_sbuf.ss_st_nlink = MD_SWAPH(sbuf.st_nlink);
		ss_sbuf.ss_st_uid = MD_SWAPH(sbuf.st_uid);
		ss_sbuf.ss_st_gid = MD_SWAPH(sbuf.st_gid);
		ss_sbuf.ss_st_rdev = MD_SWAPH(sbuf.st_rdev);
		ss_sbuf.ss_pad1 = 0;
		ss_sbuf.ss_st_size = MD_SWAPW(sbuf.st_size);
		ss_sbuf.ss_st_atime = MD_SWAPW(sbuf.st_atime);
        ss_sbuf.ss_st_spare1 = 0;
		ss_sbuf.ss_st_mtime = MD_SWAPW(sbuf.st_mtime);
        ss_sbuf.ss_st_spare2 = 0;
		ss_sbuf.ss_st_ctime = MD_SWAPW(sbuf.st_ctime);
        ss_sbuf.ss_st_spare3 = 0;
#ifndef _MSC_VER
		ss_sbuf.ss_st_blksize = MD_SWAPW(sbuf.st_blksize);
		ss_sbuf.ss_st_blocks = MD_SWAPW(sbuf.st_blocks);
#endif /* !_MSC_VER */
        ss_sbuf.ss_st_gennum = 0;
        ss_sbuf.ss_st_spare4 = 0;

		/* copy fstat() results to simulator memory */
		mem_simple->mem_bcopy(mem, Write, /*sbuf*/regs->regs_R[5],
							  &ss_sbuf, sizeof(struct ss_statbuf));
      }
      break;

    case SS_SYS_getpagesize:
      /* get target pagesize */
      regs->regs_R[2] = /* was: getpagesize() */MD_PAGE_SIZE;

      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
      break;

    case SS_SYS_setitimer:
      /* FIXME: the sigvec system call is ignored */
      regs->regs_R[2] = regs->regs_R[7] = 0;
      warn("syscall: setitimer ignored");
      break;

    case SS_SYS_getdtablesize:
#if defined(_AIX)
      /* get descriptor table size */
      regs->regs_R[2] = getdtablesize();

      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
#elif defined(__CYGWIN32__) || defined(ultrix) || defined(_MSC_VER)
      {
		/* no comparable system call found, try some reasonable defaults */
		warn("syscall: called getdtablesize()\n");
		regs->regs_R[2] = 16;
		regs->regs_R[7] = 0;
      }
#else
      {
		struct rlimit rl;

		/* get descriptor table size in rlimit structure */
		if (getrlimit(RLIMIT_NOFILE, &rl) != -1) {
		  regs->regs_R[2] = rl.rlim_cur;
		  regs->regs_R[7] = 0;
		} else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
#endif
      break;

    case SS_SYS_dup2:
      /* dup2() the file descriptor */
      regs->regs_R[2] =
		dup2(/* fd1 */regs->regs_R[4], /* fd2 */regs->regs_R[5]);

      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
      break;

    case SS_SYS_fcntl:
#ifdef _MSC_VER
      warn("syscall fcntl() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* !_MSC_VER */
      /* get fcntl() information on the file */
      regs->regs_R[2] =
		fcntl(/*fd*/regs->regs_R[4], /*cmd*/regs->regs_R[5],
			  /*arg*/regs->regs_R[6]);

      /* check for an error condition */
      if (regs->regs_R[2] != -1)
		regs->regs_R[7] = 0;
      else {
		/* got an error, return details */
		regs->regs_R[2] = errno;
		regs->regs_R[7] = 1;
	  }
#endif /* _MSC_VER */
      break;

    case SS_SYS_select:
#ifdef _MSC_VER
      warn("syscall select() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* !_MSC_VER */
      {
		fd_set readfd, writefd, exceptfd;
		fd_set *readfdp, *writefdp, *exceptfdp;
		struct timeval timeout, *timeoutp;
		word_t param5;

		/* FIXME: swap words? */

		/* read the 5th parameter (timeout) from the stack */
		mem_simple->mem_bcopy(mem,
							  Read, regs->regs_R[29]+16, &param5, sizeof(word_t));

		/* copy read file descriptor set into host memory */
		if (/*readfd*/regs->regs_R[5] != 0) {
		  mem_simple->mem_bcopy(mem, Read, /*readfd*/regs->regs_R[5],
								&readfd, sizeof(fd_set));
		  readfdp = &readfd;
		} else
		  readfdp = NULL;

		/* copy write file descriptor set into host memory */
		if (/*writefd*/regs->regs_R[6] != 0) {
		  mem_simple->mem_bcopy(mem, Read, /*writefd*/regs->regs_R[6],
								&writefd, sizeof(fd_set));
		  writefdp = &writefd;
		} else
		  writefdp = NULL;

		/* copy exception file descriptor set into host memory */
		if (/*exceptfd*/regs->regs_R[7] != 0) {
		  mem_simple->mem_bcopy(mem, Read, /*exceptfd*/regs->regs_R[7],
								&exceptfd, sizeof(fd_set));
		  exceptfdp = &exceptfd;
		} else
		  exceptfdp = NULL;

		/* copy timeout value into host memory */
		if (/*timeout*/param5 != 0) {
		  mem_simple->mem_bcopy(mem, Read, /*timeout*/param5,
								&timeout, sizeof(struct timeval));
		  timeoutp = &timeout;
		} else
		  timeoutp = NULL;

#if defined(hpux) || defined(__hpux)
		/* select() on the specified file descriptors */
		/*result*/regs->regs_R[2] =
					select(/*nfd*/regs->regs_R[4],
						   (int *)readfdp, (int *)writefdp, (int *)exceptfdp, timeoutp);
#else
		/* select() on the specified file descriptors */
		/*result*/regs->regs_R[2] =
					select(/*nfd*/regs->regs_R[4],
						   readfdp, writefdp, exceptfdp, timeoutp);
#endif

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, return details */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* copy read file descriptor set to target memory */
		if (/*readfd*/regs->regs_R[5] != 0)
		  mem_simple->mem_bcopy(mem, Write, /*readfd*/regs->regs_R[5],
								&readfd, sizeof(fd_set));

		/* copy write file descriptor set to target memory */
		if (/*writefd*/regs->regs_R[6] != 0)
		  mem_simple->mem_bcopy(mem, Write, /*writefd*/regs->regs_R[6],
								&writefd, sizeof(fd_set));

		/* copy exception file descriptor set to target memory */
		if (/*exceptfd*/regs->regs_R[7] != 0)
		  mem_simple->mem_bcopy(mem, Write, /*exceptfd*/regs->regs_R[7],
								&exceptfd, sizeof(fd_set));

		/* copy timeout value result to target memory */
		if (/* timeout */param5 != 0)
		  mem_simple->mem_bcopy(mem, Write, /*timeout*/param5,
								&timeout, sizeof(struct timeval));
      }
#endif
      break;

    case SS_SYS_sigvec:
      /* FIXME: the sigvec system call is ignored */
      regs->regs_R[2] = regs->regs_R[7] = 0;
      warn("syscall: sigvec ignored");
      break;

    case SS_SYS_sigblock:
      /* FIXME: the sigblock system call is ignored */
      regs->regs_R[2] = regs->regs_R[7] = 0;
      warn("syscall: sigblock ignored");
      break;

    case SS_SYS_sigsetmask:
      /* FIXME: the sigsetmask system call is ignored */
      regs->regs_R[2] = regs->regs_R[7] = 0;
      warn("syscall: sigsetmask ignored");
      break;

    case SS_SYS_gettimeofday:
#ifdef _MSC_VER
      warn("syscall gettimeofday() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* _MSC_VER */
      {
		struct ss_timeval ss_tv;
		struct timeval tv, *tvp;
		struct ss_timezone ss_tz;
		struct timezone tz, *tzp;

		if (/*timeval*/regs->regs_R[4] != 0) {
		  /* copy timeval into host memory */
		  mem_simple->mem_bcopy(mem, Read, /*timeval*/regs->regs_R[4],
								&ss_tv, sizeof(struct ss_timeval));

		  /* convert target timeval structure to host format */
		  tv.tv_sec = MD_SWAPW(ss_tv.ss_tv_sec);
		  tv.tv_usec = MD_SWAPW(ss_tv.ss_tv_usec);
		  tvp = &tv;
		} else
		  tvp = NULL;

		if (/*timezone*/regs->regs_R[5] != 0) {
		  /* copy timezone into host memory */
		  mem_simple->mem_bcopy(mem, Read, /*timezone*/regs->regs_R[5],
								&ss_tz, sizeof(struct ss_timezone));

		  /* convert target timezone structure to host format */
		  tz.tz_minuteswest = MD_SWAPW(ss_tz.ss_tz_minuteswest);
		  tz.tz_dsttime = MD_SWAPW(ss_tz.ss_tz_dsttime);
		  tzp = &tz;
		} else
		  tzp = NULL;

		/* get time of day */
		/*result*/regs->regs_R[2] = gettimeofday(tvp, tzp);

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, indicate result */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		if (/*timeval*/regs->regs_R[4] != 0) {
		  /* convert host timeval structure to target format */
		  ss_tv.ss_tv_sec = MD_SWAPW(tv.tv_sec);
		  ss_tv.ss_tv_usec = MD_SWAPW(tv.tv_usec);

		  /* copy timeval to target memory */
		  mem_simple->mem_bcopy(mem, Write, /*timeval*/regs->regs_R[4],
								&ss_tv, sizeof(struct ss_timeval));
		}

		if (/*timezone*/regs->regs_R[5] != 0) {
		  /* convert host timezone structure to target format */
		  ss_tz.ss_tz_minuteswest = MD_SWAPW(tz.tz_minuteswest);
		  ss_tz.ss_tz_dsttime = MD_SWAPW(tz.tz_dsttime);

		  /* copy timezone to target memory */
		  mem_simple->mem_bcopy(mem, Write, /*timezone*/regs->regs_R[5],
								&ss_tz, sizeof(struct ss_timezone));
		}
      }
#endif /* !_MSC_VER */
      break;

    case SS_SYS_getrusage:
#if defined(__svr4__) || defined(__USLC__) || defined(hpux) || defined(__hpux) || defined(_AIX)
      {
		struct tms tms_buf;
		struct ss_rusage rusage;

		/* get user and system times */
		if (times(&tms_buf) != -1) {
		  /* no error */
		  regs->regs_R[2] = 0;
		  regs->regs_R[7] = 0;
		} else {
		  /* got an error, indicate result */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* initialize target rusage result structure */
#if defined(__svr4__)
		memset(&rusage, '\0', sizeof(struct ss_rusage));
#else /* !defined(__svr4__) */
		bzero(&rusage, sizeof(struct ss_rusage));
#endif

		/* convert from host rusage structure to target format */
		rusage.ss_ru_utime.ss_tv_sec = tms_buf.tms_utime/CLK_TCK;
		rusage.ss_ru_utime.ss_tv_sec = MD_SWAPW(rusage.ss_ru_utime.ss_tv_sec);
		rusage.ss_ru_utime.ss_tv_usec = 0;
		rusage.ss_ru_stime.ss_tv_sec = tms_buf.tms_stime/CLK_TCK;
		rusage.ss_ru_stime.ss_tv_sec = MD_SWAPW(rusage.ss_ru_stime.ss_tv_sec);
		rusage.ss_ru_stime.ss_tv_usec = 0;

		/* copy rusage results into target memory */
		mem_simple->mem_bcopy(mem, Write, /*rusage*/regs->regs_R[5],
							  &rusage, sizeof(struct ss_rusage));
      }
#elif defined(__unix__) || defined(unix)
      {
		struct rusage local_rusage;
		struct ss_rusage rusage;

		/* get rusage information */
		/*result*/regs->regs_R[2] =
					getrusage(/*who*/regs->regs_R[4], &local_rusage);

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, indicate result */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* convert from host rusage structure to target format */
		rusage.ss_ru_utime.ss_tv_sec = local_rusage.ru_utime.tv_sec;
		rusage.ss_ru_utime.ss_tv_usec = local_rusage.ru_utime.tv_usec;
		rusage.ss_ru_utime.ss_tv_sec = MD_SWAPW(local_rusage.ru_utime.tv_sec);
		rusage.ss_ru_utime.ss_tv_usec =
		  MD_SWAPW(local_rusage.ru_utime.tv_usec);
		rusage.ss_ru_stime.ss_tv_sec = local_rusage.ru_stime.tv_sec;
		rusage.ss_ru_stime.ss_tv_usec = local_rusage.ru_stime.tv_usec;
		rusage.ss_ru_stime.ss_tv_sec =
		  MD_SWAPW(local_rusage.ru_stime.tv_sec);
		rusage.ss_ru_stime.ss_tv_usec =
		  MD_SWAPW(local_rusage.ru_stime.tv_usec);
		rusage.ss_ru_maxrss = MD_SWAPW(local_rusage.ru_maxrss);
		rusage.ss_ru_ixrss = MD_SWAPW(local_rusage.ru_ixrss);
		rusage.ss_ru_idrss = MD_SWAPW(local_rusage.ru_idrss);
		rusage.ss_ru_isrss = MD_SWAPW(local_rusage.ru_isrss);
		rusage.ss_ru_minflt = MD_SWAPW(local_rusage.ru_minflt);
		rusage.ss_ru_majflt = MD_SWAPW(local_rusage.ru_majflt);
		rusage.ss_ru_nswap = MD_SWAPW(local_rusage.ru_nswap);
		rusage.ss_ru_inblock = MD_SWAPW(local_rusage.ru_inblock);
		rusage.ss_ru_oublock = MD_SWAPW(local_rusage.ru_oublock);
		rusage.ss_ru_msgsnd = MD_SWAPW(local_rusage.ru_msgsnd);
		rusage.ss_ru_msgrcv = MD_SWAPW(local_rusage.ru_msgrcv);
		rusage.ss_ru_nsignals = MD_SWAPW(local_rusage.ru_nsignals);
		rusage.ss_ru_nvcsw = MD_SWAPW(local_rusage.ru_nvcsw);
		rusage.ss_ru_nivcsw = MD_SWAPW(local_rusage.ru_nivcsw);

		/* copy rusage results into target memory */
		mem_simple->mem_bcopy(mem, Write, /*rusage*/regs->regs_R[5],
							  &rusage, sizeof(struct ss_rusage));
      }
#elif defined(__CYGWIN32__) || defined(_MSC_VER)
	  warn("syscall: called getrusage()\n");
	  regs->regs_R[7] = 0;
#else
#error No getrusage() implementation!
#endif
      break;

    case SS_SYS_writev:
#ifdef _MSC_VER
      warn("syscall writev() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#else /* !_MSC_VER */
      {
		int i;
		char *buf;
		struct iovec *iov;

		/* allocate host side I/O vectors */
		iov =
		  (struct iovec *)malloc(/*iovcnt*/regs->regs_R[6]
								 * sizeof(struct iovec));
		if (!iov)
		  fatal("out of virtual memory in SYS_writev");

		/* copy target side pointer data into host side vector */
		mem_simple->mem_bcopy(mem, Read, /*iov*/regs->regs_R[5],
							  iov, /*iovcnt*/regs->regs_R[6] * sizeof(struct iovec));

		/* copy target side I/O vector buffers to host memory */
		for (i=0; i < /*iovcnt*/regs->regs_R[6]; i++) {
		  iov[i].iov_base = (char *)MD_SWAPW((unsigned)iov[i].iov_base);
		  iov[i].iov_len = MD_SWAPW(iov[i].iov_len);
		  if (iov[i].iov_base != NULL) {
			buf = (char *)calloc(iov[i].iov_len, sizeof(char));
			if (!buf)
			  fatal("out of virtual memory in SYS_writev");
			mem_simple->mem_bcopy(mem, Read, (md_addr_t)iov[i].iov_base,
								  buf, iov[i].iov_len);
			iov[i].iov_base = buf;
	      }
		}

		/* perform the vector'ed write */
		/*result*/regs->regs_R[2] =
					writev(/*fd*/regs->regs_R[4], iov, /*iovcnt*/regs->regs_R[6]);

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, indicate results */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* free all the allocated memory */
		for (i=0; i < /*iovcnt*/regs->regs_R[6]; i++) {
		  if (iov[i].iov_base) {
			free(iov[i].iov_base);
			iov[i].iov_base = NULL;
	      }
		}
		free(iov);
      }
#endif /* !_MSC_VER */
      break;

    case SS_SYS_utimes:
      {
		char buf[MAXBUFSIZE];

		/* copy filename to host memory */
		mem_simple->mem_strcpy(mem, Read, /*fname*/regs->regs_R[4], buf);

		if (/*timeval*/regs->regs_R[5] == 0) {
#if defined(hpux) || defined(__hpux) || defined(linux)
		  /* no utimes() in hpux, use utime() instead */
		  /*result*/regs->regs_R[2] = utime(buf, NULL);
#elif defined(_MSC_VER)
		  /* no utimes() in MSC, use utime() instead */
		  /*result*/regs->regs_R[2] = utime(buf, NULL);
#elif defined(__svr4__) || defined(__USLC__) || defined(unix) || defined(_AIX) || defined(__alpha)
		  /*result*/regs->regs_R[2] = utimes(buf, NULL);
#elif defined(__CYGWIN32__)
		  warn("syscall: called utimes()\n");
#else
#error No utimes() implementation!
#endif
		} else {
		  struct ss_timeval ss_tval[2];
#ifndef _MSC_VER
		  struct timeval tval[2];
#endif /* !_MSC_VER */

		  /* copy timeval structure to host memory */
		  mem_simple->mem_bcopy(mem, Read, /*timeout*/regs->regs_R[5],
								ss_tval, 2*sizeof(struct ss_timeval));

#ifndef _MSC_VER
		  /* convert timeval structure to host format */
		  tval[0].tv_sec = MD_SWAPW(ss_tval[0].ss_tv_sec);
		  tval[0].tv_usec = MD_SWAPW(ss_tval[0].ss_tv_usec);
		  tval[1].tv_sec = MD_SWAPW(ss_tval[1].ss_tv_sec);
		  tval[1].tv_usec = MD_SWAPW(ss_tval[1].ss_tv_usec);
#endif /* !_MSC_VER */

#if defined(hpux) || defined(__hpux) || defined(__svr4__)
		  /* no utimes() in hpux, use utime() instead */
		  {
			struct utimbuf ubuf;

			ubuf.actime = tval[0].tv_sec;
			ubuf.modtime = tval[1].tv_sec;

			/* result */regs->regs_R[2] = utime(buf, &ubuf);
		  }
#elif defined(_MSC_VER)
		  /* no utimes() in MSC, use utime() instead */
		  {
			struct _utimbuf ubuf;

			ubuf.actime = ss_tval[0].ss_tv_sec;
			ubuf.modtime = ss_tval[1].ss_tv_sec;

			/* result */regs->regs_R[2] = utime(buf, &ubuf);
		  }
#elif defined(__USLC__) || defined(unix) || defined(_AIX) || defined(__alpha)
		  /* result */regs->regs_R[2] = utimes(buf, tval);
#elif defined(__CYGWIN32__)
		  warn("syscall: called utimes()\n");
#else
#error No utimes() implementation!
#endif
		}

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, indicate results */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}
      }
      break;

    case SS_SYS_getrlimit:
    case SS_SYS_setrlimit:
#ifdef _MSC_VER
      warn("syscall get/setrlimit() not yet implemented for MSC...");
      regs->regs_R[7] = 0;
#elif defined(__CYGWIN32__)
      warn("syscall: called get/setrlimit()\n");
      regs->regs_R[7] = 0;
#else
	  {
		/* FIXME: check this..., was: struct rlimit ss_rl; */
		struct ss_rlimit ss_rl;
		struct rlimit rl;

		/* copy rlimit structure to host memory */
		mem_simple->mem_bcopy(mem, Read, /*rlimit*/regs->regs_R[5],
							  &ss_rl, sizeof(struct ss_rlimit));

		/* convert rlimit structure to host format */
		rl.rlim_cur = MD_SWAPW(ss_rl.ss_rlim_cur);
		rl.rlim_max = MD_SWAPW(ss_rl.ss_rlim_max);

		/* get rlimit information */
		if (syscode == SS_SYS_getrlimit)
		  /*result*/regs->regs_R[2] = getrlimit(regs->regs_R[4], &rl);
		else /* syscode == SS_SYS_setrlimit */
		  /*result*/regs->regs_R[2] = setrlimit(regs->regs_R[4], &rl);

		/* check for an error condition */
		if (regs->regs_R[2] != -1)
		  regs->regs_R[7] = 0;
		else {
		  /* got an error, indicate results */
		  regs->regs_R[2] = errno;
		  regs->regs_R[7] = 1;
		}

		/* convert rlimit structure to target format */
		ss_rl.ss_rlim_cur = MD_SWAPW(rl.rlim_cur);
		ss_rl.ss_rlim_max = MD_SWAPW(rl.rlim_max);

		/* copy rlimit structure to target memory */
		mem_simple->mem_bcopy(mem, Write, /*rlimit*/regs->regs_R[5],
							  &ss_rl, sizeof(struct ss_rlimit));
      }
#endif
      break;

    default:
      panic("invalid/unimplemented system call encountered, code %d", syscode);
    }

#endif /* MD_CROSS_ENDIAN */
  
}

