/*
 * <<< vision_test.cc >>>
 *
 * --- Test program for vision
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#include <cstdlib>
#include <iostream>
#include <isis/cui_debugger.h>
#include <isis/halt_detector.h>
#include "pico_processing_element.h"
#include <isis/bus_port.h>
#include <isis/isis.h>
#include "pico_typedef.h"
#include "pico.h"
#include "receive_reg.h"
#include <sys/types.h>
#include <sys/stat.h>
#include "vision_test.h"
#include "vision_packet.h"
#include "vision_port.h"
#include "vdec.h"

#define READLINE 1
//#define DEBUG 1

#ifdef READLINE
#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#endif //READLINE

using namespace std;

//#define EVAL 1

#ifdef EVAL
		__inline__ unsigned long long int rdtsc()
		{   
		    unsigned long long int x;
			    __asm__ volatile (".byte 0x0f, 0x31" : "=A" (x));
				    return x;
					}
#endif



static const int PE_SIZE = 4;
// VAOU_SIZE * VAOU_SIZE <= vision_input_unit::BUFFER_SIZE
const int VAOU_SIZE = 6;
int PE_NUM = 10;

int main(int, char** argv)
{
	typedef pico_processing_element processing_element_type;
	typedef processing_element_type::size_type size_type;
	typedef pico_word word;
	const pico_word local_memory_address = 0;
	const pico_word local_memory_size = 0x4000;
	unsigned int verbose_level = 0;

	const int REG_SIZE = 8;
	const int SP_REG = 9;
	const int TB_REG = 17;
	const int BB_REG = 25;


	const int BUF_SIZE = 100;
	const int PRINT_LINE = 11;
	const int System_Stack_Pointer = 35;
	const int OP_LENGS = 6;

	// input unit port & packet
	vision_port Input_Unit0[PE_SIZE * PE_SIZE];
	vision_port Input_Unit1[PE_SIZE * PE_SIZE];
	packet *Input_pkt[PE_SIZE * PE_SIZE];
	//vision_port& Input_Unit0_Port() { return Input_Unit0; }
	//vision_port& Input_Unit1_Port() { return Input_Unit1; }

	// output unit port & packet
	vision_port Result_Unit0[PE_SIZE * PE_SIZE];
	vision_port Result_Unit1[PE_SIZE * PE_SIZE];
	vision_port Result_Unit;
	packet *Result_pkt;
	//vision_port& Result_Unit0_Port() { return Result_Unit0; }
	//vision_port& Result_Unit1_Port() { return Result_Unit1; }

	int pushs_count = 0;
	int pops_count = 0;

	mkdir("log", 0755);
#ifdef HISTORY
	ifstream history("log/.history", ios::out | ios::trunc);
#endif
	ofstream sst_dump("log/system_stack_trace", ios::out | ios::trunc);

	ofstream reg_dump("log/reg_dump", ios::out | ios::trunc);
	ofstream sp_dump("log/sp_dump", ios::out | ios::trunc);
	ofstream ssp_dump("log/systemp_dump", ios::out | ios::trunc);
	ofstream tb_dump("log/tb_dump", ios::out | ios::trunc);
	ofstream bb_dump("log/bb_dump", ios::out | ios::trunc);
	ofstream trans_buffer_dump("log/trans_buffer_dump", 
					ios::out | ios::trunc);

	ofstream memory_trace("log/memory_trace", ios::out | ios::trunc);
	ofstream mem_dump("log/mem_dump", ios::out | ios::trunc);

	halt_detector<word> mon;
	bus_port<word> bus_if;

	mon.port_ref().connect(bus_if);

	bool memory_dump_flag = true, register_dump_flag = true,
		 clock_flag = true, trace_flag = false, tb_dump_flag = false,
		 sp_dump_flag = false, bb_dump_flag = false, 
		 display_flag = false, hunga_flag= false,
		 trans_buffer_dump_flag = false, memory_access_flag = false,
		 system_stack_trace_flag_ = false, input_flag = false;

	ifstream file;

	// read commandline argument for simulator
	while (*++argv != NULL && **argv == '-') {
		while (*++*argv != '\0') {
			bool flag = 0;
			switch (**argv) {
			case 'b': bb_dump_flag = true; break;
			case 'c': clock_flag = false; break;
			case 'd': display_flag = true; break;
			case 'f': trans_buffer_dump_flag = true; break;
			case 'h': hunga_flag = true; break;
			case 'm': memory_access_flag = true; break;
			case 'r': register_dump_flag = false; break;
			case 's': sp_dump_flag = true; break;
			case 't': tb_dump_flag = true; break;
			case 'v': verbose_level++; break;
			case 'y': system_stack_trace_flag_ = true; break;
			case 'p': PE_NUM = *++*argv - '0'; break;
			case 'i': input_flag = true; break;
			//case 'p': PE_NUM = atoi(*argv); break;
			default: break;
			}
			if (flag) break;
		}

	}

	//cout << PE_NUM << endl;
	if(PE_NUM >= PE_SIZE * PE_SIZE){
		cerr << "Report PE Number is over !" << endl;
		exit(1);
	}

	if (*argv == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}
	
	// setup simulation conditions
	processing_element_type pe[PE_SIZE * PE_SIZE];

	VideoDecoder vdec = VideoDecoder(PE_SIZE, PE_SIZE, 1, VAOU_SIZE,
#if defined(__V4L__)
									 VideoDecoder::RGB555,
#else
									 VideoDecoder::RGB888,
#endif
									 VideoDecoder::SVIDEO,
									 VideoDecoder::NTSC);

	int i, k = 0;

	/* connect PE */

	for(k = 0; k < PE_SIZE; k++){
		for(i = 0; i < PE_SIZE - 1; i++){
			left_right_connect_function(pe[i + k * PE_SIZE],
											pe[i+ k * PE_SIZE + 1]);
		}
	}

	for(k = 0; k < PE_SIZE; k++){
		for(i = 0; i < PE_SIZE - 1; i++){
			up_down_connect_function(pe[i * PE_SIZE + k],
											pe[(i + 1) * PE_SIZE + k]);
		}
	}

	/* connect Input and Output */
	for(k = 0; k < PE_SIZE * PE_SIZE; k++){
		input_output_connect_function(pe[k]);
	}

	/* connect Input Data Port */
	for(k = 0; k < PE_SIZE * PE_SIZE; k++){
		pe[k].processor().cont_unit0->before_input().connect(
													Input_Unit0[k]);
		pe[k].processor().cont_unit1->before_input().connect(
													Input_Unit1[k]);
	}


	/* connect Output Data Port */
	for(k = 0; k < PE_SIZE * PE_SIZE; k++){
		pe[k].processor().cont_unit0->result_output().connect(
													Result_Unit);
		pe[k].processor().cont_unit1->result_output().connect(
													Result_Unit);
	}

	/*  load program */

	int halt_adr = 0x0024;

	for(k = 0; k < (PE_SIZE * PE_SIZE) ; k++){
		if (!pe[k].load(*argv, halt_adr)) {
			cerr << *argv << ": No such file or directory." << endl;
			return 1;
		}
	}


	mon.set_address(halt_adr);
	//cout << "Halt Address : " << hex << halt_adr << endl;

	if(memory_dump_flag){
		pe[PE_NUM].processor().local_memory -> dump(mem_dump);
	}

	for(i = 0; i < PE_SIZE; i++){
		pe[i].set_file_table_size(16);
		pe[i].set_standard_input_stream(cin);
		pe[i].set_standard_output_stream(cout);
		pe[i].set_standard_error_stream(cerr);
		pe[i].set_commandline_argument((const char* const*)(argv));
	}

	/* timer setting */
	//pe.processor().set_timer_address(0x32fe);
	//pe.processor().set_timer_count(0x1);
	

	// show simulation conditions

	if (verbose_level >= 1) {
		cout << "processor:   pico" << endl;
		cout << "memory:      0x" << hex << setw(8) << setfill('0')
			 << local_memory_address << " - 0x" << setw(8) << setfill('0')
			 << local_memory_address + local_memory_size - 1 << dec << endl;
		cout << "arguments:   \"";
		for (int i = 0; argv[i] != NULL; i++) {
			if (i != 0) cout << ' ';
			cout << argv[i];
		}
		cout << '"' << endl << endl;
	}

#ifdef EVAL
	unsigned long long int count_time =  rdtsc();
	unsigned long long int count_time2;
#endif

	if (!clock_flag) {
		// execute simulation in non-interactive mode
		// evaluate version
		int cl = 0;
		
#ifndef EVAL
		while (1) {
		
			if (verbose_level >= 2) {
				cout << "clk:" << pe[0].timer_clock_value() << endl
					 << pe[0].processor() << endl << endl;
			} else if (trace_flag) {
				cout << "0x" << hex <<
					pe[0].processor().print_program_counter()
					 << dec << endl;
			}

			if (pe[PE_NUM].is_halt()){
				cout << "Processor is halt! " << endl;
				break;
			}

			if (pe[PE_NUM].is_bus_error()) {
				cout << hex
					 << "bus error(I:0x" << pe[PE_NUM].processor().print_program_counter()
					 << ", D:0x" << pe[PE_NUM].bus_error_address() << ')'
					 << dec << endl;
				break;
			}
			for(i = 0; i < PE_SIZE * PE_SIZE; i++){
				bus_if.send_single_write_request(pe[i].processor().program_counter());
			}
			

			for(i = 0; i < PE_SIZE * PE_SIZE; i++){
				pe[i].clock_in();
				pe[i].clock_out();
			}

#endif
#ifndef EVAL
				mon.clock_in();
				mon.clock_out();

#endif
#ifdef EVAL
			cl++;
			if(cl > 100000){
				count_time2 = rdtsc();
				cout << (count_time2 - count_time) / (1.533 * 1024 * 1024 * 1024 ) << endl;
				break;
			}

#endif
		//reg_dump << setw(8) << cl << ": ";
#ifndef EVAL		
		if(register_dump_flag){
			reg_dump << setw(8) << hex << 
					pe[PE_NUM].processor().print_program_counter() << ": ";
			for(i = 0; i < REG_SIZE; i++){
				reg_dump << setw(5) << hex << 
					pe[PE_NUM].processor().register_file(i) << " ";
			}
			reg_dump << endl;

			sp_dump << setw(8) << hex << 
				pe[PE_NUM].processor().print_program_counter() << ": ";
			for(i = 0; i < REG_SIZE; i++){
				sp_dump << setw(5) << hex << 
					pe[PE_NUM].processor().register_file(i + SP_REG) << " ";
			}
			sp_dump << endl;

			tb_dump << setw(8) << hex << 
					pe[PE_NUM].processor().print_program_counter()  << ": ";
			for(i = 0; i < REG_SIZE; i++){
				tb_dump << setw(5) << hex <<
					pe[PE_NUM].processor().register_file(i + TB_REG) << " ";
			}
			tb_dump << endl;

			bb_dump << setw(8) << hex << 
					pe[PE_NUM].processor().print_program_counter()  << ": ";
			for(i = 0; i < REG_SIZE; i++){
				bb_dump << setw(5) << hex << 
					pe[PE_NUM].processor().register_file(i + BB_REG) << " ";
			}
			bb_dump << endl;
		}

		if(mon.is_halt()){
			ofstream mem_dump2("log/mem_dump2", ios::out | ios::trunc);
			pe[PE_NUM].processor().local_memory -> dump(mem_dump2);
			break;
		}
		

		}

#endif
		return pe[PE_NUM].commandline_status();

	} else {
		ofstream mem_dump2("log/mem_dump2", ios::out | ios::trunc);


		// set HALT address 
		for(i = 0; i < PE_SIZE * PE_SIZE; i++){
			bus_if.send_single_write_request(pe[i].processor().program_counter());
		}

		bool break_pc_ = false, break_mem_ = false, 
		     break_reg_ = false, break_op_ = false;
		int break_pc_value_ = 0, break_mem_value_ = 0,
			break_mem_addr_ = 0, break_op_value_ = 0,
		    break_reg_value_ = 0, break_reg_num_ = 0;
		//char break_opcode_[OP_LENGS];
		string break_opcode_;
		int	roop = 0;
		int flag_display = 0;
		int asm_pc1 = 0, asm_pc2 = 0, asm_pc3 = 0, asm_pc4 = 0;

		string wb_opcode_ = "";
		int system_stack_count_ = 0;
		int stop_watch_ = 0;
		int all_clock_ = 0;
		int sim_command = 0;
		bool stop_watch_flag = false;
		bool input_data_ = false;
		int input_x, input_y, input_size;
		string command, opcode;
		int c_address, c_value;
#ifdef READLINE
		char history_command[20];
		int history_num = 0;

		//initialize_readline();
		read_history("log/.history");

		/*
		completion_function("f");
		completion_function("f+");
		completion_function("exit");
		*/
#endif

		// execute simulation in interactive mode 
		while(1){

#ifndef READLINE
			cout << "Input Command : ";
			getline(cin, command);
#else
			command = readline("Input Command : ");
			//cout << command << " " << command.length() << endl;

			for(history_num = 0; history_num < command.length();
													history_num++){
				history_command[history_num] = command[history_num];
			}
			history_command[history_num] = (char)NULL;

			add_history(history_command);

#endif //READLINE
			sim_command = command_analysis_function(command, opcode,
								c_address, c_value);

			switch(sim_command){
			case F: //f+
				c_value == 0 ? roop = 1 : roop = c_value;
				break;
			case T:
				if(stop_watch_flag){
					stop_watch_flag = false;
				}
				else{
					stop_watch_flag = true;
					stop_watch_ = 0;
				}
				
				break;
			case PC: //pc

				cout << "\tIF PC : " << hex << setw(8) << 
					pe[PE_NUM].processor().print_program_counter() << 
				"\tRF PC : " << hex << setw(8) << asm_pc2 << endl;
				cout << "\tEX PC : " << hex << setw(8) << asm_pc3 <<

				"\tWB PC : " << setw(8) << asm_pc4  << endl;
						
				cout <<	"\tSystem Stack Pointer : " <<
					pe[PE_NUM].processor().register_file(
						System_Stack_Pointer) << endl;
				cout << "\tTotal Clock : " << setw(8) << dec << all_clock_ << endl;
				if(stop_watch_flag){
					cout << "\tClock       : " << setw(8) << dec << stop_watch_ << endl;
					}
				break;
			case ASM: //asm
			{
			  //  gcc ver3.0.2 didn't accept "ios::nocreate"
//  				ifstream program_file(*argv, ios::in | ios::nocreate);

//  				if(program_file.bad()){
//  					return 1;
//  				}
			    ifstream program_file(*argv, ios::in);

				if(!program_file){
					return 1;
				}

				int h;
				string s1;
				asm_pc4 /= 2;
				asm_pc4 -= (PRINT_LINE / 2);

				for(h = 0; h <= asm_pc4; h++){
					getline(program_file, s1);
				}

				if(asm_pc4 < 0){
					h = (asm_pc4 * -1);
				}
				else{
					h = 0;
				}

				int program_index;
				for(; h < PRINT_LINE; h++){
					getline(program_file, s1);
					//program_index = s1.find('#');
					//s1 = s1.substr(program_index + 2);
					cout << s1 << endl;
				}
					
				program_file.close();
				
				break;
			}
			case REGALL: //regall

				cout << "PE:  PC :   r0    r1    r2    r3" << 
					"    r4    r5    r6    r7" << endl;

				int j;
				for(int i = 0; i < PE_SIZE*PE_SIZE; i++){

					cout << setw(2) << i << ": "
					<< setw(4) <<
					pe[i].processor().print_program_counter()
					<< ":";

					for(j = 0; j < REG_SIZE; j++){
						cout << setw(5) << hex << 
						pe[i].processor().register_file(j) << " ";
					}
					cout << endl;
				}
				break;
			case REG: //reg
				//reg
				cout << "Program Counter : " << 
					pe[PE_NUM].processor().print_program_counter()
					<< endl;
					for(int j = 0; j < REG_SIZE; j++){
						cout << setw(4) << hex << 
						pe[PE_NUM].processor().register_file(j) << " ";
					}
					cout << endl;
				break;
			case EXIT:
			case QUIT:
				// end simulation
				pe[PE_NUM].processor().local_memory -> dump(mem_dump2);
				#ifdef READLINE
				write_history("log/.history");
				#endif
				return 0;
				break;
			case SYSTEM_STACK: //System Stack
			{
					cout << "System Stack Pointer : " << hex << 
					setw(4) << setfill('0') <<
					pe[PE_NUM].processor().register_file(
						System_Stack_Pointer) << endl;
					cout << setfill(' ');
				int addr = pe[PE_NUM].processor().register_file(
					System_Stack_Pointer);
					addr = addr - (addr % 0x10);

				for(;addr < System_Stack_Initial_Address; addr += 0x10){
					cout << setfill('0') << setw(4) << hex
					 << addr << ": ";

					for(int k = 0; k < 16; k += 2){
					cout << setw(4) << hex << 
				pe[PE_NUM].processor().local_memory -> read(addr + k)
				<< " ";
					}
					cout << endl;
				}
					cout << setfill(' ');
			}
				break;
			case DISPLAY:
				if(display_flag){
					display_flag = false;
					//wb_opcode_ = "";
				}
				else{
					display_flag = true;
				}
				break;
			case MEM:
			{
				//mem@xxxx

					c_address = c_address - (c_address % 0x10);

					cout << setfill('0') << setw(4) << hex
					 << c_address << ": ";

					for(k = 0; k < 16; k += 2){
					cout << setfill('0') << setw(4) << hex << 
				pe[PE_NUM].processor().local_memory -> read(c_address + k)
				<< " ";
					}

					cout << setfill(' ');
					cout << endl;
			}
				break;
			case SET_MEMORY:
				//set mem@xxx=xxxx
					
				break_mem_ = true;
				roop = 10;

				break;
			case SET_PC:

				break_pc_ = true;
				roop = 10;
						
				break;
			case SET_REGISTER:	//register
				break_reg_ = true;
				roop = 10;

				break;
			case SET_OPCODE: //opcode
			{
				break_op_ = true;
				roop = 1;
				//roop = value;

				int i = 0;
				break_opcode_ = opcode;
				
				for(; i < break_opcode_.length(); i++){
					break_opcode_[i] = toupper(break_opcode_[i]);
				}

			}
				break;
			case CAPTURE:
			{
				roop = 0;
				vdec.capture();
			}
				break;
			case INPUT:
			{
				roop = 1;
				//int i, j, k;
				input_data_ = true;
				//input_x = input_y = input_size = 0;
				input_size = 0;

#if 0
				for (k = 0; k < VAOU_SIZE * VAOU_SIZE; k++) {
					for(j = 0; j < PE_SIZE; j++){
						for(i = 0; i < PE_SIZE; i++){
							int num = j * PE_SIZE + i;

							Input_pkt[num] = new vision_packet<D>;
							((vision_packet<D>*)Input_pkt[num]) -> data() = (D)vdec.get(i, j, 0);
							//((vision_packet<D>*)Input_pkt[num]) -> data() = (D)vdec.get(i, j, 1);
							Input_Unit0[num].put(Input_pkt[num]);
						}
					}

					for(i = 0; i < PE_SIZE * PE_SIZE; i++){
						Input_pkt[i] = new vision_packet<D>;
						((vision_packet<D>*)Input_pkt[i]) -> data() = i*2;
						Input_Unit1[i].put(Input_pkt[i]);
					}
					vdec.next();
				}
#endif
			}
				break;
			case SP:
				if(sp_dump_flag){
					sp_dump_flag = false;
				}
				else{
					sp_dump_flag = true;
				}
				break;
			case TB:
				if(tb_dump_flag){
					tb_dump_flag = false;
				}
				else{
					tb_dump_flag = true;
				}
				break;
			case BB:
				if(bb_dump_flag){
					bb_dump_flag = false;
				}
				else{
					bb_dump_flag = true;
				}
				break;
				
			default:
				break;
			}


			// loop
			while(roop > 0){
				roop--;

			if(stop_watch_flag){
				stop_watch_++;
			}

			// send data to before input unit
			if(input_data_ == true){

				if(input_size < VAOU_SIZE * VAOU_SIZE){
					input_size++;
					cout << "Send Input Data " << input_size << endl;
#if 1
					for(input_x = 0; input_x < PE_SIZE; input_x++){
						for(input_y = 0; input_y < PE_SIZE; input_y++){
							int num = input_x * PE_SIZE + input_y;

							pe[num].processor().cont_unit0->front_input().put
							  ((D)vdec.get(input_y, input_x, 0));
						}
					}
#else
					for(input_x = 0; input_x < PE_SIZE; input_x++){
						for(input_y = 0; input_y < PE_SIZE; input_y++){
							int num = input_x * PE_SIZE + input_y;

							Input_pkt[num] = new vision_packet<D>;
							((vision_packet<D>*)Input_pkt[num]) -> data() = (D)vdec.get(input_y, input_x, 0);
							Input_Unit0[num].put(Input_pkt[num]);
						}
					}
#endif

					vdec.next();
				}
				else{
					input_data_ = false;
				}

			}
				
			all_clock_++;
			asm_pc4 = asm_pc3;
			asm_pc3 = asm_pc2;
			asm_pc2 = asm_pc1;

			if(pe[PE_NUM].processor().print_program_counter() != 
				asm_pc1 + 2){
				//asm_pc2 = asm_pc3;
				asm_pc2 = -2;
			}

			asm_pc1 = pe[PE_NUM].processor().print_program_counter();
				ssp_dump << "System Stack Pointer : " << hex << 
					setw(4) << setfill('0') <<
					pe[PE_NUM].processor().register_file(
						System_Stack_Pointer) << endl;
					ssp_dump << setfill(' ');
//		cout << setw(4) << asm_pc1 << asm_pc2 << asm_pc3 << asm_pc4 << endl;
			//	mon.clock_in();
			//	mon.clock_out();

			if(display_flag){

				cout << "\tWB PC : " << setw(8) << hex << asm_pc4 //<< 
				//"\t\t " << pe[PE_NUM].processor().opcode_string(4) 
				<< "\t\t " << wb_opcode_ << endl;

				cout << "\tEX PC : " << hex << setw(8) << asm_pc3 <<
				"\t\t " << pe[PE_NUM].processor().opcode_string(4) 
				<< endl;

				cout << "\tRF PC : " << hex << setw(8) << asm_pc2 <<
				"\t\t " << pe[PE_NUM].processor().opcode_string(3) 
				<< endl;

				cout << "\tIF PC : " << hex << setw(8) << 
					pe[PE_NUM].processor().print_program_counter() << 
				"\t\t " << pe[PE_NUM].processor().opcode_string(2) 
				<< endl;

				cout <<	"\tSystem Stack Pointer : " <<
					pe[PE_NUM].processor().register_file(
						System_Stack_Pointer) ;

					cout <<"\tFlag : " ; 

					switch(flag_display){
					case 1:
						cout << "ZERO FLAG";
						break;
					case 10:	
						cout << "NEGATIVE FLAG";
						break;
					default:
						break;
				}
				cout << endl;


				cout << "\tTotal Clock : " << setw(8) << dec << all_clock_ << endl;
				if(stop_watch_flag){
					cout << "\tClock       : " << setw(8) << dec << stop_watch_ << endl;
					}

				cout << endl;

				flag_display = pe[PE_NUM].processor().register_file(33);


			}
				wb_opcode_ = pe[PE_NUM].processor().opcode_string(4);

if(system_stack_trace_flag_)
{

	if(asm_pc3 >= 0){
	system_stack_count_ = wb_opcode_.find("PUSHS");

	if(system_stack_count_ >= 0){
		pushs_count++;
		sst_dump << "PC : " << setw(4) << hex << asm_pc3 << " ";
		sst_dump << "PUSHS : " << setw(4) << pushs_count << " ";
		//ksst_dump << "PUSHS : " << setw(4) << pushs_count << endl;
		sst_dump << "System Stack Pointer : " << hex << 
		setw(4) << setfill('0') <<
		pe[PE_NUM].processor().register_file(
		System_Stack_Pointer) << endl;
		sst_dump << setfill(' ');
	}
	else{
		system_stack_count_ = wb_opcode_.find("POPS");

		if(system_stack_count_ >= 0){
			pushs_count--;
			sst_dump << "PC : " << setw(4) << hex << asm_pc3 << " ";
			sst_dump << "POPS  : " << setw(4) << pushs_count  << " ";
			sst_dump << "System Stack Pointer : " << hex << 
			setw(4) << setfill('0') <<
			pe[PE_NUM].processor().register_file(
			System_Stack_Pointer) << endl;
			sst_dump << setfill(' ');
		}
		else{
			system_stack_count_ = wb_opcode_.find("CALL");

			if(system_stack_count_ >= 0){
				pushs_count++;
				sst_dump << "PC : " << setw(4) << hex << asm_pc3 << " ";
				sst_dump << "CALL  : " << setw(4) << pushs_count << " ";
				sst_dump << "System Stack Pointer : " << hex << 
				setw(4) << setfill('0') <<
				pe[PE_NUM].processor().register_file(
				System_Stack_Pointer) << endl;
				sst_dump << setfill(' ');
			}
			else{
				system_stack_count_ = wb_opcode_.find("RTN");

				if(system_stack_count_ >= 0){
					pushs_count--;
					sst_dump << "PC : " << setw(4) << hex << asm_pc3 << " ";
					sst_dump << "RTN   : " << setw(4) << pushs_count << " ";
					sst_dump << "System Stack Pointer : " << hex << 
					setw(4) << setfill('0') <<
					pe[PE_NUM].processor().register_file(
					System_Stack_Pointer) << endl;
					sst_dump << setfill(' ');
				}
			}
		}
	}
	}
}

			if(memory_access_flag){
				switch(pe[PE_NUM].processor().print_memory_access()){
				case 1:
					if(display_flag){
					cout << setw(4) << hex << 
					pe[PE_NUM].processor().print_program_counter() << 
					": READ MEMORY  : ADDRESS ";
					cout << setfill('0') << setw(4) << 
					pe[PE_NUM].processor().print_memory_address() <<
					" : DATA " << setw(4) << 
					pe[PE_NUM].processor().local_memory ->
					read(pe[PE_NUM].processor().print_memory_address())
					<< setfill(' ') << endl; 
					}
					memory_trace << setw(4) << hex << 
					pe[PE_NUM].processor().print_program_counter() << 
					": READ MEMORY  : ADDRESS ";
					memory_trace << setfill('0') << setw(4) << 
					pe[PE_NUM].processor().print_memory_address() <<
					" : DATA " << setw(4) << 
					pe[PE_NUM].processor().local_memory ->
					read(pe[PE_NUM].processor().print_memory_address())
					<< setfill(' ') << endl; 
					break;
				case 2:
					if(display_flag){
					cout << setw(4) << hex << 
					pe[PE_NUM].processor().print_program_counter() << 
					": WRITE MEMORY : ADDRESS ";
					cout << setfill('0') << setw(4) << 
					pe[PE_NUM].processor().print_memory_address() <<
					" : DATA " << setw(4) << 
					pe[PE_NUM].processor().local_memory ->
					read(pe[PE_NUM].processor().print_memory_address())
					<< setfill(' ') << endl; 
					}
					memory_trace << setw(4) << hex << 
					pe[PE_NUM].processor().print_program_counter() << 
					": WRITE MEMORY : ADDRESS ";
					memory_trace << setfill('0') << setw(4) << 
					pe[PE_NUM].processor().print_memory_address() <<
					" : DATA " << setw(4) << 
					pe[PE_NUM].processor().local_memory ->
					read(pe[PE_NUM].processor().print_memory_address())
					<< setfill(' ') << endl; 
					break;

				default:
					break;

				}

			}
			
					if(display_flag){
						cout << "Num :     0     1     2     3     "
						 << "4     5     6     7" << endl;
		//				 cout << "---+-----+-----+-----+-----+-----+"
		//				  << endl;
					}
				if(register_dump_flag){

					if(display_flag){
						cout << "Reg : ";

						for(i = 0; i < REG_SIZE; i++){
							cout << setw(5) << hex << 
							pe[PE_NUM].processor().register_file(i) 
							<< " ";
						}
						cout << endl;
					}

					if(hunga_flag){

						reg_dump << "\t" <<  setw(4) << hex << 
						setfill('0') << 
						pe[PE_NUM].processor().print_program_counter() ;

						for(i = 0; i < REG_SIZE; i++){

							reg_dump << "\t" << setw(4) << hex <<
							setfill('0') << 
							pe[PE_NUM].processor().register_file(i) ;
						}
						reg_dump << endl;
					}
					else{
					reg_dump << setw(8) << hex << 
						pe[PE_NUM].processor().print_program_counter() 
						<< ": ";

					for(i = 0; i < REG_SIZE; i++){

						reg_dump << setw(5) << hex << 
							pe[PE_NUM].processor().register_file(i)
							<< " ";
					}
					reg_dump << endl;
					}
				}
			
				if(sp_dump_flag){
					if(display_flag){
						cout << "Sp  : ";
						for(i = 0; i < REG_SIZE; i++){
							cout  << setw(5) << hex << 
							pe[PE_NUM].processor().register_file(i + SP_REG) << " ";
						}
						cout << endl;

					}
					if(hunga_flag){
						sp_dump << "\t" << setw(4) << hex << 
							setfill('0') << 
						pe[PE_NUM].processor().print_program_counter() ;
						for(i = 0; i < REG_SIZE; i++){
							sp_dump << "\t" << setw(4) << hex <<
							setfill('0') << 
							pe[PE_NUM].processor().register_file(
									i + SP_REG);
						}
						sp_dump << endl;
					}
					else{
						sp_dump << setw(8) << hex << 
						pe[PE_NUM].processor().print_program_counter()
						<< ": ";

						for(i = 0; i < REG_SIZE; i++){
							sp_dump << setw(5) << hex << 
								pe[PE_NUM].processor().register_file(i + SP_REG) << " ";
						}
						sp_dump << endl;
					}
				}
			
				if(tb_dump_flag){
					if(display_flag){
						cout << "Tb  : ";
						tb_dump << setw(8) << hex << 
						pe[PE_NUM].processor().print_program_counter()  << ": ";

						for(i = 0; i < REG_SIZE; i++){
							cout << setw(5) << hex <<
							pe[PE_NUM].processor().register_file(i + TB_REG) << " ";
							tb_dump  << setw(5) << hex <<
							pe[PE_NUM].processor().register_file(i + TB_REG) << " ";
						}
						cout << endl;
						tb_dump << endl;
					}

					if(hunga_flag){
						tb_dump << "\t" << setw(4) << hex <<
						setfill('0') << 
						pe[PE_NUM].processor().print_program_counter();

						for(i = 0; i < REG_SIZE; i++){
							tb_dump  << "\t" << setw(4) << hex <<
							setfill('0') << 
							pe[PE_NUM].processor().register_file(
									i + TB_REG);
						}
						tb_dump << endl;
					} else {
						tb_dump << setw(8) << hex << 
						pe[PE_NUM].processor().print_program_counter() 
						<< ": ";

					for(i = 0; i < REG_SIZE; i++){
						tb_dump  << setw(5) << hex <<
						pe[PE_NUM].processor().register_file(i + TB_REG) << " ";
					}
					tb_dump << endl;
					}
				}

				if(bb_dump_flag){	
					if(display_flag){
						bb_dump << setw(8) << hex << 
						pe[PE_NUM].processor().print_program_counter()  << ": ";
						cout << "Bb  : ";
						for(i = 0; i < REG_SIZE; i++){
							cout << setw(5) << hex << 
							pe[PE_NUM].processor().register_file(i + BB_REG) << " ";
							bb_dump << setw(5) << hex << 
							pe[PE_NUM].processor().register_file(i + BB_REG) << " ";
						}
						cout << endl;
						bb_dump << endl;
					}

					if(hunga_flag){	
						bb_dump << "\t" << setw(4) << hex <<
						setfill('0') << 
						pe[PE_NUM].processor().print_program_counter();

						for(i = 0; i < REG_SIZE; i++){
							bb_dump << setw(4) << hex <<
							setfill('0') << 
							pe[PE_NUM].processor().register_file(
										i + BB_REG);
						}
						bb_dump << endl;
					}
					else{
						bb_dump << setw(8) << hex << 
						pe[PE_NUM].processor().print_program_counter() 
						<< ": ";
						for(i = 0; i < REG_SIZE; i++){
							bb_dump << setw(5) << hex << 
							pe[PE_NUM].processor().register_file(i + BB_REG) << " ";
						}
						bb_dump << endl;
					}
				}

				if(trans_buffer_dump_flag){
					trans_buffer_dump << setw(8) << hex <<
					asm_pc4
					<< ": " <<  setw(5) << hex << 

					pe[PE_NUM].processor().receive_data0().left_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().left_status
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().right_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().right_status
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().up_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().up_status
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().down_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data0().down_status
					<< endl; 

					trans_buffer_dump << setw(8) << hex <<
					asm_pc4
					<< ": " <<  setw(5) << hex << 

					pe[PE_NUM].processor().receive_data1().left_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().left_status
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().right_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().right_status
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().up_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().up_status
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().down_data
					<< " " << setw(5) << hex <<
					pe[PE_NUM].processor().receive_data1().down_status
					<< endl;

				}

					if(display_flag){
						cout << endl;
					}
			if(break_pc_){
				roop++;
				if(c_value == asm_pc3){
					break_pc_ = false;
					roop = 0;
				}

			}

			if(break_mem_){
				roop++;
				if(c_value ==
					pe[PE_NUM].processor().local_memory -> 
					read(c_address)){

					break_mem_ = false;
					roop = 0;
				}

			}

			if(break_reg_){
				roop++;
				if(c_value ==
					pe[PE_NUM].processor().register_file(c_address)){

					break_reg_ = false;
					roop = 0;
				}
			}

			if(break_op_){ //opcode
				roop++;

				int k = wb_opcode_.find(break_opcode_);

				if(k >= 0){
					if(wb_opcode_[break_opcode_.length()] == ' '){
					
						if(k >= 0 && asm_pc3 >= 0){
							roop--;
							if(roop == 0){
								break_op_ = false;
							}
					}
					}
				}

			}

				for(i = 0; i < PE_SIZE * PE_SIZE; i++){
					bus_if.send_single_write_request(
						pe[i].processor().program_counter());
				}

				for(i = 0; i < PE_SIZE * PE_SIZE; i++){
					pe[i].clock_in();
					pe[i].clock_out();
				}
				#ifdef INPUT
				if(input_flag){
					if(Result_Unit.have_packet()){
						cout << "receive packet " << endl;
						//Input Data
					for(i = 0; i < PE_SIZE * PE_SIZE; i++){
						Input_pkt[i] = new vision_packet<D>;
						((vision_packet<D>*)Input_pkt[i]) -> data() = i;
						Input_Unit0[i].put(Input_pkt[i]);
					}

					for(i = 0; i < PE_SIZE * PE_SIZE; i++){
						Input_pkt[i] = new vision_packet<D>;
						((vision_packet<D>*)Input_pkt[i]) -> data() = i*2;
						Input_Unit1[i].put(Input_pkt[i]);
					}
					}
				}
				#endif //INPUT

				mon.clock_in();
				mon.clock_out();

		if(mon.is_halt()){
			ofstream mem_dump2("log/mem_dump2", ios::out | ios::trunc);
			pe[PE_NUM].processor().local_memory -> dump(mem_dump2);
			#ifdef READLINE
			write_history("log/.history");
			#endif
			exit(1);
			break;

		}
#ifdef DEBUG
		cout << roop << " " << break_op_ << " " <<  break_pc_ << " " << break_mem_ <<" " <<   break_reg_ << endl;

#endif
			}
		
		}
	} 
	// not reached
}

template <class D>
void input_output_connect_function(D& a)
{
	a.processor().cont_unit0->before_input().connect(a.processor().cont_unit0->result_output());

	a.processor().cont_unit1->before_input().connect(a.processor().cont_unit1->result_output());

}

template <class D>
void left_right_connect_function(D& a, D& b)
{
	b.processor().cont_unit0->left_data_input().connect(a.processor().cont_unit1->right_data_output());
	b.processor().cont_unit0->left_status_input().connect(a.processor().cont_unit1->right_status_output());
	b.processor().cont_unit0->left_input_send().connect(a.processor().cont_unit1->right_output_send());
	b.processor().cont_unit0->left_output_receive().connect(a.processor().cont_unit1->right_input_receive());


	b.processor().cont_unit0->left_data_output().connect(a.processor().cont_unit1->right_data_input());
	b.processor().cont_unit0->left_status_output().connect(a.processor().cont_unit1->right_status_input());

	b.processor().cont_unit0->left_output_send().connect(a.processor().cont_unit1->right_input_send());
	b.processor().cont_unit0->left_input_receive().connect(a.processor().cont_unit1->right_output_receive());
}


template <class D>
void up_down_connect_function(D& a, D& b)
{
	b.processor().cont_unit1->up_data_output().connect(a.processor().cont_unit0->down_data_input());
	b.processor().cont_unit1->up_status_output().connect(a.processor().cont_unit0->down_status_input());
	b.processor().cont_unit1->up_output_send().connect(a.processor().cont_unit0->down_input_send());
	b.processor().cont_unit1->up_input_receive().connect(a.processor().cont_unit0->down_output_receive());


	b.processor().cont_unit1->up_data_input().connect(a.processor().cont_unit0->down_data_output());
	b.processor().cont_unit1->up_status_input().connect(a.processor().cont_unit0->down_status_output());

	b.processor().cont_unit1->up_input_send().connect(a.processor().cont_unit0->down_output_send());
	b.processor().cont_unit1->up_output_receive().connect(a.processor().cont_unit0->down_input_receive());
}

int command_analysis_function(string& command, string& opcode, 
						int& address, int& value)
{

	int space = 0, plus = 0, atmark = 0, equal = 0;
	int k = 0;

	space = command.find_first_of(" ");
	plus = command.find_first_of("+");
	atmark = command.find_first_of("@");
	equal = command.find_first_of("=");
	value = 0;
	address = 0;

	while(isspace(command[k])){
		k++;
	}

	if(plus != command.npos){
		space = plus;
	}

	opcode = command.substr(k, space);

	if(plus != command.npos){
		//forward
		k = plus + 1;
		while(isspace(command[k])){
			k++;
		}

		value = string_to_num10(command.substr(k, command.length()));

	}
	else if(equal != command.npos){
		
		if(atmark != command.npos){
			//set mem@xxxx=yyyy

			k = atmark + 1;
			while(isspace(command[k])){
				k++;
			}

			address = string_to_num16(command.substr(k, equal));

			k = equal + 1;
			while(isspace(command[k])){
				k++;
			}

			value = string_to_num16(command.substr(k, command.length()));

		}
		else{
			//set opcode=xxxx
			//set pc=xxxx
			//set rx=yyyy
			//set 

			k = equal + 1;
			while(isspace(command[k])){
				k++;
			}

			value = string_to_num16(command.substr(k, command.length()));

		}

	}
	else if(atmark != command.npos){
		//mem@xxxx
		k = atmark + 1;
		while(isspace(command[k])){
			k++;
		}

		address = string_to_num16(command.substr(k, command.length()));
	}
	
	//cout << command << ": " << opcode << " ";
	//cout << address << " " << value << endl;

	//decide opcode
	if(opcode == "f" || opcode == "forward"){
		return F;
	}

	if(opcode == "set"){
		while(isspace(command[space])){
			space++;
		}

		switch(command[space]){
		case 'p':
			//cout << "PC" << " " << hex << value << endl;
			return SET_PC;
		case 'm':
		//	cout << "MEM" << endl;
			return SET_MEMORY;
		case 'o':
			equal++;
			while(isspace(command[equal])){
				equal++;
			}
			opcode = command.substr(equal, command.length());
			#ifdef DEBUG
			printf("%s\n", opcode.c_str());
			cout << "OP : " << opcode << " " << hex << value << endl;
			int hoge;
			cin >> hoge;
			#endif
			return SET_OPCODE;
		case 'r':
			address = command[++space] - '0';
		//	cout << "REG" << address << endl;
			return SET_REGISTER;
		default:
			break;
			
		}
		return SET;
	}

	if(opcode == "system" || opcode == "h"){
		return SYSTEM_STACK;
	}

	if(opcode == "mem"){
		return MEM;
	}

	if(opcode == "pc"){
		return PC;
	}

	if(opcode == "reg"){
		return REG;
	}

	if(opcode == "regall"){
		return REGALL;
	}

	if(opcode == "asm"){
		return ASM;
	}

	if(opcode == "d"){
		return DISPLAY;
	}

	if(opcode == "t"){
		return T;
	}

	if(opcode == "exit" || opcode == "e"){
		return EXIT;
	}

	if(opcode == "quit" || opcode == "q"){
		return QUIT;
	}

	if(opcode == "capture"){
		return CAPTURE;
	}

	if(opcode == "input"){
		return INPUT;
	}

	if(opcode == "sp"){
		return SP;
	}

	if(opcode == "tb"){
		return TB;
	}

	if(opcode == "bb"){
		return BB;
	}
}

int string_to_num16(string value)
{
	int k = 0, num = 0;

	//while(isalnum(value[k])){
	while(k < value.length()){
		if(isdigit(value[k])){
			num = (value[k] - '0') 
				+ num * 16;
			k++;
		}
		else{
			num = ((value[k] - 'a' + 10))
				+ num * 16;
			k++;
		}
	}

	return num;
}

int string_to_num10(string value)
{
	int k = 0, num = 0;
	//while(isdigit(value[k])){
	while(k < value.length()){
		num = (value[k] -'0') + num * 10;
			k++;
	}

	return num;
}

/*
void initialize_readline(void)
{

	rl_readline_name = "vision_test";

	rl_attempted_completion_function = 
}
*/
