/* -*- C++ -*-
 *
 * <<< pico_pipeline_stage.h >>>
 *
 * --- pico pipeline state class 'pico_pipeline_stage'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef PICO_PIPELINE_STAGE_H
#define PICO_PIPELINE_STAGE_H 1

#include <string>
#include "pico_typedef.h"
#include "pico_instruction.h"
#include "pico_register_file.h"


class pico_pipeline_stage
{
public:
	typedef pico_word address_type;
	typedef pico_word data_type;
	typedef signed_pico_word signed_data_type;
	typedef unsigned_pico_word unsigned_data_type;
	static const size_t sizeof_data_type = sizeof_pico_word;
	
private:
	typedef pico_pipeline_stage thisclass;
	struct register_set{
		data_type value;
		int number;
		inline register_set(void);
		inline register_set(data_type, int);
		inline void clear(void);
	};	
	enum io_flag_value{
		int_in =1,
		int_out = 2,
		cp_in = 4,
		cp_out = 8,
		dst_out =16
	};

	pico_instruction buf;
	register_set src1, src2, dst;
	address_type acc_adr;
	data_type acc_dat;
	address_type dst_pc;
	int reg_in, io_flags;
	//register_set zero_flag, negative_flag;

	static const int FLAG_REGISTER = 33;
	static const int PREFIX_REGISTER = 34;
	static const int SYSTEM_STACK = 35;
	static const int ZERO_FLAG = 1;
	static const int NEGATIVE_FLAG = 10;
	bool change_flag;
	bool branch_flag_;
public:
	pico_pipeline_stage(void);
	pico_pipeline_stage(const thisclass &);
	~pico_pipeline_stage();
	pico_instruction::operation_type operation(void) const
		{ return buf.operation(); }
	pico_instruction::opcode_type opcode(void) const
		{ return buf.opcode(); }
	bool is_i_type() const { return buf.is_i_type(); }
	

	bool is_destination_output(void) const { return io_flags & dst_out; }
	data_type image(void) const { return buf.image(); }
	int source1_number(void) const { return src1.number; }
	int source2_number(void) const { return src2.number; }
	data_type source1(void) const { return src1.value; }
	data_type source2(void) const { return src2.value; }
	
	address_type destination_pc(void) const { return dst_pc; }
	int destination_number(void) const {return dst.number; }
	data_type destination(void) const { return dst.value; }
	data_type access_data(void) const { return acc_dat; }
	address_type access_address(void) const { return acc_adr; }
	address_type word_aligned_access_address(void) const 
		{ return access_address() & ~pico_word(sizeof_data_type - 1); }

	std::string opcode_string(void) const { return buf.opcode_string(); }
    std::string operation_string(void) const { return buf.operation_string(); }

	bool is_branch(void) const { return branch_flag_; }
	void instruction_fetch(address_type, data_type);
	void register_fetch(const pico_register_file&);
	void execute(pico_register_file&);
	void mul_store(data_type, data_type);
	void load_fetch_for_big_endian(data_type);
	void load_fetch_for_little_endian(data_type);
	void partial_word_store_fetch_for_big_endian(data_type);
	void partial_word_store_fetch_for_little_endian(data_type);
	void writeback(pico_register_file&);
	inline void forwarding(const pico_pipeline_stage&);
	void clear(void);
	void output(std::ostream& os) const;

};

std::ostream& operator<<(std::ostream&, const pico_pipeline_stage&);

inline void pico_pipeline_stage::forwarding(const pico_pipeline_stage& st)
{
	if(!st.is_destination_output()){
		return;
	}
	const int dst_number = st.destination_number();

	switch(reg_in){
	case 1:
		if(source1_number() == dst_number){
			src1.value = st.destination();
		}
		break;
	case 2:
		if(source1_number() == dst_number){
			src1.value = st.destination();
		}
		if(source2_number() == dst_number){
			src2.value = st.destination();
		}
		break;
	}

}

#endif /* PICO_PIPELINE_STAGE_H */
