/* -*- C++ -*-
 *
 * <<< pico_instruction.h >>>
 *
 * --- pico instruction class 'r3000_instruction'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef PICO_INSTRUCTION_H
#define PICO_INSTRUCTION_H 1

#include <string>
#include "pico_typedef.h"


class pico_instruction
{
public:
	enum operation_type{ I_TYPE, J_TYPE, R_TYPE, S_TYPE };
	/* same place with pico_instruction.cc enum const char * name[] */
	enum opcode_type{
		NOP, ADD, ADDI, SUB, SUBI, AND, ANDI, OR, ORI,
		SL, SR, LD, LDLI, LDHI, ST, MV, BEQZ, BNEZ,
		UNKNOWN, RESERVED, XORI, JALR, BMI, BPL, JAL,
		JR, JMP, R_TYPE_OP, XOR, NOT, LB, SB, RFI, EINT,
		LIAR, SIAR, DINT, MUL, DIV, MULFI, MULFF, HALT,
		PUSH, POP, MVR2S, MVS2R, MVR2T, MVT2R, MVR2B,
		MVB2R, MVT2S, MVB2S, NEXT, LOOK, S_TYPE_OP,
		CMP, CMPI, PFX, CALL, RTN, PUSHS, POPS, MULI,
		MULFFI, MULFII, SSP2REG, REG2SSP, 
	};
	typedef pico_word data_type;
	typedef pico_word address_type;
	typedef signed_pico_word signed_data_type;
	typedef unsigned_pico_word unsigned_data_pico_type;
	static const size_t sizeof_data_type = sizeof_pico_word;
private:
	typedef pico_instruction thisclass;
	data_type image_;
	address_type pc_;
	opcode_type opcode_;
	operation_type operation_;

public:
	pico_instruction(void);
	pico_instruction(const thisclass&);
	~pico_instruction();

	void output(std::ostream&) const;
	data_type image(void) const { return image_; }
	address_type pc(void) const { return pc_; }

	operation_type operation(void) const { return operation_; };
	opcode_type opcode(void) const { return opcode_; };

	int rs(void) const { return (image() >> 5 ) & 0x07; }
	int rd(void) const { return (image() >> 8 ) & 0x07; }
	int base(void) const { return rs(); }
	data_type immediate(void) const { return image() & 0x00ffUL; }
	data_type long_immediate(void) const { return image() & 0x007ffUL; }

	signed_data_type signed_immediate(void) const {
		return (image() & 0x80UL) ?
			(signed_data_type(immediate()) - 0x100UL) :
			(signed_data_type(immediate()));
	}
	signed_data_type signed_long_immediate(void) const {
		return (image() & 0x400UL) ?
			(signed_data_type((long_immediate() - 0x800UL) << 1)):
			(signed_data_type(long_immediate() << 1));
	}
	bool is_i_type(void) const { return operation() == I_TYPE; }
	bool is_j_type(void) const { return operation() == J_TYPE; }
	bool is_r_type(void) const { return operation() == R_TYPE; }
	bool is_s_type(void) const { return operation() == S_TYPE; }

	address_type branch_address(void) const{
		return address_type((pc() + sizeof_data_type) + 
			(signed_immediate() << 1) );
	}
  
    std::string opcode_string(void) const;
    std::string operation_string(void) const;
	void set(address_type, data_type);
	void clear(void) { set(0, 0); }
};

std::ostream& operator<<(std::ostream&, const pico_instruction&);

#endif /* PICO_INSTRUCTION_H */
