/*
 * <<< pico.cc >>>
 *
 * --- pico class 'pico'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <string>  // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include "pico.h"
#include "pico_integer_unit.h"
#include "pico_timer_unit.h"

using namespace std;

pico::pico(void)
	: bus_if(new pico_bus_interface),
      local_memory(new mapped_memory<word>(0, LOCAL_MEMORY_SIZE)),
	  cont_unit0(new vision_control_unit<word>()),
	  cont_unit1(new vision_control_unit<word>()),
	  ti_unit(new pico_timer_unit(*local_memory)),
	  int_unit(new pico_integer_unit(*local_memory, *cont_unit0, *cont_unit1, *ti_unit)),
	  halt_flag_(false),
	  freq_mag_(1)
{

}

pico::pico(const pico& a)
	: bus_if(new pico_bus_interface(*(a.bus_if))),
      local_memory(new mapped_memory<word>(0, LOCAL_MEMORY_SIZE)),
	  cont_unit0(new vision_control_unit<word>()),
	  cont_unit1(new vision_control_unit<word>()),
	  ti_unit(new pico_timer_unit(*local_memory)),
	  int_unit(new pico_integer_unit(*(a.int_unit), *local_memory, *cont_unit0, *cont_unit1, *ti_unit)),
	  halt_flag_(a.halt_flag_),
	  freq_mag_(a.freq_mag_)
{

}

pico::~pico()
{
	delete int_unit;
	delete local_memory;
	delete ti_unit;
	delete bus_if;
	delete cont_unit0;
	delete cont_unit1;
}

pico::data_type pico::register_file(size_t i) const
{
	return int_unit->register_file(i);
}

pico::data_type& pico::register_file(size_t i)
{
	return int_unit->register_file(i);
}

pico::address_type pico::program_counter(void) const
{
	return int_unit->pc();
}

pico::address_type& pico::program_counter(void)
{
	return int_unit->pc();
}

pico::address_type pico::print_program_counter(void) const
{
	return int_unit -> print_pc();
}

pico::address_type& pico::print_program_counter(void)
{
	return int_unit -> print_pc();
}

pico::address_type pico::print_memory_address(void)
{
	return int_unit -> memory_address();
}

pico::address_type pico::print_memory_access(void)
{
	return int_unit -> memory_access_mode();
}

bool pico::is_halt(void) const
{
	return halt_flag_;
}

bool pico::is_stall(void) const
{
	return false;
}


bool pico::is_reading(void) const
{
	return false;
}

bool pico::is_writing(void) const
{
	return false;
}


bool pico::is_user_mode(void) const
{
	return false;
}

bool pico::is_kernel_mode(void) const
{
	return false;
}
pico::data_type pico::hi(void) const
{
	return int_unit->hi();
}

pico::data_type pico::lo(void) const
{
	return int_unit->lo();
}

void pico::clock_in(void)
{
	int_unit->clock();
	ti_unit->clock_in();
	cont_unit0 -> clock_in();
	cont_unit1 -> clock_in();
}

void pico::clock_out(void)
{
	cont_unit0->clock_out();
	cont_unit1->clock_out();
}

void pico::reset(void)
{
	int_unit->reset();
}

void pico::halt(void)
{
	halt_flag_ = true;
}

size_t pico::frequency_magnification(void) const
{
	return freq_mag_;
}

void pico::set_frequency_magnification(size_t a)
{
	freq_mag_ = a;
}


const port& pico::port_ref(void) const
{
	return *bus_if;
}

port& pico::port_ref(void)
{
	return *bus_if;
}

void pico::output(ostream& os) const
{
	os << int_unit;
#ifdef DEBUG
	os << "\nwbu: " << wb_unit
	   //<< "\nmau: " << ma_unit
	   << "\nbus: " << bus_if;
	os.flush();
#endif // DEBUG
}

void pico::set_timer_address(address_type a)
{

	 ti_unit->timer_memory_address() = a; 
}

void pico::set_timer_count(data_type d)
{
	 ti_unit->timer_clock_count() = d; 
}




bool pico::output(ostream& os, const string& s) const
{
	if (s == "bus_if" || s == "busif") {
		os << bus_if;
		return true;
	} else if (s == "int_unit" || s == "int") {
		os << int_unit;
		return true;
	}
	return int_unit->output(os, s);
}
