/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"
`include "nrm.vh"

module nrm2(CLK, RST,
           SET, SADDR, SDIN, 
           MRE, MDOUT,
           NID, DU,
           START);
   
   input          CLK, RST;

   input [5:0]    SET;
   input [7:0]    NID;
   input          START;

   input [10:0]   SADDR;
   input [31:0]   SDIN;
   input [1:0]    DU;

   input          MRE;
   output [31:0]  MDOUT;

   // DUNIT001
   wire [5:0]     SET001;
   wire           MRE001, START001;
   wire [31:0]    MDOUT001, DIN001, DOUT001;
   wire [1:0]     IREQ001, OREQ001;
   wire           IRDY001, ORDY001;


   // DUNIT002
   wire [5:0]     SET002;
   wire           MRE002, START002;
   wire [31:0]    MDOUT002, DIN002, DOUT002;
   wire [1:0]     IREQ002, OREQ002;
   wire           IRDY002, ORDY002;

   // UPDT800
   wire [5:0]  SET800;
   wire [1:0]  IREQ800;
   wire        IRDY800;
   wire [31:0] DIN800;

   wire [1:0]  OREQ800;
   wire        ORDY800;
   wire [31:0] DOUT800;

   // DGTB700
   wire [5:0]  SET700;
   wire [1:0]  IREQ700;
   wire        IRDY700;
   wire [31:0] DIN700;

   wire [1:0]  OREQ700;
   wire        ORDY700;
   wire [31:0] DOUT700;

   // PCAL500
   wire [5:0]  SET500;
   wire [1:0]  IREQ500;
   wire        IRDY500;
   wire [31:0] DIN500;

   wire [1:0]  OREQ500;
   wire        ORDY500;
   wire [31:0] DOUT500;

   // TCAL400
   wire [5:0]  SET400;
   wire [1:0]  IREQ400;
   wire        IRDY400;
   wire [31:0] DIN400;

   wire [1:0]  OREQ400;
   wire        ORDY400;
   wire [31:0] DOUT400;

   // TMOD300
   wire [1:0]  IREQ300;
   wire        IRDY300;
   wire [31:0] DIN300;

   wire [1:0]  OREQ300;
   wire        ORDY300;
   wire [31:0] DOUT300;

   // DEMUX001
   wire [31:0] DMO001, DMO002;

   wire [31:0] DMO800, DMO700, DMO500, DMO400, DMO300;
   wire [31:0] DMO801, DMO701, DMO501, DMO401, DMO301;
   wire [31:0] DMO802, DMO702, DMO502, DMO402, DMO302;

   wire [1:0]  IREQ201, IREQ202;
   wire        IRDY201, IRDY202;
   
   // UPDT800
   assign      SET800 = (NID == 8'd80) ? SET : 6'b0;

   cunit_updt updt800(.CLK(CLK),
                      .UWE(SET800[1:0]), .UDADR(SADDR), .UDDIN(SDIN),
                      .IREQ0(IREQ800), .IRDY0(IRDY800), .DIN0(DIN800),
                      .OREQ0(OREQ800), .ORDY0(ORDY800), .DOUT0(DOUT800),
                      .RST(RST));


   // DGTB700
   assign      SET700 = (NID == 8'd70) ? SET : 6'b0;

   cunit_dgtb dgtb700(.CLK(CLK),
                      .DGWE(SET700[0]), .DGDIN(SDIN),
                      .IREQ0(IREQ700), .IRDY0(IRDY700), .DIN0(DIN700),
                      .OREQ0(OREQ700), .ORDY0(ORDY700), .DOUT0(DOUT700),
                      .RST(RST));

   // PCAL500
   assign      SET500 = (NID == 8'd50) ? SET : 6'b0;

   cunit_pcal pcal500(.CLK(CLK),
                      .RCWE(SET500[0]), .RCADR(SADDR[9:0]), .RCDIN(SDIN),
                      .IREQ0(IREQ500), .IRDY0(IRDY500), .DIN0(DIN500),
                      .OREQ0(OREQ500), .ORDY0(ORDY500), .DOUT0(DOUT500),
                      .RST(RST));

   // TCAL400
   assign      SET400 = (NID == 8'd40) ? SET : 6'b0;

   cunit_tcal tcal400(.CLK(CLK),
                     .SET(SET400[0]), .RNDIN(SDIN),
                      .IREQ0(IREQ400), .IRDY0(IRDY400), .DIN0(DIN400),
                      .OREQ0(OREQ400), .ORDY0(ORDY400), .DOUT0(DOUT400),
                      .RST(RST));
   // TMOD300

   cunit_tmod tmod300(.CLK(CLK), 
                      .IREQ0(IREQ300), .IRDY0(IRDY300), .DIN0(DIN300),
                      .OREQ0(OREQ300), .ORDY0(ORDY300), .DOUT0(DOUT300),
                      .RST(RST));

   // ============================================================================
   wire [1:0]  OREQ801, OREQ802;
   wire        ORDY801, ORDY802;
   wire [31:0] DOUT801,  DOUT802;

   wire [1:0]  OREQ701, OREQ702;
   wire        ORDY701, ORDY702;
   wire [31:0] DOUT701,  DOUT702;

   wire [1:0]  OREQ501, OREQ502;
   wire        ORDY501, ORDY502;
   wire [31:0] DOUT501,  DOUT502;

   wire [1:0]  OREQ401, OREQ402;
   wire        ORDY401, ORDY402;
   wire [31:0] DOUT401,  DOUT402;

   wire [1:0]  OREQ301, OREQ302;
   wire        ORDY301, ORDY302;
   wire [31:0] DOUT301,  DOUT302;

   wire [1:0]  IREQ801, IREQ802;
   wire        IRDY801, IRDY802;
   wire [31:0] DIN801,  DIN802;

   wire [1:0]  IREQ701, IREQ702;
   wire        IRDY701, IRDY702;
   wire [31:0] DIN701,  DIN702;

   wire [1:0]  IREQ501, IREQ502;
   wire        IRDY501, IRDY502;
   wire [31:0] DIN501,  DIN502;

   wire [1:0]  IREQ401, IREQ402;
   wire        IRDY401, IRDY402;
   wire [31:0] DIN401,  DIN402;

   wire [1:0]  IREQ301, IREQ302;
   wire        IRDY301, IRDY302;
   wire [31:0] DIN301,  DIN302;

   bus2   bus801(.CLK(CLK),
                 .IREQ0(IREQ801), .IRDY0(IRDY801), .DIN0(DMO001),
                 .IREQ1(IREQ802), .IRDY1(IRDY802), .DIN1(DMO002),
                 .OREQ(IREQ800),  .ORDY(IRDY800),  .DOUT(DIN800),
                 .RST(RST));

   bus2   bus701(.CLK(CLK),
                 .IREQ0(IREQ701), .IRDY0(IRDY701), .DIN0(DMO001),
                 .IREQ1(IREQ702), .IRDY1(IRDY702), .DIN1(DMO002),
                 .OREQ(IREQ700),  .ORDY(IRDY700),  .DOUT(DIN700),
                 .RST(RST));

   bus2   bus501(.CLK(CLK),
                 .IREQ0(IREQ501), .IRDY0(IRDY501), .DIN0(DMO001),
                 .IREQ1(IREQ502), .IRDY1(IRDY502), .DIN1(DMO002),
                 .OREQ(IREQ500),  .ORDY(IRDY500),  .DOUT(DIN500),
                 .RST(RST));

   bus2   bus401(.CLK(CLK),
                 .IREQ0(IREQ401), .IRDY0(IRDY401), .DIN0(DMO001),
                 .IREQ1(IREQ402), .IRDY1(IRDY402), .DIN1(DMO002),
                 .OREQ(IREQ400),  .ORDY(IRDY400),  .DOUT(DIN400),
                 .RST(RST));

   bus2   bus301(.CLK(CLK),
                 .IREQ0(IREQ301), .IRDY0(IRDY301), .DIN0(DMO001),
                 .IREQ1(IREQ302), .IRDY1(IRDY302), .DIN1(DMO002),
                 .OREQ(IREQ300),  .ORDY(IRDY300),  .DOUT(DIN300),
                 .RST(RST));

   demux2 demux801(.CLK(CLK),
                   .IREQ(OREQ800), .IRDY(ORDY800), .DIN(DOUT800), .DOUT(DMO800),
                   .OREQ0(OREQ801), .ORDY0(ORDY801),
                   .OREQ1(OREQ802), .ORDY1(ORDY802),
                   .RST(RST));

   demux2 demux701(.CLK(CLK),
                   .IREQ(OREQ700), .IRDY(ORDY700), .DIN(DOUT700), .DOUT(DMO700),
                   .OREQ0(OREQ701), .ORDY0(ORDY701),
                   .OREQ1(OREQ702), .ORDY1(ORDY702),
                   .RST(RST));

   demux2 demux501(.CLK(CLK),
                   .IREQ(OREQ500), .IRDY(ORDY500), .DIN(DOUT500), .DOUT(DMO500),
                   .OREQ0(OREQ501), .ORDY0(ORDY501),
                   .OREQ1(OREQ502), .ORDY1(ORDY502),
                   .RST(RST));

   demux2 demux401(.CLK(CLK),
                   .IREQ(OREQ400), .IRDY(ORDY400), .DIN(DOUT400), .DOUT(DMO400),
                   .OREQ0(OREQ401), .ORDY0(ORDY401),
                   .OREQ1(OREQ402), .ORDY1(ORDY402),
                   .RST(RST));

   demux2 demux301(.CLK(CLK),
                   .IREQ(OREQ300), .IRDY(ORDY300), .DIN(DOUT300), .DOUT(DMO300),
                   .OREQ0(OREQ301), .ORDY0(ORDY301),
                   .OREQ1(OREQ302), .ORDY1(ORDY302),
                   .RST(RST));
   
   // ============================================================================

   demux6 demux001(.CLK(CLK),
                   .IREQ(OREQ001), .IRDY(ORDY001), .DIN(DOUT001), .DOUT(DMO001),
                   .OREQ0(IREQ801), .ORDY0(IRDY801),
                   .OREQ1(IREQ701), .ORDY1(IRDY701),
                   .OREQ2(IREQ501), .ORDY2(IRDY501),
                   .OREQ3(IREQ401), .ORDY3(IRDY401),
                   .OREQ4(IREQ301), .ORDY4(IRDY301),
                   .OREQ5(IREQ201), .ORDY5(IRDY201),
                   .RST(RST));

   bus6   bus001(.CLK(CLK),
                 .IREQ0(OREQ801), .IRDY0(ORDY801), .DIN0(DMO800),
                 .IREQ1(OREQ701), .IRDY1(ORDY701), .DIN1(DMO700),
                 .IREQ2(OREQ501), .IRDY2(ORDY501), .DIN2(DMO500),
                 .IREQ3(OREQ401), .IRDY3(ORDY401), .DIN3(DMO400),
                 .IREQ4(OREQ301), .IRDY4(ORDY301), .DIN4(DMO300),
                 .IREQ5(IREQ201), .IRDY5(IRDY201), .DIN5(DMO001),
                 .OREQ(IREQ001),  .ORDY(IRDY001),  .DOUT(DIN001),
                 .RST(RST));

   // ============================================================================

   demux6 demux002(.CLK(CLK),
                   .IREQ(OREQ002), .IRDY(ORDY002), .DIN(DOUT002), .DOUT(DMO002),
                   .OREQ0(IREQ802), .ORDY0(IRDY802),
                   .OREQ1(IREQ702), .ORDY1(IRDY702),
                   .OREQ2(IREQ502), .ORDY2(IRDY502),
                   .OREQ3(IREQ402), .ORDY3(IRDY402),
                   .OREQ4(IREQ302), .ORDY4(IRDY302),
                   .OREQ5(IREQ202), .ORDY5(IRDY202),
                   .RST(RST));

   bus6   bus002(.CLK(CLK),
                 .IREQ0(OREQ802), .IRDY0(ORDY802), .DIN0(DMO800),
                 .IREQ1(OREQ702), .IRDY1(ORDY702), .DIN1(DMO700),
                 .IREQ2(OREQ502), .IRDY2(ORDY502), .DIN2(DMO500),
                 .IREQ3(OREQ402), .IRDY3(ORDY402), .DIN3(DMO400),
                 .IREQ4(OREQ302), .IRDY4(ORDY302), .DIN4(DMO300),
                 .IREQ5(IREQ202), .IRDY5(IRDY202), .DIN5(DMO002),
                 .OREQ(IREQ002),  .ORDY(IRDY002),  .DOUT(DIN002),
                 .RST(RST));

   // ============================================================================
   
   assign         SET001   = (NID == 8'd1) ? SET : 6'b0;
   assign         MRE001   = (NID == 8'd1) ? MRE : 6'b0;
   assign         START001 = (NID == 8'd1) ? START : `DISABLE;

   dunit dunit1(.CLK(CLK),
                .SET(SET001), .SADDR(SADDR[9:0]), .SDIN(SDIN),
                .MRE(MRE), .MADDR(SADDR[9:0]), .MDOUT(MDOUT001),
                .ID(8'd1), .START(START001),
                .IREQ(IREQ001), .IRDY(IRDY001), .DIN(DIN001),
                .OREQ(OREQ001), .ORDY(ORDY001), .DOUT(DOUT001),
                .RST(RST));
   
// ============================================================================
   
   assign         SET002   = (NID == 8'd2) ? SET : 6'b0;
   assign         MRE002   = (NID == 8'd2) ? MRE : 6'b0;
   assign         START002 = (NID == 8'd2) ? START : `DISABLE;

   dunit dunit2(.CLK(CLK),
                .SET(SET002), .SADDR(SADDR[9:0]), .SDIN(SDIN),
                .MRE(MRE), .MADDR(SADDR[9:0]), .MDOUT(MDOUT002),
                .ID(8'd2), .START(START002),
                .IREQ(IREQ002), .IRDY(IRDY002), .DIN(DIN002),
                .OREQ(OREQ002), .ORDY(ORDY002), .DOUT(DOUT002),
                .RST(RST));
   
   reg [31:0]  MDOUT;

   always@(posedge CLK) begin
      if(NID == 8'd1) begin
         MDOUT <= MDOUT001;
      end
      else if(NID == 8'd2) begin
         MDOUT <= MDOUT002;
      end
   end

endmodule // nrm
