/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"
`include "nrm.vh"

module nrm1(CLK, RST,
           SET, SADDR, SDIN, 
           MRE, MDOUT,
           NID, DU,
           START);
   
   input          CLK, RST;

   input [5:0]    SET;
   input [7:0]    NID;
   input          START;

   input [10:0]   SADDR;
   input [31:0]   SDIN;
   input [1:0]    DU;

   input          MRE;
   output [31:0]  MDOUT;

   // DUNIT001
   wire [5:0]     SET001;
   wire           MRE001, START001;
   wire [31:0]    MDOUT001, DIN001, DOUT001;
   wire [1:0]     IREQ001, OREQ001;
   wire           IRDY001, ORDY001;


   // UPDT800
   wire [5:0]  SET800;
   wire [1:0]  IREQ800;
   wire        IRDY800;
   wire [31:0] DIN800;

   wire [1:0]  OREQ800;
   wire        ORDY800;
   wire [31:0] DOUT800;

   // DGTB700
   wire [5:0]  SET700;
   wire [1:0]  IREQ700;
   wire        IRDY700;
   wire [31:0] DIN700;

   wire [1:0]  OREQ700;
   wire        ORDY700;
   wire [31:0] DOUT700;

   // PCAL500
   wire [5:0]  SET500;
   wire [1:0]  IREQ500;
   wire        IRDY500;
   wire [31:0] DIN500;

   wire [1:0]  OREQ500;
   wire        ORDY500;
   wire [31:0] DOUT500;

   // TCAL400
   wire [5:0]  SET400;
   wire [1:0]  IREQ400;
   wire        IRDY400;
   wire [31:0] DIN400;

   wire [1:0]  OREQ400;
   wire        ORDY400;
   wire [31:0] DOUT400;

   // TMOD300
   wire [1:0]  IREQ300;
   wire        IRDY300;
   wire [31:0] DIN300;

   wire [1:0]  OREQ300;
   wire        ORDY300;
   wire [31:0] DOUT300;

   // DEMUX001
   wire [31:0] DMO001;
   wire [1:0]  IREQ200;
   wire        IRDY200;

   
   // UPDT800
   assign      SET800 = (NID == 8'd80) ? SET : 6'b0;
   assign      DIN800 = DMO001;

   cunit_updt updt800(.CLK(CLK),
                      .UWE(SET800[1:0]), .UDADR(SADDR), .UDDIN(SDIN),
                      .IREQ0(IREQ800), .IRDY0(IRDY800), .DIN0(DIN800),
                      .OREQ0(OREQ800), .ORDY0(ORDY800), .DOUT0(DOUT800),
                      .RST(RST));


   // DGTB700
   assign      SET700 = (NID == 8'd70) ? SET : 6'b0;
   assign      DIN700 = DMO001;

   cunit_dgtb dgtb700(.CLK(CLK),
                      .DGWE(SET700[0]), .DGDIN(SDIN),
                      .IREQ0(IREQ700), .IRDY0(IRDY700), .DIN0(DIN700),
                      .OREQ0(OREQ700), .ORDY0(ORDY700), .DOUT0(DOUT700),
                      .RST(RST));

   // PCAL500
   assign      SET500 = (NID == 8'd50) ? SET : 6'b0;
   assign      DIN500 = DMO001;

   cunit_pcal pcal500(.CLK(CLK),
                      .RCWE(SET500[0]), .RCADR(SADDR[9:0]), .RCDIN(SDIN),
                      .IREQ0(IREQ500), .IRDY0(IRDY500), .DIN0(DIN500),
                      .OREQ0(OREQ500), .ORDY0(ORDY500), .DOUT0(DOUT500),
                      .RST(RST));

   // TCAL400
   assign      SET400 = (NID == 8'd40) ? SET : 6'b0;
   assign      DIN400 = DMO001;

   cunit_tcal tcal400(.CLK(CLK),
                     .SET(SET400[0]), .RNDIN(SDIN),
                      .IREQ0(IREQ400), .IRDY0(IRDY400), .DIN0(DIN400),
                      .OREQ0(OREQ400), .ORDY0(ORDY400), .DOUT0(DOUT400),
                      .RST(RST));
   // TMOD300
   assign      DIN300 = DMO001;

   cunit_tmod tmod300(.CLK(CLK), 
                      .IREQ0(IREQ300), .IRDY0(IRDY300), .DIN0(DIN300),
                      .OREQ0(OREQ300), .ORDY0(ORDY300), .DOUT0(DOUT300),
                      .RST(RST));

   demux6 demux001(.CLK(CLK),
                   .IREQ(OREQ001), .IRDY(ORDY001), .DIN(DOUT001), .DOUT(DMO001),
                   .OREQ0(IREQ800), .ORDY0(IRDY800),
                   .OREQ1(IREQ700), .ORDY1(IRDY700),
                   .OREQ2(IREQ500), .ORDY2(IRDY500),
                   .OREQ3(IREQ400), .ORDY3(IRDY400),
                   .OREQ4(IREQ300), .ORDY4(IRDY300),
                   .OREQ5(IREQ200), .ORDY5(IRDY200),
                   .RST(RST));

   bus6   bus001(.CLK(CLK),
                 .IREQ0(OREQ800), .IRDY0(ORDY800), .DIN0(DOUT800),
                 .IREQ1(OREQ700), .IRDY1(ORDY700), .DIN1(DOUT700),
                 .IREQ2(OREQ500), .IRDY2(ORDY500), .DIN2(DOUT500),
                 .IREQ3(OREQ400), .IRDY3(ORDY400), .DIN3(DOUT400),
                 .IREQ4(OREQ300), .IRDY4(ORDY300), .DIN4(DOUT300),
                 .IREQ5(IREQ200), .IRDY5(IRDY200), .DIN5(DMO001),
                 .OREQ(IREQ001),  .ORDY(IRDY001),  .DOUT(DIN001),
                 .RST(RST));

   assign         SET001   = (NID == 8'd1) ? SET : 6'b0;
   assign         MRE001   = (NID == 8'd1) ? MRE : 6'b0;
   assign         START001 = (NID == 8'd1) ? START : `DISABLE;

   dunit dunit1(.CLK(CLK),
                  .SET(SET001), .SADDR(SADDR[9:0]), .SDIN(SDIN),
                  .MRE(MRE), .MADDR(SADDR[9:0]), .MDOUT(MDOUT001),
                  .ID(8'd1), .START(START001),
                  .IREQ(IREQ001), .IRDY(IRDY001), .DIN(DIN001),
                  .OREQ(OREQ001), .ORDY(ORDY001), .DOUT(DOUT001),
                  .RST(RST));

   reg [31:0]  MDOUT;

   always@(posedge CLK) begin
      if(NID == 8'd1) begin
         MDOUT <= MDOUT001;
      end
   end

endmodule // nrm
