/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "define.vh"

module sln(CLK,
           IN,
           OUT);
   
   input CLK;
   input [31:0] IN;
   output [31:0] OUT;


   wire [9:0]    fst_addr;
   
   wire [9:0]    sta_addr, stb_addr, stc_addr;
   wire [31:0]   sta_dout, stb_dout, stc_dout;

   wire [31:0]   fm1_a, fm1_b, fm1_q;
   wire [31:0]   fm2_a, fm2_b, fm2_q;
   wire [31:0]   fm3_a, fm3_b, fm3_q;
   wire [31:0]   fm4_a, fm4_b, fm4_q;
   wire [31:0]   fa1_a, fa1_b, fa1_q;
   wire [31:0]   fa2_a, fa2_b, fa2_q;
   wire [31:0]   fa3_a, fa3_b, fa3_q;

   reg [23*3-1:0]    frac;
   wire [7:0]    exp;
   wire [8:0]    dexp;

   reg [13:0]   fraca, fracb;

   
   wire [31:0]   itof9_q;

   wire [31:0]   ftof14a_q, ftof14b_q;
   
   reg [10*26-1 : 0] sr_badr;
   reg [8*29-1 : 0] sr_exp;

   assign        exp = sr_exp[8*29-1:8*28];
   assign        dexp = {1'b0, exp} - 9'd127;

   assign        fst_addr = IN[22:13];

   assign        sta_addr = sr_badr[10*2-1 : 10*1];
   assign        stb_addr = sr_badr[10*10-1 : 10*9];
   assign        stc_addr = sr_badr[10*14-1 : 10*13];

   assign        fm1_a = sta_dout;
   assign        fm1_b = {1'b0, 8'h7F, frac[23*3-1:23*2]};

   assign        fa1_a = fm1_q;
   assign        fa1_b = stb_dout;

   assign        fm2_a = ftof14a_q;
   assign        fm2_b = ftof14b_q;

   assign        fa2_a = fa1_q;
   assign        fa2_b = fm3_q;

   assign        fm3_a = fm2_q;
   assign        fm3_b = stc_dout;

   assign        fa3_a = fa2_q;
   assign        fa3_b = itof9_q;

   assign        fm4_a = fa3_q;
   assign        fm4_b = 32'h3F317218; // 1/log2(e) = 0.69314718055994530941723212145817..

   assign        OUT = fm4_q;
   
   always@(posedge CLK) begin
      frac    <= {frac[23*2-1:0], IN[22:0]};
      fraca <= {1'b0,IN[12:0]};
      fracb <= {1'b1,13'b0} - {1'b0, IN[12:0]};
      sr_badr <= {sr_badr[10*25-1 : 0], fst_addr};
      sr_exp  <= {sr_exp[8*28-1:0], IN[30:23]};
   end

   ftof14 ftof14a(.clk(CLK),
                  .a(fraca),
                  .result(ftof14a_q));
   
   ftof14 ftof14b(.clk(CLK),
                  .a(fracb),
                  .result(ftof14b_q));
   
   itof9 itof9(.clk(CLK),
               .a(dexp),
               .result(itof9_q));

   
   fmul_mu fm1(.clk(CLK),
               .a(fm1_a),
               .b(fm1_b),
               .result(fm1_q));

   fmul_mu fm2(.clk(CLK),
               .a(fm2_a),
               .b(fm2_b),
               .result(fm2_q));

   fmul_mu fm3(.clk(CLK),
               .a(fm3_a),
               .b(fm3_b),
               .result(fm3_q));

   fmul_mu fm4(.clk(CLK),
               .a(fm4_a),
               .b(fm4_b),
               .result(fm4_q));

   fadd_hs_fu fa1(.clk(CLK),
                  .a(fa1_a),
                  .b(fa1_b),
                  .result(fa1_q));

   fadd_hs_fu fa2(.clk(CLK),
                  .a(fa2_a),
                  .b(fa2_b),
                  .result(fa2_q));

   fadd_hs_fu fa3(.clk(CLK),
                  .a(fa3_a),
                  .b(fa3_b),
                  .result(fa3_q));

   sln_table_a sta(.clka(CLK),
                   .addra(sta_addr),
                   .douta(sta_dout));

   sln_table_b stb(.clka(CLK),
                   .addra(stb_addr),
                   .douta(stb_dout));

   sln_table_c stc(.clka(CLK),
                   .addra(stc_addr),
                   .douta(stc_dout));
   
endmodule // sln
