/* NRM-FPGA: an NRM solver on an FPGA
 * 
 * Copyright (C) 2004-2009, Amano Lab., Keio University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

`include "ipq.vh"
`include "define.vh"

module ipq(CLK,
           RST,
           INST,
           RIN,
           DIN,
           DOUT,
           BUSY);

   input         CLK;
   input         RST;

   input  [2:0]  INST;
   input  [9:0]  RIN;
   input  [41:0] DIN;
   output [41:0] DOUT;
   output        BUSY;

   reg           BUSY;

   wire         tindx_wea,tindx_web;
   wire [9:0]   tindx_addra,tindx_addrb;
   wire [9:0]   tindx_dina,tindx_dinb;
   wire [9:0]   tindx_douta,tindx_doutb;
   
   wire         tdata_wea,tdata_web;
   wire [9:0]   tdata_addra,tdata_addrb;
   wire [41:0]  tdata_dina,tdata_dinb;
   wire [41:0]  tdata_douta,tdata_doutb;

   wire [31:0]  lt0_a, lt0_b;
   wire         lt0_q;
   
   reg [1:0]    read_flag;
   reg [3:0]    updt_flag_up;
   reg [5:0]    updt_flag_dw;
   reg          updt;

   reg [9:0]    addr;
   reg [9:0]    ptr;
   reg [31:0]   data;
   reg          dw_cmp;
   wire         ex_pn, ex_rc, ex_lc, ex_end;

   reg          pn_flag;

   reg [41:0]   root;

   reg          iwe_flag;
   
   assign       ex_pn  =  updt_flag_up[3]  & lt0_q;
   assign       ex_rc  = (updt_flag_dw[4] & ~dw_cmp & lt0_q) |
                         (updt_flag_dw[5] & lt0_q);
   assign       ex_lc  = (updt_flag_dw[4] & dw_cmp & ~lt0_q) |
                         (updt_flag_dw[5] & ~lt0_q);
   assign       ex_end = updt_flag_dw[4] & dw_cmp & lt0_q;

   assign       tindx_wea   = (ex_pn | ex_rc | ex_lc) ? `WRITE : `READ;

   assign       tindx_addra = ((INST == `IPQ_READ) | (INST == `IPQ_UPDT)) ? RIN :
                              (ex_rc) ? tdata_douta[41:32] :
                                        tdata_doutb[41:32];
   assign       tindx_dina  = ptr;

   assign       tindx_web   = (INST == `IPQ_IWEI | (ex_pn | ex_rc | ex_lc)) ? `WRITE : `READ;
   assign       tindx_addrb = (INST == `IPQ_IWEI | INST == `IPQ_IWET) ? DIN[41:32] :
                              (ex_rc | ex_lc) ? addr :
                              (ex_pn) ? tdata_douta[41:32] : 10'd0;
   assign       tindx_dinb  = (INST == `IPQ_IWEI) ? DIN[9:0] :
                              (ex_rc) ? {ptr[8:0],1'b0} :
                              (ex_lc) ? {ptr[8:0],1'b1} :
                              (ex_pn) ? {1'b0,ptr[9:1]} : 10'd0;

   assign       tdata_wea   = (updt | ex_pn | ex_rc | ex_lc) ? `WRITE : `READ;
   assign       tdata_addra = (read_flag[0] | updt) ? tindx_douta :
                              (|updt_flag_dw[3:0] | (updt_flag_dw[4] & ~ex_rc & ~ex_lc)) ? {ptr[8:0],1'b0} : 
                              ptr;
   
   assign       tdata_dina  = (updt)  ? {addr, data} :
                              (ex_rc) ? tdata_douta :
                                        tdata_doutb;

   assign       tdata_web   = (iwe_flag | ex_pn | ex_rc | ex_lc) ? `WRITE : `READ;

   assign       tdata_addrb = (iwe_flag) ? tindx_doutb :
                              (|updt_flag_up[3:0])  ? {1'b0, ptr[9:1]} :
                              (updt)  ? {tindx_douta[8:0], 1'b0} : // Don't care
                              (ex_rc) ? {ptr[8:0], 1'b0} :
                                        {ptr[8:0], 1'b1};

   assign       tdata_dinb  = (ex_pn) ? tdata_douta : 
                                        {addr, data};
   
   assign       lt0_a = (|updt_flag_dw[2:1]) ? data : tdata_douta[31:0];
   assign       lt0_b = (updt_flag_dw[1])    ? tdata_douta[31:0] : tdata_doutb[31:0];

   assign       DOUT = (read_flag[1]) ? {10'd0,tdata_douta} : root;

   always@(posedge CLK) begin
      if(tindx_wea & tindx_web & (tindx_addra == tindx_addrb)) begin
         $display("tindx : %d pn:%d",tindx_addra, ex_pn);
      end
      if(tdata_wea & tdata_web & (tdata_addra == tdata_addrb)) begin
         $display("tdata : %d pn:%d",tdata_addra, ex_pn);
      end
   end

//   updt_flag_up[2] & lt0_q
//   updt_flag_dw[5] & (&dw_cmp[1:0]) --- END

   always@(posedge CLK) begin
      if(RST) iwe_flag <= `DISABLE;
      else begin
         iwe_flag <= (INST == `IPQ_IWET);
      end
   end

   always@(posedge CLK) begin
      if((tdata_wea & (tdata_addra == 10'd1)))
        root <= tdata_dina;
      else if(tdata_web & (tdata_addrb == 10'd1)) begin
         if(iwe_flag)
           root <= {addr, data};
         else
           root <= tdata_dinb;
      end
   end

   always@(posedge CLK) begin
      if(RST) 
        read_flag <= 2'b0;
      else
        read_flag <= {read_flag[0], (INST == `IPQ_READ)};
   end

   // UPDATE
   always@(posedge CLK) begin
      if(INST == `IPQ_UPDT) pn_flag <= `DISABLE;
      else if(ex_pn)        pn_flag <= `ENABLE;
   end

   always@(posedge CLK) begin
      if(updt)
        ptr <= tindx_douta;
      else if(ex_pn)
        ptr <= {1'b0,ptr[9:1]};
//      else if(ex_rc & ~ptr[9])
      else if(ex_rc & ~ptr[8])
        ptr <= {ptr[8:0],1'b0};
//      else if(ex_lc & ~ptr[9])
      else if(ex_lc & ~ptr[8])
        ptr <= {ptr[8:0],1'b1};
   end
   
   always@(posedge CLK) begin
      if(RST)                  dw_cmp <= `DISABLE;
      else if(updt_flag_dw[3]) dw_cmp <= lt0_q;
   end

   always@(posedge CLK) begin
      if(RST)
        updt_flag_up <= 4'b0;
      else
        updt_flag_up <= {updt_flag_up[2:0], 
                         ( (updt & (tindx_douta != 10'd1)) | (ex_pn & (tdata_addrb != 10'd1)) )};

      if(RST)
        updt_flag_dw <= 6'b0;
      else
        updt_flag_dw <= {(updt_flag_dw[4] & ~ex_end & ~ex_rc & ~ex_lc),
                         updt_flag_dw[3:0],
//                         ((updt_flag_up[3] & ~lt0_q & ~pn_flag) |
                         ((updt_flag_up[3] & ~lt0_q & ~pn_flag & ~ptr[9]) |
//                          ((ex_rc | ex_lc) & ~ptr[9]) |
                          ((ex_rc | ex_lc) & ~ptr[8]) |
                          (updt & (tindx_douta == 10'd1)))};
   end

   always@(posedge CLK) begin
      if(INST == `IPQ_UPDT)      addr <= RIN;
      else if(INST == `IPQ_IWET) addr <= DIN[41:32];
      if(INST == `IPQ_UPDT | INST == `IPQ_IWET) data <= DIN[31:0];
   end

   always@(posedge CLK) begin
      if(RST) begin
         updt <= `DISABLE;
         BUSY <= `DISABLE;
      end
      else begin
         if(INST == `IPQ_UPDT)                     BUSY <= `ENABLE;
//         else if((ex_end | ((ex_rc | ex_lc) & ptr[9])) |
         else if((ex_end | ((ex_rc | ex_lc) & ptr[8])) |
//                 (updt_flag_up[3] & ~lt0_q & pn_flag)  |
                 (updt_flag_up[3] & ~lt0_q & (pn_flag | ptr[9])) |
                 (ex_pn & (tdata_addrb == 10'd1))) BUSY <= `DISABLE;
         if(INST == `IPQ_UPDT) updt <= `ENABLE;
         else if(updt)         updt <= `DISABLE;
      end
   end

   dpram10x10 tindx(.clka(CLK), .clkb(CLK),
                    .wea(tindx_wea), .web(tindx_web),
                    .addra(tindx_addra), .addrb(tindx_addrb),
                    .dina(tindx_dina), .dinb(tindx_dinb),
                    .douta(tindx_douta), .doutb(tindx_doutb));

   dpram42x10 tdata(.clka(CLK), .clkb(CLK),
                    .wea(tdata_wea), .web(tdata_web),
                    .addra(tdata_addra), .addrb(tdata_addrb),
                    .dina(tdata_dina), .dinb(tdata_dinb),
                    .douta(tdata_douta), .doutb(tdata_doutb));

   lt lt0(.CLK(CLK), .a(lt0_a), .b(lt0_b), .q(lt0_q));

endmodule // ipq
