/**
 * SBML-SSAlib bridge: a bridge library to connect SSAlib and SBML
 *
 * Copyright (C) 2004-2009, SBML-SSA Team, Keio/Seikei University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *                     
 * @file    SBMLDtoSSAPConverter.cpp
 * @brief   SBML Document -> SSA Problem converter
 * @author  Keio/Seikei SBML-SSA Team.
 */

#include "SBMLDtoSSAPConverter.h"
#include "ASTNodeCalculator.h"
#include <map>

const string debugStrHeader = "[DEBUG](SBMLDtoSSAPConverter) ";

/**
 * Constructor
 */   
SBMLDtoSSAPConverter::SBMLDtoSSAPConverter()
{}
 

/**
 * Destructor
 */
SBMLDtoSSAPConverter::~SBMLDtoSSAPConverter()
{}


/**
 * normalize (this function may not be required)
 */
SBMLDocument*
SBMLDtoSSAPConverter::normalizeSBMLDocument(SBMLDocument *sbmlDocument)
{
  if (!sbmlDocument) return sbmlDocument;

  // not implemented yet.

  return sbmlDocument;
}

/**
 *
 * convert
 *
 * (NOTE) NULL will be returned if the conversion failed.
 *
 * (NOTE) The following SBML components are not supported:
 *        1. FunctionDefinition
 *        2. Event
 *       (There may be problem with other SBML components.)
 *
 * (NOTE) unit conversion and evaluation of kineticlaw's math 
 *        would need to be improved.
 */
SSAProblem* 
SBMLDtoSSAPConverter::convert(SBMLDocument *sbmlDocument)
{
  // should not happen
  if (!sbmlDocument) return 0;

  Model *model = sbmlDocument->getModel();

  // should not happen
  if (!model) return 0;

  SSAProblem *problem = new Problem;
  map<string,Participant*>  mapOfParticipants;
  int problemOffset = 1;

  problem->initialConditions = new InitialCondition*[model->getNumSpecies() + problemOffset];
  problem->reactions         = new SSAReaction*[model->getNumReactions() + problemOffset];

  // (NOTE)
  //
  // Currently (2009-08-04), NULL needs to be set to the tail of 
  // list of Problem.initialConditions and Problem.reactions.
  //
  problem->initialConditions[model->getNumSpecies()] = 0;
  problem->reactions[model->getNumReactions()] = 0;

  ASTNodeCalculator astCalculator;

  // ---------------------------------------------------
  //
  // create a list of Participants and InitialCondition
  //
  // ---------------------------------------------------

  for (unsigned int i=0; i < model->getNumSpecies(); i++)
  { 
    Species *sp = model->getSpecies(i);
    if (!sp) continue;  // should be modified
 
    string spId(sp->getId());

    // creates a Participant 
    //
    Participant *pp = new Participant;
    unsigned int strSize = spId.length();
    pp->name = new char[strSize+1];
    strncpy(pp->name, spId.c_str(),strSize+1);
    mapOfParticipants.insert(pair<string,Participant*>(spId,pp));

    // creates a InitialCondition
    //
    InitialCondition *ic = new InitialCondition;

    double population; 

    // for unit check and unit conversion
    UnitDefinition *ud = 0;

    // ---------------------------------------------------------------------------
    //
    // Checks Unit of each species
    //
    
    if (sp->isSetInitialAmount())
    {
      population = sp->getInitialAmount();

      // get the unit of the population
      ud = sp->getDerivedUnitDefinition();

      cout << "Population (" << spId << ") is " << population << " (substance)" << endl;
    }
    else if (sp->isSetInitialConcentration())
    {
      population = sp->getInitialConcentration();
      Compartment *comp = model->getCompartment(sp->getCompartment());
      double compSize = comp->getSize(); 
      population *= compSize;

      // get the unit of the population
      ud = UnitDefinition::combine(sp->getDerivedUnitDefinition(),
                                   comp->getDerivedUnitDefinition()); 
                                   
    }
    else
    {
      // population should be set by initialAssignment or AssignmentRule

      Compartment *comp = model->getCompartment(sp->getCompartment());

      //
      // get the unit of the population
      //
      // (Reference: 4.8.5: The substanceUnits and hasOnlySubstanceUnit attributes
      //  in SBML Level 2 Version 4 Specification) 
      //
      if ( !sp->getHasOnlySubstanceUnits() && comp->getSpatialDimensions() > 0 )
      {
        // unit of this species is substance/size (concentration)
        ud = UnitDefinition::combine(sp->getDerivedUnitDefinition(),
                                     comp->getDerivedUnitDefinition()); 
      }
      else
      {
        // unit of this species is substance (amount)
        ud = sp->getDerivedUnitDefinition();
      }
    }

    // -------------------------------------------------------------------------
    // Check if the unit of the population is valid.
    //
    //  (NOTE) Currently, only 'mole' or 'item' kind based unit is supported
    //         (i.e. glam, kilogram and dimensions are not supported.)
    //

    //
    // TODO (error message should be logged).
    //
    if (ud->getNumUnits() > 1) return 0;

    UnitKind_t kind = ud->getUnit(0)->getKind();

    if ( (kind != UNIT_KIND_MOLE) && (kind != UNIT_KIND_ITEM) ) 
    {
      cerr << "The unit of species (" << spId << ") is not mole nor item : " 
           << UnitDefinition::printUnits(ud) << endl;
      return 0;
    }

    cout << debugStrHeader << "UnitDefinition (" << spId << ") is " << ud->printUnits(ud) << endl;


    // -------------------------------------------------------------------------
    //
    // For species with InitialAssignment and AssignmentRule
    //
    // ** TODO **
    //
    //  1. math needs to be calculated
    //
    if (model->getInitialAssignment(sp->getId()))
    {
      const ASTNode *math = model->getInitialAssignment(sp->getId())->getMath();
      population = astCalculator.evalMath(math,model);
    }
    else if (model->getRule(sp->getId()))
    {
      const ASTNode *math = model->getRule(sp->getId())->getMath();
    
      // (TODO) math need to be calculated
      //
      // population = someclass::calcMath(math,model);
      //
      population = astCalculator.evalMath(math,model);
    }


    // ---------------------------------------------------------------------------
    //
    // Unit Conversion
    //
    //  mole -> items
    //
    if (kind == UNIT_KIND_MOLE)
    {
      population *= AvogadroConstant;
    }

    int    exp        = ud->getUnit(0)->getExponent();
    int    scale      = ud->getUnit(0)->getScale();
    double multiplier = ud->getUnit(0)->getMultiplier();

    population = pow(multiplier * population * pow(10, (double)scale),  (double)exp );
    
    cout << debugStrHeader << "Population (" << spId << ") is " 
         << static_cast<u64>(population) << " (items)" << endl;

    // ---------------------------------------------------------------------------

    ic->participant = pp;
    ic->population  = static_cast<u64>(population);
    problem->initialConditions[i] = ic;
  }

  // ----------------------------------------------------------
  //
  // create a list of SSAReactions
  //
  // -----------------------------------------------------------

  for (unsigned int i=0; i < model->getNumReactions(); i++)
  { 
    Reaction *reac = model->getReaction(i);
    if (!reac) continue;  // should be modified

    unsigned int termListOffset = 1;

    SSAReaction *ssaReac = new SSAReaction;
    ssaReac->reactants   = new Term*[reac->getNumReactants() + termListOffset]; 
    ssaReac->products    = new Term*[reac->getNumProducts() + termListOffset]; 

    ssaReac->reactants[reac->getNumReactants()] = 0;
    ssaReac->products[reac->getNumProducts()]   = 0;

    //
    // create listOf ssaTerm* (reactants)
    //
    for (unsigned int t=0; t < reac->getNumReactants(); t++)
    {
      SpeciesReference *spr = reac->getReactant(t);

      Term* term = new Term;

      if (spr->isSetStoichiometryMath())
      {
        const ASTNode *math = spr->getStoichiometryMath()->getMath();
        term->coefficient = astCalculator.evalMath(math,model);
      }
      else
      {
        term->coefficient = spr->getStoichiometry(); 
      }

      map<string,Participant*>::iterator itp = mapOfParticipants.find(spr->getSpecies());
      if (itp != mapOfParticipants.end())
      {
        term->participant = (*itp).second;
      }
      else
      {
        // this should not happen ...
      }

      ssaReac->reactants[t] = term;
    } 

    //
    // create listOf ssaTerm* (products)
    //
    for (unsigned int t=0; t < reac->getNumProducts(); t++)
    {
      SpeciesReference *spr = reac->getProduct(t);

      Term* term = new Term;

      if (spr->isSetStoichiometryMath())
      {
        const ASTNode *math = spr->getStoichiometryMath()->getMath();
        term->coefficient = astCalculator.evalMath(math,model);
      }
      else
      {
        term->coefficient = spr->getStoichiometry(); 
      }

      map<string,Participant*>::iterator itp = mapOfParticipants.find(spr->getSpecies());
      if (itp != mapOfParticipants.end())
      {
        term->participant = (*itp).second;
      }
      else
      {
        // this should not happen ...
      }

      ssaReac->products[t] = term;
    } 

    // ------------------------------------------------------------------------------
    //
    // get a propensity value
    //

    KineticLaw *kl = reac->getKineticLaw();        

    // would need to be improved in the future
    ssaReac->propensity = astCalculator.evalMathOfMassAction(kl,model);

    cout << debugStrHeader << "propensity " << reac->getId() << " " << ssaReac->propensity << endl;

    // ------------------------------------------------------------------------------

    problem->reactions[i] = ssaReac;
  }

  return problem;
}

/**
 * printError
 */
void 
SBMLDtoSSAPConverter::printErrors()
{
  // not implemented yet.
}


