/**
 * SBML-SSAlib bridge: a bridge library to connect SSAlib and SBML
 *
 * Copyright (C) 2004-2009, SBML-SSA Team, Keio/Seikei University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *                    
 * createCaDrivenPSSP.cpp
 * Creates an SBML file of Post-Synaptic Signaling Pathway which is
 * slightly modified to drive a species (Ca)
 *
 * This program is based on an example program (createExampleSBML.cpp)
 * bundled with libSBML. 
 *
 */

#include <iostream>
#include <list>
#include <sbml/SBMLTypes.h>

using namespace std;

//
// (cited from j.c implemented by John)
//
// To drive a species, we modify any reactions where it's a reactant to
// conserve it (it's a product in equal number too).  That is, the product
// stoichiometry is always forced to equal the reactant stoichiometry for
// the driven species, even if the reactant stoichiometry is zero.  Then
// the reactions conserve the species and we can set it to whatever value
// we like.
//

//------------------------------------------------------------------------------

//
// Functions for creating an SBML file
//

// top module for creating a model
SBMLDocument* createPostSynapticSignalingPathway(int level, int version);

// sub functions invoked from the createPostSynapticSignalingPathway function
void createUnitDefinitions(Model *model);
void createGlobalParameters(Model *model);
void createCompartment(Model *model);
void createSpecies(Model *model);
void createReactionsOfCalciumBinding(Model *model);
void createReactionsOfAssociation(Model *model);
void createReactionsOfPhosphorylation(Model *model);
void createEventsOfSpikeTrain(Model *model);

// helper functions for creating one SBase element 
// (Parameter, Reaction, Species and etc.)

Parameter*
createGlobalParameter(Model *model, const string& gparaid, 
                      double value, const string& unitid); 

Species* 
createSpecies(Model *model, const string& compid, const string& spid,
              double initialamount, const string& unitid);

Reaction* 
createReaction(Model *model, const string& reactionId, 
               const list<string> &lstOfReactant, 
               const list<string> &lstOfProduct, 
               const string &parameterId);

ASTNode* createMathInTriggerBasedOnTime(double time);


//
// Helper functions for validating and writing the SBML documents created.
//
bool validateSBML(SBMLDocument *sbmlDoc);
bool writeSBML(const SBMLDocument *sbmlDoc, const string& filename);


//
// Usage
//
void usage(const char* myname)
{
  cout << "usage: " << myname;
  cout << " -l<n>  : Set the SBML Level   " << endl;
  cout << " -v<n>  : Set the SBML Version " << endl;
  cout << "  -h     : Show this message"    << endl;
  cout << endl;
}


//------------------------------------------------------------------------------

int
main (int argc, char* argv[])
{
  // output file name (default).
  string filename       = "pssp-ca.xml"; 
  SBMLDocument* sbmlDoc = 0;
  bool SBMLok           = false;

  //
  // SBML Level and Version (default)
  //
  unsigned int level   = 2;
  unsigned int version = 4;

  int c;
  while((c = getopt(argc, argv, "l:v:")) != -1) {
    switch (c) {
    case 'l':
      level = atoi(optarg);
      break;
    case 'v':
      version = atoi(optarg);
      break;
    case 'h':
    case '?':
    default:
      usage(argv[0]);
      return 1;
    }
  }

  if (optind < argc)
    filename.assign(argv[optind]);

  try
  {
    sbmlDoc = createPostSynapticSignalingPathway(level, version);
    SBMLok  = validateSBML(sbmlDoc);
    if (SBMLok) writeSBML(sbmlDoc, filename);
    delete sbmlDoc;
    if (!SBMLok) return 1;

  }
  catch (std::bad_alloc& e)
  {
    cerr << e.what() << ": Unable to allocate memory." << endl;
    return 1;
  }
  catch (...)
  {
    cerr << "Unexpected exceptional condition encountered." << endl;
    return 1;
  }
 
  return 0;
}


/**
 *
 * createPostSynapticSignalingPathway()
 *
 */
SBMLDocument* createPostSynapticSignalingPathway(int level, int version)
{
  //---------------------------------------------------------------------------
  // Creates an SBMLDocument object 
  //---------------------------------------------------------------------------

  SBMLDocument* sbmlDoc = new SBMLDocument(level,version);

  //---------------------------------------------------------------------------
  // Creates a Model object inside the SBMLDocument object. 
  //---------------------------------------------------------------------------

  Model* model = sbmlDoc->createModel();
  model->setId("PostSynapticSignalingPathway");

  //---------------------------------------------------------------------------
  // Creates UnitDefinition objects inside the Model object.
  //---------------------------------------------------------------------------

  createUnitDefinitions(model);

  //---------------------------------------------------------------------------
  // Creates Global Parameter objects (representing reaction rates)
  // inside the Model object. 
  //---------------------------------------------------------------------------

  createGlobalParameters(model);
  
  //---------------------------------------------------------------------------
  // Creates a Compartment object inside the Model object. 
  //---------------------------------------------------------------------------

  createCompartment(model);

  //---------------------------------------------------------------------------
  // Creates Species objects inside the Model object. 
  //---------------------------------------------------------------------------

  createSpecies(model);

  //---------------------------------------------------------------------------
  // Creates Reaction objects inside the Model object. 
  //---------------------------------------------------------------------------

  createReactionsOfCalciumBinding(model);
  createReactionsOfAssociation(model);
  createReactionsOfPhosphorylation(model);

  //---------------------------------------------------------------------------
  // Creates Event objects inside the Model object.
  //---------------------------------------------------------------------------

  createEventsOfSpikeTrain(model);


  // Returns the created SBMLDocument object.
  // The returned object must be explicitly deleted by the caller,
  // otherwise a memory leak will happen.

  return sbmlDoc;

}


//-------------------------------------------------------------------------------------
// 
// Sub modules for creating a model 
//
//-------------------------------------------------------------------------------------

/**
 *
 * createUnitDefinitions
 *
 *  1) Redefines "substance" as micro mole
 *  2) Defines   "per_second"        (unit for parameter for bindings)
 *  3) Defines   "per_uM_per_second" (unit for parameter for dissociating)
 *
 */
void createUnitDefinitions(Model *model)
{
  //---------------------------------------------------------------------------
  // Creates UnitDefinition objects inside the Model object.
  //---------------------------------------------------------------------------

  // Temporary pointers
  UnitDefinition *unitdef;
  Unit* unit;

  // Redefines "substance" as micro mole

  unitdef = model->createUnitDefinition();
  unitdef->setId("substance");
  unitdef->setName("substance_mM");

  unit = unitdef->createUnit();
  unit->setKind(UNIT_KIND_MOLE);
  unit->setScale(-6);

  // defines "s^{-1}" (per_second)

  unitdef = model->createUnitDefinition();
  unitdef->setId("per_second");
  unitdef->setName("per_second");

  unit = unitdef->createUnit();
  unit->setKind(UNIT_KIND_SECOND);
  unit->setExponent(-1);

  // defines "uM^{-1} * s^{-1}" (per_uM_per_second) 

  unitdef = model->createUnitDefinition();
  unitdef->setId("per_uM_per_second");
  unitdef->setName("per_uM_per_second");

  unit = unitdef->createUnit();
  unit->setKind(UNIT_KIND_MOLE);
  unit->setScale(-6);
  unit->setExponent(-1);

  unit = unitdef->createUnit();
  unit->setKind(UNIT_KIND_SECOND);
  unit->setExponent(-1);
}


/**
 *
 * createGlobalParameters
 *
 * creates 47 global parameters
 *
 */
void createGlobalParameters(Model *model)
{
  Parameter *gpara;

  const string unitB = "per_uM_per_second"; // unit for bindings
  const string unitD = "per_second";        // unit for dissociating

  // Creates parameters (alphaFwd{i} = {5.4, 5.4} (i = 0,1) )

  createGlobalParameter(model,"alphaFwd_0", 5.4, unitB); 
  createGlobalParameter(model,"alphaFwd_1", 5.4, unitB); 

  // Creates parameters (alphaBwd{i} = {40, 9.3} (i = 0,1) )

  createGlobalParameter(model,"alphaBwd_0", 40,  unitD); 
  createGlobalParameter(model,"alphaBwd_1", 9.3, unitD); 

  // Creates parameters (betaFwd{i} = {140,140} (i=0,1) )

  createGlobalParameter(model,"betaFwd_0", 140, unitB); 
  createGlobalParameter(model,"betaFwd_1", 140, unitB); 

  // Creates parameters (betaBwd{i} = {2500,800} (i=0,1) )  

  createGlobalParameter(model,"betaBwd_0", 2500, unitD); 
  createGlobalParameter(model,"betaBwd_1", 800,  unitD); 

  // Creates parameters (gammaFwd{i} = {44,44} (i=0,1) )

  createGlobalParameter(model,"gammaFwd_0", 44,  unitB); 
  createGlobalParameter(model,"gammaFwd_1", 44,  unitB); 

  // Creates parameters (gammaBwd{i} = {33,2.7} (i=0,1) )  

  createGlobalParameter(model,"gammaBwd_0", 33,  unitD); 
  createGlobalParameter(model,"gammaBwd_1", 2.7, unitD); 

  // Creates parameters (deltaFwd{i} = {76,76} (i=0,1) )

  createGlobalParameter(model,"deltaFwd_0", 76, unitB); 
  createGlobalParameter(model,"deltaFwd_1", 76, unitB); 

  // Creates parameters (deltaBwd{i} = {300,33} (i=0,1) )  

  createGlobalParameter(model,"deltaBwd_0", 300, unitD); 
  createGlobalParameter(model,"deltaBwd_1", 33,  unitD); 

  // Creates a parameter (zetaFwd  = 50)

  createGlobalParameter(model,"zetaFwd", 50, unitB); 

  // Creates a parameters (zetaBwd = 1000)

  createGlobalParameter(model,"zetaBwd", 1000, unitD);

  //
  // Creates parameters (epsilonFwd)
  //

  double epsilonFwd[3][3]={ { 0.0038,0.022,0.12 },
                            { 0.059,3.4,1.9 },
                            { 0.92,5.2,37 } };

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
       ostringstream oss;
       oss << "epsilonFwd_" << i << j;
       const string parameterId(oss.str());
       
       createGlobalParameter(model,parameterId, epsilonFwd[i][j], unitB); 
    }
  }

  //
  // Creates parameters (epsilonBwd)
  //

  double epsilonBwd[3][3]={ { 5.5,3.1,1.7 },
                            { 6.1,3.4,1.9 },
                            { 6.8,3.8,1.7 } };
  
  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
       ostringstream oss;
       oss << "epsilonBwd_" << i << j;
       const string parameterId(oss.str());
       
       createGlobalParameter(model,parameterId, epsilonBwd[i][j], unitD); 
    }
  }

  //
  // Creates parameters (etaFwd)
  //

  double etaFwd[3][3]={ { 0,0.06,0.12 },
                        { 0.032,0.094,0.154 },
                        { 0.064,0.124,0.96 } };  

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
       ostringstream oss;
       oss << "etaFwd_" << i << j;
       const string parameterId(oss.str());
       
       createGlobalParameter(model,parameterId, etaFwd[i][j], unitD); 
    }
  }
  
  //
  // Creates parameters (CaSpikeOn=20 uM/litre * 6e-18 litre = 1.2e-16 uM, CaSpikeOff=0)
  //
  
  createGlobalParameter(model, "CaSpikeOn",  1.2e-16, "substance");
  createGlobalParameter(model, "CaSpikeOff",       0, "substance");
    
}   


/**
 *
 * createCompartment
 *
 */
void createCompartment(Model *model)
{
  Compartment* comp;
  const string compId = "psd";

  // Creates a Compartment object ("psd")

  comp = model->createCompartment();
  comp->setId(compId);
 
  // Sets the "size" attribute of the Compartment object.

  comp->setSize(6e-18);
  comp->setUnits("litre");
}


/**
 *
 * createSpecies
 *
 * creates 146 species
 *
 */
void createSpecies(Model *model)
{
  // Temporary pointer (reused more than once below).
  
  Species *sp;
  const string spUnit = "substance";
  //const string spUnit = "item";

  // There is only one compartment in this model.
  const Compartment *comp = model->getCompartment(0);
  if (!comp) return; // this should not happen.
  
  const string compId = comp->getId();

  //---------------------------------------------------------------------------
  // Creates a Species object "Ca": Free (aqueous) Calcium ion Ca2+
  //---------------------------------------------------------------------------

  // Create the Species objects inside the Model object. 

  sp = createSpecies(model, compId, "Ca", 0, spUnit);

  //---------------------------------------------------------------------------
  // Creates a Species object "CaMKII": Unassociated single Calmodulin-dependent 
  // Protein Kinase II subunit 
  //---------------------------------------------------------------------------

  sp = createSpecies(model, compId, "CaMKII", 4.8e-16, spUnit);  // 80uM/litre * 6.0e-18litre = 4.8e-16 uM

  // (CaMKII=80uM/litre, volume=6e-18liters) 80e-6 * 6e-18 * 6.0221415e+23 = 289 items
  //sp = createSpecies(model, compId, "CaMKII", spUnit, 289);

  //---------------------------------------------------------------------------
  // Creates Species objects "CaMij"  (0<=i,j<=2): Calmodulin with i Ca2+ ions 
  // bound at the carboxyl terminus and j Ca2+ ions bound at the amino terminus 
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
       ostringstream oss;
       oss << "CaM_" << i << j;
       const string spid(oss.str());
       double initialAmount = 0;
       
       if (i==0 && j==0) initialAmount = 1.8e-16; // 30uM/litre * 6.0e-18 litre = 1.8e-16 uM

       // (CaM00=30uM/litre, volume=6e-18liters) => 30e-6 * 6e-18 * 6.0221415e+23 = 108 items)
       //if (i==0 && j==0) initialAmount = 108;

       sp = createSpecies(model, compId, spid, initialAmount, spUnit);
    }
  }       

  //---------------------------------------------------------------------------
  // Creates Species objects:
  //  "Kij"  (0<=i,j<=2): Complex of one CaMKII and one CaMij
  //  "pKij" (0<=i,j<=2): Kij with phosphorylated CaMKII 
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
       ostringstream oss;
       oss << "K_" << i << j;
       const string spid_k(oss.str());

       sp = createSpecies(model, compId, spid_k, 0, spUnit);

       oss.str("");
       oss << "pK_" << i << j;
       const string spid_pk(oss.str());

       sp = createSpecies(model, compId, spid_pk, 0, spUnit);
     
    }
  }       

  //---------------------------------------------------------------------------
  // Creates Species objects "Dijkl" (0<=i,j,k,l<=2): Complex of one Kij and 
  // one Kkl
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      for (int k=0; k < 3; k++)
      {     
        for (int l=0; l < 3; l++)
	{
	   //
	   // skip duplicated pairs (e.g, K00 + K00,  K01 + K10 & K10 + K01)
	   //
	   if ( (i<k) || (i==k && j<l) )
           {	      
	     ostringstream oss;
	     oss << "D_" << i << j << k << l;
	     const string spid(oss.str());

             sp = createSpecies(model, compId, spid, 0, spUnit);
	   }
	}
      }
    }
  }       


  //---------------------------------------------------------------------------
  // Creates Species objects "pDijkl" (0<=i,j,k,l<=2): Semiphosphorylated doublet: 
  // a complex of one pKij and one Kkl 
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      for (int k=0; k < 3; k++)
      {     
        for (int l=0; l < 3; l++)
	{
          ostringstream oss;
          oss << "pD_" << i << j << k << l;
          const string spid(oss.str());

          sp = createSpecies(model, compId, spid, 0, spUnit);
	}
      }
    }
  }       
 
}


/**
 *
 * createReactionsOfCalciumBinding
 *
 * Creates 48 Reactions in Calcium Binding
 *
 *
 * (cited from j.c implemented by John)
 *
 * To drive a species, we modify any reactions where it's a reactant to
 * conserve it (it's a product in equal number too).  That is, the product
 * stoichiometry is always forced to equal the reactant stoichiometry for
 * the driven species, even if the reactant stoichiometry is zero.  Then
 * the reactions conserve the species and we can set it to whatever value
 * we like.
 *
 */
void createReactionsOfCalciumBinding(Model *model)
{
  // Temporary objects.

  list<string> lstOfReactants;
  list<string> lstOfProducts;
  ostringstream oss;
  
  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Calcium Binding (Alpha)"
  //
  //  Ca + CaMij -> CaM(i+1)j  + Ca <alpha_i^{->}>   (0<=i<=1, 0<=j<=2)
  //       CaMij <- CaM(i+1)j       <alpha_i^{<-}>   (0<=i<=1, 0<=j<=2)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 2; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      // ==================================================
      // Calcium Binding Alpha (Forward)
      // ==================================================
      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_AlphaFwd_" << i << j;
      const string rid_f(oss.str());

      //-------------------------      

      lstOfReactants.push_back("Ca");
    
      oss.str("");
      oss << "CaM_" << i << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "CaM_" << i+1 << j;
      lstOfProducts.push_back(oss.str());

      lstOfProducts.push_back("Ca");

      oss.str("");
      oss << "alphaFwd_" << i;
      const string parameterId_f(oss.str());

      //-------------------------      
      
      createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);

      // ==================================================
      // Calcium Binding Alpha (Backward)
      // ==================================================
      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_AlphaBwd_" << i << j;
      const string rid_b(oss.str());

      //-------------------------      

      oss.str("");
      oss << "CaM_" << i+1 << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "CaM_" << i << j;
      lstOfProducts.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "alphaBwd_" << i;
      const string parameterId_b(oss.str());

      //-------------------------      
      
      createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);
    }
  }


  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Calcium Binding (Beta)"
  //
  //   Ca + CaMij -> CaMi(j+1) +Ca  <beta_j^{->}>   (0<=i<=2, 0<=j<=1)
  //        CaMij <- CaMi(j+1)      <beta_j^{<-}>   (0<=i<=2, 0<=j<=1)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 2; j++)
    {
      // ==================================================
      // Calcium Binding Beta (Forward)
      // ==================================================
      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_BetaFwd_" << i << j;
      const string rid_f(oss.str());

      //-------------------------      

      lstOfReactants.push_back("Ca");

      oss.str("");
      oss << "CaM_" << i << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "CaM_" << i << j+1;
      lstOfProducts.push_back(oss.str());

      lstOfProducts.push_back("Ca");

      //-------------------------      

      oss.str("");
      oss << "betaFwd_" << j;
      const string parameterId_f(oss.str());

      //-------------------------      

      createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);

      // ==================================================
      // Calcium Binding Beta (Backward)
      // ==================================================
      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_BetaBwd_" << i << j;
      const string rid_b(oss.str());

      //-------------------------

      oss.str("");
      oss << "CaM_" << i << j+1;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "CaM_" << i << j;
      lstOfProducts.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "betaBwd_" << j;
      const string parameterId_b(oss.str());

      //-------------------------      
      
      createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);
    }
  }

  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Calcium Binding (Gamma)"
  //
  //    Ca + Kij -> K(i+1)j + Ca <gamma_i^{->}>   (0<=i<=1, 0<=j<=2)
  //         Kij <- K(i+1)j      <gamma_i^{<-}>   (0<=i<=1, 0<=j<=2)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 2; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      // ==================================================
      // Calcium Binding Gamme (Forward)
      // ==================================================
      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_GammaFwd_" << i << j;
      const string rid_f(oss.str());

      //-------------------------      

      lstOfReactants.push_back("Ca");
    
      oss.str("");
      oss << "K_" << i << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i+1 << j;
      lstOfProducts.push_back(oss.str());

      lstOfProducts.push_back("Ca");

      //-------------------------      

      oss.str("");
      oss << "gammaFwd_" << i;
      const string parameterId_f(oss.str());

      //-------------------------      
      
      createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);


      // ==================================================
      // Calcium Binding Gamme (Backward)
      // ==================================================

      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_GammaBwd_" << i << j;
      const string rid_b(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i+1 << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i << j;
      lstOfProducts.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "gammaBwd_" << i;

      const string parameterId_b(oss.str());
      //-------------------------      
      
      createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);

    }
  }


  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Calcium Binding (Delta)"
  //
  //    Ca + Kij -> Ki(j+1) + Ca <delta_j^{->}>   (0<=i<=2, 0<=j<=1)
  //         Kij <- Ki(j+1)      <delta_j^{<-}>   (0<=i<=2, 0<=j<=1)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 2; j++)
    {
      // ==================================================
      // Calcium Binding Delta (Forward)
      // ==================================================
      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_DeltaFwd_" << i << j;
      const string rid_f(oss.str());

      //-------------------------      

      lstOfReactants.push_back("Ca");
    
      oss.str("");
      oss << "K_" << i << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i << j+1;
      lstOfProducts.push_back(oss.str());

      lstOfProducts.push_back("Ca");

      //-------------------------      

      oss.str("");
      oss << "deltaFwd_" << j;
      const string parameterId_f(oss.str());

      //-------------------------      
      
      createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);


      // ==================================================
      // Calcium Binding Delta (Backward)
      // ==================================================

      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "CalciumBinding_DeltaBwd_" << i << j;
      const string rid_b(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i << j+1;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i << j;
      lstOfProducts.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "deltaBwd_" << j;

      const string parameterId_b(oss.str());
      //-------------------------      
      
      createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);

    }
  }

}


/**
 *
 * createReactionsOfAssociation
 *
 * Creates 252 Reactions in Association
 *
 */
void createReactionsOfAssociation(Model *model)
{
  // Temporary objects.

  list<string> lstOfReactants;
  list<string> lstOfProducts;
  ostringstream oss;
  
  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Association (epsilon)"
  //
  //    CaMKII + CaMij -> Kij  <epsilon_{ij}^{->}>   (0<=i<=2, 0<=j<=2)
  //    CaMKII + CaMij <- Kij  <epsilon_{ij}^{<-}>   (0<=i<=2, 0<=j<=2)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      // ==================================================
      // Association epsilon (Forward)
      // ==================================================

      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "Association_epsilonFwd_" << i << j;
      const string rid_f(oss.str());

      //-------------------------      

      lstOfReactants.push_back("CaMKII");
    
      oss.str("");
      oss << "CaM_" << i << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i << j;
      lstOfProducts.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "epsilonFwd_" << i << j;
      const string parameterId_f(oss.str());

      //-------------------------      
      
      createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);


      // ==================================================
      // Association epsilon (Backward)
      // ==================================================

      lstOfReactants.clear();
      lstOfProducts.clear();
      oss.str("");

      oss << "Association_EpsilonBwd_" << i << j;
      const string rid_b(oss.str());

      //-------------------------      

      oss.str("");
      oss << "K_" << i << j;
      lstOfReactants.push_back(oss.str());

      //-------------------------      

      lstOfProducts.push_back("CaMKII");

      oss.str("");
      oss << "CaM_" << i << j;
      lstOfProducts.push_back(oss.str());

      //-------------------------      

      oss.str("");
      oss << "epsilonBwd_" << i << j;
      const string parameterId_b(oss.str());

      //-------------------------      
      
      createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);

    }
  }


  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Association D (zeta)"
  //
  //    Kij + Kkl -> Dijkl  <zeta_{->}>   (0<=i,j,k,l<=2, ij<=kl)
  //    Kij + Kkl <- Dijkl  <zeta_{<-}>   (0<=i,j,k,l<=2, ij<=kl)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      for (int k=0; k < 3; k++)
      {     
        for (int l=0; l < 3; l++)
        {
	  if ( (i<k) || (i==k && j<l) )
          {
            // ==================================================
	    // Association D zeta (Forward)
	    // ==================================================
            lstOfReactants.clear();
            lstOfProducts.clear();
            oss.str("");

            oss << "Association_D_zetaFwd_" << i << j << k << l;
            const string rid_f(oss.str());

            //-------------------------      

            oss.str("");
            oss << "K_" << i << j;
            lstOfReactants.push_back(oss.str());
  
            oss.str("");
            oss << "K_" << k << l;
            lstOfReactants.push_back(oss.str());

            //-------------------------      

            oss.str("");
            oss << "D_" << i << j << k << l;
            lstOfProducts.push_back(oss.str());

            oss.str("");
            oss << "zetaFwd";
            const string parameterId_f(oss.str());

            //-------------------------      
    
            createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);


            // ==================================================
            // Association D zeta (Backward)
            // ==================================================

            lstOfReactants.clear();
            lstOfProducts.clear();
            oss.str("");

            oss << "Association_D_zetaBwd_" << i << j << k << l;
            const string rid_b(oss.str());

            //-------------------------      

            oss.str("");
            oss << "D_" << i << j << k << l;
            lstOfReactants.push_back(oss.str());

            //-------------------------      

            oss.str("");
            oss << "K_" << i << j;
            lstOfProducts.push_back(oss.str());
  
            oss.str("");
            oss << "K_" << k << l;
            lstOfProducts.push_back(oss.str());

            oss.str("");
            oss << "zetaBwd";
            const string parameterId_b(oss.str());

            //-------------------------      
    
            createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);
          }
        }
      }
    }
  }


  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Association pD (zeta)"
  //
  //    pKij + Kkl -> pDijkl  <zeta_{->}>   (0<=i,j,k,l<=2)
  //    pKij + Kkl <- pDijkl  <zeta_{<-}>   (0<=i,j,k,l<=2)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      for (int k=0; k < 3; k++)
      {     
        for (int l=0; l < 3; l++)
        {
          // ==================================================
          // Association pD zeta (Forward)
          // ==================================================
          lstOfReactants.clear();
          lstOfProducts.clear();
          oss.str("");

          oss << "Association_pD_zetaFwd_" << i << j << k << l;
          const string rid_f(oss.str());

          //-------------------------      

          oss.str("");
          oss << "pK_" << i << j;
          lstOfReactants.push_back(oss.str());

          oss.str("");
          oss << "K_" << k << l;
          lstOfReactants.push_back(oss.str());

          //-------------------------      

          oss.str("");
          oss << "pD_" << i << j << k << l;
          lstOfProducts.push_back(oss.str());

          oss.str("");
          oss << "zetaFwd";
          const string parameterId_f(oss.str());

          //-------------------------      
  
          createReaction(model, rid_f, lstOfReactants, lstOfProducts, parameterId_f);


          // ==================================================
          // Association pD zeta (Backward)
          // ==================================================

          lstOfReactants.clear();
          lstOfProducts.clear();
          oss.str("");

          oss << "Association_pD_zetaBwd_" << i << j << k << l;
          const string rid_b(oss.str());

          //-------------------------      

          oss.str("");
          oss << "pD_" << i << j << k << l;
          lstOfReactants.push_back(oss.str());

          //-------------------------      

          oss.str("");
          oss << "pK_" << i << j;
          lstOfProducts.push_back(oss.str());

          oss.str("");
          oss << "K_" << k << l;
          lstOfProducts.push_back(oss.str());

          oss.str("");
          oss << "zetaBwd";
          const string parameterId_b(oss.str());

          //-------------------------      
  
          createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);
        }
      }
    }
  }
}


/**
 *
 * createReactionsOfPhosphorylation
 *
 * Creates 153 Reactions in Phosphorylation
 *
 */
void createReactionsOfPhosphorylation(Model *model)
{
  // Temporary objects.

  list<string> lstOfReactants;
  list<string> lstOfProducts;
  ostringstream oss;

  //---------------------------------------------------------------------------
  // Creates Reaction objects : "Phosphorylation (eta)"
  //
  //   Dijkl -> pKij +  Kkl <eta_{ij}^{->}>  (0<=i,j,k,l<=2, ij<=kl)
  //   Dijkl ->  Kij + pKkl <eta_{kl}^{->}>  (0<=i,j,k,l<=2, ij<=kl)
  //  pDijkl -> pKij + pKkl <eta_{kl}^{->}>  (0<=i,j,k,l<=2, ij<=kl)
  //
  //---------------------------------------------------------------------------

  for (int i=0; i < 3; i++)
  {     
    for (int j=0; j < 3; j++)
    {
      for (int k=0; k < 3; k++)
      {     
        for (int l=0; l < 3; l++)
        {
	  if ( (i<k) || (i==k && j<l) )
          {
            //==================================================
	    // Phosphorylation eta A
	    //==================================================
            lstOfReactants.clear();
            lstOfProducts.clear();
            oss.str("");

            oss << "Phosphorylation_eta_A_" << i << j << k << l;
            const string rid_a(oss.str());

            //-------------------------      

            oss.str("");
            oss << "D_" << i << j << k << l;
            lstOfReactants.push_back(oss.str());
  
            //-------------------------      

            oss.str("");
            oss << "pK_" << i << j;
            lstOfProducts.push_back(oss.str());

            oss.str("");
            oss << "K_" << k << l;
            lstOfProducts.push_back(oss.str());
	    
            // ------------------------------

            oss.str("");
            oss << "etaFwd_" << i << j;
            const string parameterId_a(oss.str());

            //-------------------------      
    
            createReaction(model, rid_a, lstOfReactants, lstOfProducts, parameterId_a);


            //==================================================
	    // Phosphorylation eta B
	    //==================================================

            lstOfProducts.clear();
            oss.str("");

            oss << "Phosphorylation_eta_B_" << i << j << k << l;
            const string rid_b(oss.str());

            //-------------------------      

            oss.str("");
            oss << "K_" << i << j;
            lstOfProducts.push_back(oss.str());

            oss.str("");
            oss << "pK_" << k << l;
            lstOfProducts.push_back(oss.str());
	    
            // ------------------------------

            oss.str("");
            oss << "etaFwd_" << k << l;
            const string parameterId_b(oss.str());

            //-------------------------      
    
            createReaction(model, rid_b, lstOfReactants, lstOfProducts, parameterId_b);
          }

          // ==================================================
          // Phosphorylation eta C
          // ==================================================

          lstOfReactants.clear();
          lstOfProducts.clear();
          oss.str("");

          oss << "Phosphorylation_eta_C_" << i << j << k << l;
          const string rid_c(oss.str());

          //-------------------------      

          oss.str("");
          oss << "pD_" << i << j << k << l;
          lstOfReactants.push_back(oss.str());

          //-------------------------      

          oss.str("");
          oss << "pK_" << i << j;
          lstOfProducts.push_back(oss.str());

          oss.str("");
          oss << "pK_" << k << l;
          lstOfProducts.push_back(oss.str());
	    
          // ------------------------------

          oss.str("");
          oss << "etaFwd_" << k << l;
          const string parameterId_c(oss.str());

          //-------------------------      
  
          createReaction(model, rid_c, lstOfReactants, lstOfProducts, parameterId_c);

        }
      }
    }
  }
}


/**
 *
 * createEventsOfSpikeTrain
 *
 *  20uM Ca Spike Train
 *
 *  (1) 2.00 - 2.05 (time)
 *  (2) 2.50 - 2.55 (time)
 *  (3) 3.00 - 3.05 (time)
 *  (4) 3.50 - 3.55 (time)
 *  (5) 4.00 - 4.05 (time)
 *
 */
void createEventsOfSpikeTrain(Model *model)
{
  Event  *ev;
  Trigger *trigger;
  EventAssignment *eva;

  const int spikeNum = 5;
  const double spikeStartTime  = 2.00;
  const double spikeOnPeriod   = 0.05;
  const double spikeOffPeriod  = 0.45;
  double time = spikeStartTime;
  
  for (int i=0; i < spikeNum; i++)
  {
    //
    // Creates an event for SpikeOnPeriod
    //     
    ev = model->createEvent();
    trigger = ev->createTrigger();
    trigger->setMath(createMathInTriggerBasedOnTime(time));
    eva = ev->createEventAssignment();
    eva->setVariable("Ca");
    eva->setMath(SBML_parseFormula("CaSpikeOn"));

    time += spikeOnPeriod;
    
    //
    // Creates an event for SpikeOffPeriod
    //     
    ev = model->createEvent();
    trigger = ev->createTrigger();
    trigger->setMath(createMathInTriggerBasedOnTime(time));
    eva = ev->createEventAssignment();
    eva->setVariable("Ca");
    eva->setMath(SBML_parseFormula("CaSpikeOff"));

    time += spikeOffPeriod;
  }    
}


//--------------------------------------------------------------------------------------
//
// Helper functions for creating an SBase element (Parameter, Reaction, Species and etc.)
//
//--------------------------------------------------------------------------------------

/**
 *
 * createGlobalParameter
 *
 */
Parameter*
createGlobalParameter(Model *model, const string& gparaid, double value, 
                      const string& unitid)
{
  Parameter* gpara = model->createParameter();
  gpara->setId(gparaid);
  if (!unitid.empty())
    gpara->setUnits(unitid);
  gpara->setValue(value);

  return gpara;
}


/**
 *
 * createSpecies
 *
 */
Species* 
createSpecies(Model *model, const string& compid, const string& spid,
              double initialamount, const string& unitid)
{
  Species *sp = model->createSpecies();
  sp->setCompartment(compid);
  sp->setId(spid);
  sp->setName(spid);
  sp->setHasOnlySubstanceUnits(true);
  if (!unitid.empty())
    sp->setUnits(unitid);
  sp->setInitialAmount(initialamount);
  
  return sp;
}


/**
 *
 * createReaction
 *
 */
Reaction* 
createReaction(Model *model, const string& reactionId, 
               const list<string> &lstOfReactant,
               const list<string> &lstOfProduct, 
               const string &parameterId)
{
  Reaction *reaction = model->createReaction();
  reaction->setId(reactionId);
  reaction->setReversible(false);

  // Creates one or more Reactant objects.

  list<string>::const_iterator itString = lstOfReactant.begin();
  while (itString != lstOfReactant.end())
  {
    SpeciesReference *spr = reaction->createReactant();
    spr->setSpecies(*itString);
    ++itString;
  }    

  // Creates one or more Product objects.

  itString = lstOfProduct.begin();
  while (itString != lstOfProduct.end())
  {
    SpeciesReference *spr = reaction->createProduct();
    spr->setSpecies(*itString);
    ++itString;
  }    

  // Creates a KineticLaw object inside the Reaction object 

  KineticLaw *kl = reaction->createKineticLaw();

  // Creates an ASTNode object with the given double value.

  //ASTNode astMath;
  //astMath.setName(parameterId.c_str());

  // set the Math element
  
  ostringstream mathString;

  mathString << parameterId.c_str();

  itString = lstOfReactant.begin();
  while (itString != lstOfReactant.end())
  {
     mathString << '*' << *itString;
     ++itString;
  }    
  
  kl->setMath(SBML_parseFormula(mathString.str().c_str()));   
  //kl->setMath(&astMath);   

  return reaction;
}


ASTNode* 
createMathInTriggerBasedOnTime(double time)
{
  ostringstream oss;
   
  const string csymbolTime = 
     "<csymbol encoding=\"text\" definitionURL=\"http://www.sbml.org/sbml/symbols/time\">"
     " t "
     "</csymbol>";     

  oss << "<math xmlns=\"http://www.w3.org/1998/Math/MathML\">"
      << "  <apply>"
      << "    <geq/>"
      << csymbolTime 
      << "    <cn>" << time << "</cn>"
      << "  </apply>"
      << "</math>";

  return readMathMLFromString(oss.str().c_str());
}


//---------------------------------------------------------------------
//
// Helper functions for validating and writing the given SBMLDocument.
//
//----------------------------------------------------------------------

/**
 *  
 *  Validates the given SBMLDocument.
 *
 *   This function is based on validateSBML.cpp implemented by
 *   Sarah Keating, Ben Bornstein, and Michael Hucka.
 *
 */
bool validateSBML (SBMLDocument* sbmlDoc)
{
  if (!sbmlDoc)
  {
    cerr << "validateExampleSBML: given a null SBML Document" << endl;
    return false;
  }
 
  string consistencyMessages;
  string validationMessages;
  bool noProblems                     = true;
  unsigned int numCheckFailures       = 0;
  unsigned int numConsistencyErrors   = 0;
  unsigned int numConsistencyWarnings = 0;
  unsigned int numValidationErrors    = 0;
  unsigned int numValidationWarnings  = 0;

  // LibSBML 3.3 is lenient when generating models from scratch using the
  // API for creating objects.  Once the whole model is done and before it
  // gets written out, it's important to check that the whole model is in
  // fact complete, consistent and valid.

  numCheckFailures = sbmlDoc->checkInternalConsistency();
  if ( numCheckFailures > 0 )
  {
    noProblems = false;
    for (unsigned int i = 0; i < numCheckFailures; i++)
    {
      const SBMLError* sbmlErr = sbmlDoc->getError(i);
      if ( sbmlErr->isFatal() || sbmlErr->isError() )
      {
        ++numConsistencyErrors;
      }
      else
      {
        ++numConsistencyWarnings;
      }      
    } 
    ostringstream oss;
    sbmlDoc->printErrors(oss);
    consistencyMessages = oss.str(); 
  }

  // If the internal checks fail, it makes little sense to attempt
  // further validation, because the model may be too compromised to
  // be properly interpreted.

  if (numConsistencyErrors > 0)
  {
    consistencyMessages += "Further validation aborted."; 
  }
  else
  {
    numCheckFailures = sbmlDoc->checkConsistency();
    if ( numCheckFailures > 0 )
    {
      noProblems = false;
      for (unsigned int i = 0; i < numCheckFailures; i++)
      {
        const SBMLError* sbmlErr = sbmlDoc->getError(i);
        if ( sbmlErr->isFatal() || sbmlErr->isError() )
        {
          ++numValidationErrors;
        }
        else
        {
          ++numValidationWarnings;
        }      
      } 
      ostringstream oss;
      sbmlDoc->printErrors(oss);
      validationMessages = oss.str(); 
    }
  }

  if (noProblems)
    return true;
  else
  {
    if (numConsistencyErrors > 0)
    {
      cout << "ERROR: encountered " << numConsistencyErrors 
           << " consistency error" << (numConsistencyErrors == 1 ? "" : "s")
	   << " in model '" << sbmlDoc->getModel()->getId() << "'." << endl;
    }
    if (numConsistencyWarnings > 0)
    {
      cout << "Notice: encountered " << numConsistencyWarnings
           << " consistency warning" << (numConsistencyWarnings == 1 ? "" : "s")
	   << " in model '" << sbmlDoc->getModel()->getId() << "'." << endl;
    }
    cout << endl << consistencyMessages;

    if (numValidationErrors > 0)
    {
      cout << "ERROR: encountered " << numValidationErrors
           << " validation error" << (numValidationErrors == 1 ? "" : "s")
	   << " in model '" << sbmlDoc->getModel()->getId() << "'." << endl;
    }
    if (numValidationWarnings > 0)
    {
      cout << "Notice: encountered " << numValidationWarnings
           << " validation warning" << (numValidationWarnings == 1 ? "" : "s")
	   << " in model '" << sbmlDoc->getModel()->getId() << "'." << endl;
    }
    cout << endl << validationMessages;

    return (numConsistencyErrors == 0 && numValidationErrors == 0);
  }
}


/**
 *
 * Writes the given SBMLDocument to the given file.
 *
 */ 
bool writeSBML(const SBMLDocument* sbmlDoc, const string& filename)
{
  SBMLWriter sbmlWriter;

  bool result = sbmlWriter.writeSBML(sbmlDoc, filename);

  if (result)
  {
    cout << "Wrote file \"" << filename << "\"" << endl;
    return true;
  }
  else
  {
    cerr << "Failed to write \"" << filename << "\"" << endl;
    return false;
  }
}
