/**
 * SBML-SSAlib bridge: a bridge library to connect SSAlib and SBML
 *
 * Copyright (C) 2004-2009, SBML-SSA Team, Keio/Seikei University.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *                    
 * @file    SSAPtoSBMLDConverter.cpp
 * @brief   SSA Problem -> SBML Document converter
 * @author  Keio/Seikei SBML-SSA Team.
 */

#include "SSAPtoSBMLDConverter.h"

/**
 * Constructor
 */   
SSAPtoSBMLDConverter::SSAPtoSBMLDConverter() {}
 
/**
 * Destructor
 */
SSAPtoSBMLDConverter::~SSAPtoSBMLDConverter() {}

/**
 * convert
 */
SBMLDocument*
SSAPtoSBMLDConverter::convert(SSAProblem *problem, int level, int version)
{
  SBMLDocument *sbmlDocument = new SBMLDocument(level, version);

  //------------------------------------------------------------
  //Creates a Model object inside the SBMLDocument object.
  //------------------------------------------------------------
  
  Model* model = sbmlDocument->createModel();
  model->setId("stochastic_model");//?
  
  //------------------------------------------------------------
  //Creates a Compartment object inside the Model object.
  //------------------------------------------------------------

  Compartment* comp;
  
  comp = model->createCompartment();
  comp->setId("default");
  comp->setSize(1);
  
  //------------------------------------------------------------
  //Creates Reaction objects inside the Model object.
  //------------------------------------------------------------

  SSAReaction** reactions = problem->reactions;
  int num_parameters = 0;
  int num_reactions  = 0;
  ostringstream reacString;
  ostringstream unitString;  
  map<string, int> mapOfUnit;
  map<string, int>::iterator itp;
  int unit_exp;
  map<char*, double> mapOfSpecies;
  map<char*, double>::iterator its;
  
  while (*reactions)
  {
    // unit definition
    unitString.str("");
    unit_exp = 1;
    
    // Creates a Reaction object in the Model object
    Reaction* reaction = model->createReaction();
    reacString.str("");
    reacString << 'r' << num_reactions;
    ++num_reactions;
    reaction->setId(reacString.str().c_str());
		    
    // Creates a KineticLaw object in the Reaction object
    KineticLaw* kl = reaction->createKineticLaw();
    ostringstream mathString;
    
    mathString.str("");
    
    // Creates a Parameter object in the KineticLaw object
    Parameter* param;
    ostringstream parameter_id;

    param = kl->createParameter();
    parameter_id.str("");
    parameter_id << "k" << num_parameters;
    ++num_parameters;
    param->setId(parameter_id.str().c_str());
    param->setValue((*reactions)->propensity);

    mathString << parameter_id.str();
    
    // Creates reactants for reaction
    Term** terms = (*reactions)->reactants;
    
    while (*terms)
    {
      SpeciesReference* spr = reaction->createReactant();
      char* spid = (*terms)->participant->name;
      spr->setSpecies(spid);
      spr->setStoichiometry((*terms)->coefficient);

      // ast node
      mathString << '*' << spid;
      
      // unit definition
      if ( unit_exp < 1 )
	unitString << "per_item_";
      --unit_exp;

      // species
      its = mapOfSpecies.find(spid);
      if ( its == mapOfSpecies.end() )
	mapOfSpecies.insert(pair<char*, double>(spid,0));
      
      ++terms;
    }

    // Creates products for reaction
    terms = (*reactions)->products;
    while (*terms)
    {
      SpeciesReference* spr = reaction->createProduct();
      char* spid = (*terms)->participant->name;
      spr->setSpecies(spid);
      spr->setStoichiometry((*terms)->coefficient);

      // species
      its = mapOfSpecies.find(spid);
      if ( its == mapOfSpecies.end() )
	mapOfSpecies.insert(pair<char*, double>(spid,-1));

      ++terms;
    }

    // Creates an ASTNode object
    kl->setMath(SBML_parseFormula(mathString.str().c_str()));

    // insert unit definition id to map object
    unitString << "per_second";
    string unitId = unitString.str();
    itp = mapOfUnit.find(unitId);
    if ( itp == mapOfUnit.end() )
      mapOfUnit.insert(pair<string, int>(unitId,unit_exp));

    // set unit definition to parameter object
    param->setUnits(unitId.c_str());
    
    ++reactions;
  }

  //------------------------------------------------------------
  //Creates UnitDefinition objects inside the Model object.
  //------------------------------------------------------------

  UnitDefinition* unitdef;
  Unit* unit;

  unitdef = model->createUnitDefinition();
  unitdef->setId("substance");
  unitdef->setName("substance_item");

  unit = unitdef->createUnit();
  unit->setKind(UNIT_KIND_ITEM);
  
  for ( itp = mapOfUnit.begin(); itp != mapOfUnit.end(); itp++ ) {
    string id = itp->first;
    int exp   = itp->second;
    
    unitdef = model->createUnitDefinition();
    unitdef->setId(id.c_str());
    unitdef->setName(id.c_str());

    if ( exp != 0 ) {
      unit = unitdef->createUnit();
      unit->setKind(UNIT_KIND_ITEM);
      unit->setExponent(exp);
    }
    
    unit = unitdef->createUnit();
    unit->setKind(UNIT_KIND_SECOND);
    unit->setExponent(-1);
  }
  
  //------------------------------------------------------------
  //Creates Species objects inside the Model object.
  //------------------------------------------------------------

  Species* sp;
  const char* compid = comp->getId().c_str();
  InitialCondition** icl = problem->initialConditions;
  
  while (*icl)
  {
    char* spid = (*icl)->participant->name;
    double ia = static_cast<double>((*icl)->population);

    its = mapOfSpecies.find(spid);
    if ( its != mapOfSpecies.end() )
      its->second = ia;
    else
      cerr << "species(" << spid << ") not found." << endl; // this should not happen.
    
    ++icl;
  }

  for ( its = mapOfSpecies.begin(); its != mapOfSpecies.end(); its++ ) {
    char* spid = its->first;
    double ia  = its->second;

    sp = model->createSpecies();
    sp->setHasOnlySubstanceUnits(true);
    sp->setCompartment(compid);
    sp->setId(spid);
    sp->setName(spid);
    if ( ia >= 0 )
      sp->setInitialAmount(ia);
  }
  
  return sbmlDocument;
}
