`include "mydefinition.h"

module lcd( clk, rst, 
	    lcdEnable, 
	    lcdRegisterSelect, 
	    lcdReadWrite,
	    command,
	    commandEnable,
	    busy,
	    dataBitHigh, dataBitLow);

   input clk, rst;
   input [`LCDCOMMAND] command;
   input 	       commandEnable;
   
   output lcdEnable;          // 0:disable
   output lcdRegisterSelect;  // 0:WRITE->instructionReg,READ->BusyFlash 
                              // 1,select read or write Register
   output lcdReadWrite;       // 0:receive data 1:display data
   output [`LCDDATA] dataBitLow;
   output 	     busy;

   inout [`LCDDATA]  dataBitHigh;
   
   reg 		     lcdEnable;
   reg 		     lcdRegisterSelect;
   reg 		     lcdReadWrite;
   reg [`LCDDATA]    dataBitHighRegister;
   reg 		     busy;

   reg 		     decodeEnable;
   reg [`LCDDATA_RSRW] storedDecodeData;
  
   wire [`LCDPULSE] pulseCounter;
   wire 	     secondSignal;
   wire 	     secondPulse;
   wire 	     thirdSignal;
   wire 	     thirdPulse;
   wire 	     fourthSignal;
   wire 	     fourthPulse;
   wire 	     initializeEnd;

   wire 	     storingCommandEnable;
   wire [`LCDDATA_RSRW] decodeData;
   wire 		decodeDataEnable;
   
   assign 	     dataBitLow = ( 1'b1) ? 4'b1111 : 4'bzzzz;
   assign 	     dataBitHigh = ( ~lcdReadWrite) ? dataBitHighRegister :
		                                      4'bzzzz;

   lcdPulseCounter LPC( .clk( clk),
			.rst( rst),
			.pulseCounter( pulseCounter),
			.firstPulse( firstPulse),
			.secondPulse( secondPulse),
			.thirdPulse( thirdPulse),
			.fourthPulse( fourthPulse),
			.decodeDataEnable( decodeDataEnable));

   lcdInitializer LINIT( .clk( clk),
			 .rst( rst),
			 .pulseCounter( pulseCounter),
			 .firstSignal( firstSignal),
			 .secondSignal( secondSignal),
			 .thirdSignal( thirdSignal),
			 .fourthSignal( fourthSignal),
			 .fifthSignal( initializeEnd),
			 .firstPulse( firstPulse),
			 .secondPulse( secondPulse),
			 .thirdPulse( thirdPulse),
			 .fourthPulse( fourthPulse));

   lcdDecoder LDEC( .clk( clk),
		    .rst( rst),
		    .command( command),
		    .commandEnable( commandEnable),

		    .storingCommandEnable( storingCommandEnable),
		    .data( decodeData),
		    .dataEnable( decodeDataEnable)
		    );
   
   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 dataBitHighRegister <= `LCD_DATA_ZERO;
      end else begin
	 if ( decodeDataEnable) begin
	    dataBitHighRegister <= decodeData[3:0];
	 end else if ( decodeEnable & pulseCounter < `LCD_PULSE_MAX) begin
	    dataBitHighRegister <= storedDecodeData[3:0];
	 end else if ( fourthSignal & ( pulseCounter < `LCD_PULSE_MAX |
					fourthPulse)) begin
	    dataBitHighRegister <= `LCD_DATA_TWO; 
	 end else if ( ( firstSignal & ( pulseCounter < `LCD_PULSE_MAX |
					firstPulse)) |
		       ( secondSignal & ( pulseCounter < `LCD_PULSE_MAX |
					  secondPulse)) |
		       ( thirdSignal & ( pulseCounter < `LCD_PULSE_MAX |
					 thirdPulse))
		       ) begin
	    dataBitHighRegister <= `LCD_DATA_THREE;
	 end else begin
	    dataBitHighRegister <= `LCD_DATA_ZERO;
	 end
      end
   end

   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 lcdEnable <= `LOW;
      end else begin
	 if ( ( pulseCounter >= `LCD_PULSE_START & 
		pulseCounter <= `LCD_PULSE_END) &
	      ( ( firstSignal | secondSignal) |
		( thirdSignal | fourthSignal))
	      ) begin
	    lcdEnable <= `HIGH;
	 end else begin
	    lcdEnable <= `LOW;
	 end
      end
   end

   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 lcdRegisterSelect <= `LOW;
      end else begin
	  if ( decodeDataEnable) begin
	     lcdRegisterSelect <= decodeData[5];
	  end else if ( decodeEnable & pulseCounter <`LCD_PULSE_MAX) begin 
	     lcdRegisterSelect <= storedDecodeData[5];
	  end else if ( ( pulseCounter < `LCD_PULSE_MAX |
			  ( firstPulse | secondPulse) |
			  ( thirdPulse | fourthPulse)
			  ) &
			(
			 ( firstSignal | secondSignal) |
			 ( thirdSignal | fourthSignal) 
			 )
			) begin
	     lcdRegisterSelect <= `HIGH;
	  end else begin
	     lcdRegisterSelect <= `LOW;
	  end
      end
   end // always @ ( posedge clk or posedge rst)

   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 lcdReadWrite <= `HIGH;
      end else begin
	 if ( decodeDataEnable) begin
	    lcdReadWrite <= decodeData[4];
	 end else if ( decodeEnable & pulseCounter < `LCD_PULSE_MAX) begin 
	    lcdReadWrite <= storedDecodeData[4];
	 end else if ( ( pulseCounter < `LCD_PULSE_MAX |
			 ( firstPulse | secondPulse) |
			 ( thirdPulse | fourthPulse)
			 ) &
		       (
			( firstSignal | secondSignal) |
			( thirdSignal | fourthSignal) 
			)
		       ) begin
	    lcdReadWrite <= `LOW;
	 end else begin
	    lcdReadWrite <= `HIGH;
	 end
      end
   end
   
   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 busy <= `HIGH;
      end else begin
	 if ( initializeEnd & ~storingCommandEnable) begin
	    busy <= `LOW;
	 end else begin
	    busy <= `HIGH;
	 end
      end
   end

   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 decodeEnable <= `LOW;
      end else begin
	 if ( decodeEnable) begin
	    if ( pulseCounter ==  `LCD_PULSE_MAX - `LCD_PULSE_ONE) begin
	       decodeEnable <= `LOW;
	    end else begin
	       decodeEnable <= decodeEnable;
	    end
	 end else begin
	    decodeEnable <= decodeDataEnable;
	 end
      end
   end

   always @ ( posedge clk or posedge rst) begin
      if ( rst == `ENABLE) begin
	 storedDecodeData <= `LCD_DATA_RSRW_ZERO;
      end else begin
	 if ( decodeEnable) begin
	    if ( pulseCounter ==  `LCD_PULSE_MAX - `LCD_PULSE_ONE) begin
	       storedDecodeData <= `LCD_DATA_RSRW_ZERO;
	    end else begin
	       storedDecodeData <= storedDecodeData;
	    end
	 end else begin
	    storedDecodeData <= decodeData;
	 end
      end
   end
   
endmodule // lcd
