/*
 * <<< torus_3d_ecube_router.h >>>
 *
 * --- Router class for 3D-torus, with ecube routing
 *     Copyright (C) 2000-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TORUS_3D_ECUBE_ROUTER_H
#define TORUS_3D_ECUBE_ROUTER_H

#include <cstddef>
#include <isis/isis.h>

template <class P>
class torus_3d_ecube_router : public deterministic_router<P>
{
private:
	typedef torus_3d_ecube_router<P> thisclass;
	typedef deterministic_router<P> inherited;
public:
	typedef typename inherited::packet_type packet_type;
	typedef typename inherited::node_address_type node_address_type;
private:
	const static size_t physical_channel_size = 7;
	const static size_t virtual_channel_size = 2;
	const static size_t PLUS_X_PH_CHANNEL  = 0;
	const static size_t MINUS_X_PH_CHANNEL = 1;
	const static size_t PLUS_Y_PH_CHANNEL  = 2;
	const static size_t MINUS_Y_PH_CHANNEL = 3;
	const static size_t PLUS_Z_PH_CHANNEL  = 4;
	const static size_t MINUS_Z_PH_CHANNEL = 5;
	const static size_t COMSUMP_PH_CHANNEL = 6;
	const static size_t NORMAL_VR_CHANNEL = 0;
	const static size_t AROUND_VR_CHANNEL = 1;
	size_t x_size_, y_size_, z_size_;
protected:
	typedef typename inherited::channel_controller channel_controller;
	virtual void routing(const packet_type&, channel_controller&);
public:
	torus_3d_ecube_router(void);
	torus_3d_ecube_router(const thisclass&);
	virtual ~torus_3d_ecube_router() {}
	void set_size(size_t a, size_t b, size_t c)
		{ x_size_ = a, y_size_ = b, z_size_ = c; }
};

template <class P>
torus_3d_ecube_router<P>::torus_3d_ecube_router(void)
	: inherited(),
	  x_size_(0),
	  y_size_(0),
	  z_size_(0)
{
	set_input_size(physical_channel_size);
	set_output_size(physical_channel_size);
	for (size_t i = 0; i < physical_channel_size - 1; i++) {
		set_channel_size(i, virtual_channel_size);
		set_buffer_size(i, 1);
	}
	set_channel_size(COMSUMP_PH_CHANNEL, 1);
	set_buffer_size(COMSUMP_PH_CHANNEL, 1);
}

template <class P>
torus_3d_ecube_router<P>::torus_3d_ecube_router
	(const torus_3d_ecube_router<P>& a)
	: inherited(a),
	  x_size_(a.x_size_),
	  y_size_(a.y_size_),
	  z_size_(a.z_size_)
{}

template <class P>
void torus_3d_ecube_router<P>::routing
	(const typename torus_3d_ecube_router<P>::packet_type& pkt,
	 typename torus_3d_ecube_router<P>::channel_controller& ctl)
{
	const size_t normal_vr_channel = NORMAL_VR_CHANNEL; // to ignore error
	const size_t around_vr_channel = AROUND_VR_CHANNEL; // to ignore error
	if (pkt.destination() == node_address()) {
		// destination is here, transmit to PE
		ctl.set_destination(COMSUMP_PH_CHANNEL, normal_vr_channel);
	} else {
		// destination is not here, select channel with e-cube routing
		size_t my_x, my_y, my_z, dst_x, dst_y, dst_z,
			   curr_phch, curr_vrch, next_phch, next_vrch;
		my_x = node_address() % x_size_;
		my_y = (node_address() / x_size_) % y_size_;
		my_z = (node_address() / x_size_) / y_size_;
		dst_x = pkt.destination() % x_size_;
		dst_y = (pkt.destination() / x_size_) % y_size_;
		dst_z = (pkt.destination() / x_size_) / y_size_;
		curr_phch = ctl.physical_channel_id();
		curr_vrch = ctl.virtual_channel_id();
		if (my_x != dst_x) {
			switch (curr_phch) {
			case COMSUMP_PH_CHANNEL:
				if (my_x < dst_x) {
					if (dst_x - my_x <= my_x + x_size_ - dst_x) {
						next_phch = PLUS_X_PH_CHANNEL;
						next_vrch = ((my_x != x_size_ - 1) ?
							normal_vr_channel : around_vr_channel);
					} else {
						next_phch = MINUS_X_PH_CHANNEL;
						next_vrch = ((my_x != 0) ?
							normal_vr_channel : around_vr_channel);
					}
				} else {
					if (dst_x + x_size_ - my_x <= my_x - dst_x) {
						next_phch = PLUS_X_PH_CHANNEL;
						next_vrch = ((my_x != x_size_ - 1) ?
							normal_vr_channel : around_vr_channel);
					} else {
						next_phch = MINUS_X_PH_CHANNEL;
						next_vrch = ((my_x != 0) ?
							normal_vr_channel : around_vr_channel);
					}
				}
				break;
			case MINUS_X_PH_CHANNEL:
				next_phch = PLUS_X_PH_CHANNEL;
				if (curr_vrch == normal_vr_channel && my_x != x_size_ - 1) {
					next_vrch = normal_vr_channel;
				} else {
					next_vrch = around_vr_channel;
				}
				break;
			case PLUS_X_PH_CHANNEL:
				next_phch = MINUS_X_PH_CHANNEL;
				if (curr_vrch == normal_vr_channel && my_x != 0) {
					next_vrch = normal_vr_channel;
				} else {
					next_vrch = around_vr_channel;
				}
				break;
			default:
				assert(false);
				break;
			}
		} else if (my_y != dst_y) {
			switch (curr_phch) {
			case COMSUMP_PH_CHANNEL:
			case PLUS_X_PH_CHANNEL:
			case MINUS_X_PH_CHANNEL:
				if (my_y < dst_y) {
					if (dst_y - my_y <= my_y + y_size_ - dst_y) {
						next_phch = PLUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != y_size_ - 1) ?
							normal_vr_channel : around_vr_channel);
					} else {
						next_phch = MINUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != 0) ?
							normal_vr_channel : around_vr_channel);
					}
				} else {
					if (dst_y + y_size_ - my_y <= my_y - dst_y) {
						next_phch = PLUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != y_size_ - 1) ?
							normal_vr_channel : around_vr_channel);
					} else {
						next_phch = MINUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != 0) ?
							normal_vr_channel : around_vr_channel);
					}
				}
				break;
			case MINUS_Y_PH_CHANNEL:
				next_phch = PLUS_Y_PH_CHANNEL;
				if (curr_vrch == normal_vr_channel && my_y != y_size_ - 1) {
					next_vrch = normal_vr_channel;
				} else {
					next_vrch = around_vr_channel;
				}
				break;
			case PLUS_Y_PH_CHANNEL:
				next_phch = MINUS_Y_PH_CHANNEL;
				if (curr_vrch == normal_vr_channel && my_y != 0) {
					next_vrch = normal_vr_channel;
				} else {
					next_vrch = around_vr_channel;
				}
				break;
			default:
				assert(false);
				break;
			}
		} else {
			switch (curr_phch) {
			case COMSUMP_PH_CHANNEL:
			case PLUS_X_PH_CHANNEL:
			case MINUS_X_PH_CHANNEL:
			case PLUS_Y_PH_CHANNEL:
			case MINUS_Y_PH_CHANNEL:
				if (my_z < dst_z) {
					if (dst_z - my_z <= my_z + z_size_ - dst_z) {
						next_phch = PLUS_Z_PH_CHANNEL;
						next_vrch = ((my_z != z_size_ - 1) ?
							normal_vr_channel : around_vr_channel);
					} else {
						next_phch = MINUS_Z_PH_CHANNEL;
						next_vrch = ((my_z != 0) ?
							normal_vr_channel : around_vr_channel);
					}
				} else {
					if (dst_z + z_size_ - my_z <= my_z - dst_z) {
						next_phch = PLUS_Z_PH_CHANNEL;
						next_vrch = ((my_z != z_size_ - 1) ?
							normal_vr_channel : around_vr_channel);
					} else {
						next_phch = MINUS_Z_PH_CHANNEL;
						next_vrch = ((my_z != 0) ?
							normal_vr_channel : around_vr_channel);
					}
				}
				break;
			case MINUS_Z_PH_CHANNEL:
				next_phch = PLUS_Z_PH_CHANNEL;
				if (curr_vrch == normal_vr_channel && my_z != z_size_ - 1) {
					next_vrch = normal_vr_channel;
				} else {
					next_vrch = around_vr_channel;
				}
				break;
			case PLUS_Z_PH_CHANNEL:
				next_phch = MINUS_Z_PH_CHANNEL;
				if (curr_vrch == normal_vr_channel && my_z != 0) {
					next_vrch = normal_vr_channel;
				} else {
					next_vrch = around_vr_channel;
				}
				break;
			default:
				assert(false);
				break;
			}
		}
		ctl.set_destination(next_phch, next_vrch);
	}
}

#endif /* TORUS_3D_ECUBE_ROUTER_H */
