/*
 * <<< torus_2d_duato_router.h >>>
 *
 * --- Router class for 2D-torus, with Duato's protocol
 *     Copyright (C) 2000-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TORUS_2D_DUATO_ROUTER_H
#define TORUS_2D_DUATO_ROUTER_H

#include <cstddef>
#include <isis/isis.h>

template <class P>
class torus_2d_duato_router : public duato_router<P>
{
private:
	typedef torus_2d_duato_router<P> thisclass;
	typedef duato_router<P> inherited;
public:
	typedef typename inherited::packet_type packet_type;
	typedef typename inherited::node_address_type node_address_type;
private:
	const static size_t physical_channel_size = 5;
	const static size_t virtual_channel_size = 3;
	const static size_t PLUS_X_PH_CHANNEL  = 0;
	const static size_t MINUS_X_PH_CHANNEL = 1;
	const static size_t PLUS_Y_PH_CHANNEL  = 2;
	const static size_t MINUS_Y_PH_CHANNEL = 3;
	const static size_t COMSUMP_PH_CHANNEL = 4;
	const static size_t CF_VR_CHANNEL = 0;
	const static size_t CH_VR_CHANNEL = 1;
	const static size_t CA_VR_CHANNEL = 2;
	size_t x_size_, y_size_;
	bool check_all_resources_for_adaptive(size_t out_phch) {
		return (!is_locked_output_virtual_channel(out_phch, CF_VR_CHANNEL) &&
				!output_channel(out_phch).full(CF_VR_CHANNEL));
	}
	void allocate_all_resources_for_adaptive(size_t out_phch) {
		lock_output_virtual_channel(out_phch, CF_VR_CHANNEL);
	}
protected:
	typedef typename inherited::channel_controller channel_controller;
	virtual void adaptive_routing(packet_type&, channel_controller&);
	virtual void escape_routing(packet_type&, channel_controller&);
public:
	torus_2d_duato_router(void);
	torus_2d_duato_router(const thisclass&);
	virtual ~torus_2d_duato_router() {}
	void set_size(size_t a, size_t b) { x_size_ = a, y_size_ = b; }
};

template <class P>
torus_2d_duato_router<P>::torus_2d_duato_router(void)
	: inherited(),
	  x_size_(0),
	  y_size_(0)
{
	set_input_size(physical_channel_size);
	set_output_size(physical_channel_size);
	for (size_t i = 0; i < physical_channel_size - 1; i++) {
		set_channel_size(i, virtual_channel_size);
		set_buffer_size(i, 1);
	}
	set_channel_size(COMSUMP_PH_CHANNEL, 1);
	set_buffer_size(COMSUMP_PH_CHANNEL, 1);
}

template <class P>
torus_2d_duato_router<P>::torus_2d_duato_router
	(const torus_2d_duato_router<P>& a)
	: inherited(a),
	  x_size_(a.x_size_),
	  y_size_(a.y_size_)
{}

template <class P>
void torus_2d_duato_router<P>::adaptive_routing
	(typename torus_2d_duato_router<P>::packet_type& pkt,
	 typename torus_2d_duato_router<P>::channel_controller& ctl)
{
	if (pkt.destination() == node_address()) return;
	// destination is not here, select channel with duato routing
	size_t my_x, my_y, src_x, src_y, dst_x, dst_y;
	size_t x_distance, y_distance, dirx_phch = 0, diry_phch = 0;
	bool dirx_ok = false, diry_ok = false;
	my_x = node_address() % x_size_;
	my_y = node_address() / x_size_;
	src_x = pkt.source() % x_size_;
	src_y = pkt.source() / x_size_;
	dst_x = pkt.destination() % x_size_;
	dst_y = pkt.destination() / x_size_;
	// decide x direction
	if (my_x != dst_x) {
		if (src_x < dst_x) {
			if (dst_x - src_x <= src_x + x_size_ - dst_x) {
				dirx_phch = PLUS_X_PH_CHANNEL;
				x_distance = dst_x - src_x;
			} else {
				dirx_phch = MINUS_X_PH_CHANNEL;
				x_distance = src_x + x_size_ - dst_x;
			}
		} else {
			if (dst_x + x_size_ - src_x <= src_x - dst_x) {
				dirx_phch = PLUS_X_PH_CHANNEL;
				x_distance = dst_x + x_size_ - src_x;
			} else {
				dirx_phch = MINUS_X_PH_CHANNEL;
				x_distance = src_x - dst_x;
			}
		}
		dirx_ok = check_all_resources_for_adaptive(dirx_phch);
	} else {
		x_distance = 0;
	}
	// decide y direction
	if (my_y != dst_y) {
		if (src_y < dst_y) {
			if (dst_y - src_y <= src_y + y_size_ - dst_y) {
				diry_phch = PLUS_Y_PH_CHANNEL;
				y_distance = dst_y - src_y;
			} else {
				diry_phch = MINUS_Y_PH_CHANNEL;
				y_distance = src_y + y_size_ - dst_y;
			}
		} else {
			if (dst_y + y_size_ - src_y <= src_y - dst_y) {
				diry_phch = PLUS_Y_PH_CHANNEL;
				y_distance = dst_y + y_size_ - src_y;
			} else {
				diry_phch = MINUS_Y_PH_CHANNEL;
				y_distance = src_y - dst_y;
			}
		}
		diry_ok = check_all_resources_for_adaptive(diry_phch);
	} else {
		y_distance = 0;
	}
	// select CF channel(adaptive path) if available
	if (x_distance >= y_distance) {
		// selection order is: x-CF -> y-CF
		if (dirx_ok) {
			ctl.set_destination(dirx_phch, CF_VR_CHANNEL);
			allocate_all_resources_for_adaptive(dirx_phch);
			return;
		}
		if (y_distance > 0 && diry_ok) {
			ctl.set_destination(diry_phch, CF_VR_CHANNEL);
			allocate_all_resources_for_adaptive(diry_phch);
			return;
		}
	} else {
		// selection order is: y-CF -> x-CF
		if (diry_ok) {
			ctl.set_destination(diry_phch, CF_VR_CHANNEL);
			allocate_all_resources_for_adaptive(diry_phch);
			return;
		}
		if (x_distance > 0 && dirx_ok) {
			ctl.set_destination(dirx_phch, CF_VR_CHANNEL);
			allocate_all_resources_for_adaptive(dirx_phch);
			return;
		}
	}
}

template <class P>
void torus_2d_duato_router<P>::escape_routing
	(typename torus_2d_duato_router<P>::packet_type& pkt,
	 typename torus_2d_duato_router<P>::channel_controller& ctl)
{
	if (pkt.destination() == node_address()) {
		// destination is here, transmit to PE
		ctl.set_destination(COMSUMP_PH_CHANNEL, 0);
		return;
	}
	// destination is not here, select channel with duato routing
	size_t my_x, my_y, src_x, src_y, dst_x, dst_y;
	size_t x_distance, y_distance;
	size_t dirx_phch = 0, dirx_vrch = 0, diry_phch = 0, diry_vrch = 0;
	my_x = node_address() % x_size_;
	my_y = node_address() / x_size_;
	src_x = pkt.source() % x_size_;
	src_y = pkt.source() / x_size_;
	dst_x = pkt.destination() % x_size_;
	dst_y = pkt.destination() / x_size_;
	// decide x direction
	if (my_x != dst_x) {
		if (src_x < dst_x) {
			if (dst_x - src_x <= src_x + x_size_ - dst_x) {
				dirx_phch = PLUS_X_PH_CHANNEL;
				dirx_vrch = CH_VR_CHANNEL;
				x_distance = dst_x - src_x;
			} else {
				dirx_phch = MINUS_X_PH_CHANNEL;
				dirx_vrch = CA_VR_CHANNEL;
				x_distance = src_x + x_size_ - dst_x;
			}
		} else {
			if (dst_x + x_size_ - src_x <= src_x - dst_x) {
				dirx_phch = PLUS_X_PH_CHANNEL;
				dirx_vrch = CA_VR_CHANNEL;
				x_distance = dst_x + x_size_ - src_x;
			} else {
				dirx_phch = MINUS_X_PH_CHANNEL;
				dirx_vrch = CH_VR_CHANNEL;
				x_distance = src_x - dst_x;
			}
		}
	} else {
		x_distance = 0;
	}
	// decide y direction
	if (my_y != dst_y) {
		if (src_y < dst_y) {
			if (dst_y - src_y <= src_y + y_size_ - dst_y) {
				diry_phch = PLUS_Y_PH_CHANNEL;
				diry_vrch = CH_VR_CHANNEL;
				y_distance = dst_y - src_y;
			} else {
				diry_phch = MINUS_Y_PH_CHANNEL;
				diry_vrch = CA_VR_CHANNEL;
				y_distance = src_y + y_size_ - dst_y;
			}
		} else {
			if (dst_y + y_size_ - src_y <= src_y - dst_y) {
				diry_phch = PLUS_Y_PH_CHANNEL;
				diry_vrch = CA_VR_CHANNEL;
				y_distance = dst_y + y_size_ - src_y;
			} else {
				diry_phch = MINUS_Y_PH_CHANNEL;
				diry_vrch = CH_VR_CHANNEL;
				y_distance = src_y - dst_y;
			}
		}
	} else {
		y_distance = 0;
	}
	// select CA,CH channel (escape path)
	if (my_x != dst_x) {
		ctl.set_destination(dirx_phch, dirx_vrch);
	} else {
		ctl.set_destination(diry_phch, diry_vrch);
	}
}

#endif /* TORUS_2D_DUATO_ROUTER_H */
