/*
 * <<< random_access_test.cc >>>
 *
 * --- Test program for cache_controller class
 *     Copyright (C) 2000-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <vector>
#include <iostream>
#include <isis/isis.h>
#include "cache_protocol.h"
#include "cache_controller.h"
#include "big_word.h"
#include "bus_bridge.h"

using namespace std;

typedef unsigned int word;
typedef cache_line_PROTOCOL<word, word> cache_line_t;
typedef set_associative_cache<cache_line_t> cache_t;
typedef cache_controller<word> cache_ctl_t;
typedef big_word<word> bus_data_t;

bool VerboseMode = false;

int main(int argc, char** argv)
{
	const word top = 0, size = 0x80;
	const word cache_size = 0x40, line_size = 0x10, way_size = 2;
	const int punum = 2;
	const int iter_count = (argc > 1) ? atoi(argv[1]) : 100;
	random_access_processor<word> pu[punum];
	cache_ctl_t cache_ctl[punum];
	memory_control_unit<word> mem_ctl;
	std::vector<cache_t> cache_buf(punum, cache_t(4, 4, 1));
	arbiter sender(1, punum);
	mapped_memory<word, word> mem_buf(top, size);
	bus_port<word> lobus_if[punum], exbus_if;
	bus_port<bus_data_t> shbus_if;
	bus_bridge<word> bridge;
	for (int i = 0; i < punum; i++) {
		cache_buf[i].resize(cache_size, line_size, way_size);
		pu[i].set_access_probability(.4);
		pu[i].set_read_probability(.8);
		pu[i].set_address(mem_buf.top(), mem_buf.size());
		cache_ctl[i].set_shared_address(mem_buf.top(), mem_buf.size());
		cache_ctl[i].connect_cache(cache_buf[i]);
		pu[i].port_ref().connect(lobus_if[i]);
		cache_ctl[i].connect_local_bus_port(lobus_if[i]);
		cache_ctl[i].connect_shared_bus_port(shbus_if);
		cache_ctl[i].connect_sender_arb(sender);
		cache_ctl[i].connect_mem(&mem_buf, &mem_buf);
		cache_ctl[i].connect_mem_port(exbus_if);
	}
	mem_ctl.connect_memory(mem_buf);
	mem_ctl.port_ref().connect(exbus_if);
	bridge.connect_external_bus_port(exbus_if);
	bridge.connect_internal_bus_port(shbus_if);
	for (int iter = 0, break_count = 0; ; iter++) {
		if (VerboseMode) {
			cerr << "--- clk:" << iter << " ---" << endl;
		}
		int i;
		if (iter == iter_count) {
			for (i = 0; i < punum; i++) {
				pu[i].set_access_probability(0);
			}
		}
		if (VerboseMode) {
			for (i = 0; i < punum; i++) {
				cerr << ((i == 0) ? "in:  " : "     ")
					 << "pu" << i << ":    " << pu[i] << endl
					 << "     lobus" << i << ": " << lobus_if[i] << endl
					 << "     cache" << i << ": " << cache_ctl[i] << endl;
			}
			cerr << "     shbus:  " << shbus_if << endl
				 << "     exbus:  " << exbus_if << endl
				 << "    bridge:  " << bridge   << endl;
		}
		for (i = 0; i < punum; i++) {
			pu[i].clock_in();
			cache_ctl[i].clock_in();
		}
		bridge.clock_in();
		mem_ctl.clock_in();
		if (iter >= iter_count) {
			int flag = 1;
			for (i = 0; i < punum; i++) {
				if (!lobus_if[i].is_ready() || lobus_if[i].is_owned()) {
					flag = 0;
					break;
				}
			}
			if (!shbus_if.is_ready() || shbus_if.is_owned()) flag = 0;
			if (flag) {
				break_count++;
			} else {
				break_count = 0;
			}
		}
		if (VerboseMode) {
			for (i = 0; i < punum; i++) {
				cerr << ((i == 0) ? "out: " : "     ")
					 << "pu" << i << ":    " << pu[i] << endl
					 << "     lobus" << i << ": " << lobus_if[i] << endl
					 << "     cache" << i << ": " << cache_ctl[i] << endl;
			}
			cerr << "     shbus:  " << shbus_if << endl
				 << "     exbus:  " << exbus_if << endl
				 << "    bridge:  " << bridge   << endl;
		}
		for (i = 0; i < punum; i++) {
			pu[i].clock_out();
			cache_ctl[i].clock_out();
		}
		bridge.clock_out();
		mem_ctl.clock_out();
		if (break_count >= 5) break;
	}
	cout << mem_buf << endl;
	for (int i = 0; i < punum; i++) {
		cout << "cache" << i << ':' << endl << cache_buf[i] << endl;
	}
	return 0;
}
