/*
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2001 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <isis/isis.h>
#include "snoop_cache.h"

using namespace std;

const r3000_word local_memory_address  = 0x00000000;
const r3000_word local_memory_size	   = 0x20000000;
const r3000_word shared_memory_address = 0xa0000000;
const r3000_word shared_memory_size	   = 0x10000000;
const r3000_word sync_memory_address   = 0xb0000000;
const r3000_word sync_memory_size	   = 0x08000000;

bool PrintStatus = 0;
bool VerboseMode = false;
int DebugLevel = -1;

static const char* to_basename(const char* s)
{
	const char* head = s;
	while (*s != '\0') {
		if (*s == '/') head = s + 1;
		s++;
	}
	return head;
}

void usage(const char* name)
{
	cout << "Usage: " << name
		 << " <target> [options] [-- [target options]]" << endl
		 << "Options:" << endl
		 << "  -c,    --clock         Print execution time in clock"  << endl
		 << "  -d<n>, --delay=<n>     Set shared memory access delay" << endl
		 << "  -g,    --gdb           Use gdb mode"					  << endl
		 << "  -h,    --help          Print this message and exit"	  << endl
		 << "  -i,    --info          Print status"					  << endl
		 << "  -l<n>, --linesize=<n>  Set cache line size to n words" << endl
		 << "  -m,    --monitor       Use monitor mode"				  << endl
		 << "  -p<n>, --processor=<n> Set number of processors"		  << endl
		 << "  -s<n>, --cachesize=<n> Set cache size to n kbytes"	  << endl
		 << "  -t,    --trace         Enable address trace output"	  << endl
		 << "  -v,    --verbose       Verbose mode"					  << endl
		 << "  -w<n>, --way=<n>       Set cache way size"			  << endl
		 << "  -z<n>                  Set debug level"				  << endl
		 << "  --udp                  Use gdb mode with udp"		  << endl
		 << "  --stdin=<file>         Set standard input file"		  << endl
		 << "  --stdout=<file>        Set standard output file"		  << endl
		 << "  --stderr=<file>        Set standard error file"		  << endl;
}

void version(const char* name)
{
	cout << name << " - " << isis::version_string() << endl;
}

int main(int, char** argv)
{
	typedef snoop_cache architecture_type;
	typedef snoop_cache_element processing_element_type;
	int cache_size = 256, line_size = 16, way_size = 2;
	bool clock_flag = false, trace_flag = false, cui_monitor_flag = false,
		 gui_monitor_flag = false, gdb_flag = false, udp_flag = false;
	int punum = 1, delay = 0;
	argument_parser arg((const char* const*)(argv + 1));
	if (arg.defined('c') || arg.defined("clock"))
		clock_flag = true;
	if (arg.defined("gui"))
		gui_monitor_flag = true;
	if (arg.defined('g') || arg.defined("gdb"))
		gdb_flag = true;
	if (arg.defined('h') || arg.defined("help"))
		{ usage(to_basename(argv[0])); return 0; }
	if (arg.defined('i') || arg.defined("info"))
		PrintStatus = true;
	if (arg.defined('m') || arg.defined("monitor"))
		cui_monitor_flag = true;
	if (arg.defined('t') || arg.defined("trace"))
		trace_flag = true;
	if (arg.defined("udp"))
		udp_flag = true;
	if (arg.defined('v') || arg.defined("verbose"))
		VerboseMode = true;
	if (arg['d'] != NULL)		  delay = atoi(arg['d']);
	if (arg["delay"] != NULL)	  delay = atoi(arg["delay"]);
	if (arg['l'] != NULL)		  line_size = atoi(arg['l']);
	if (arg["linesize"] != NULL)  line_size = atoi(arg["linesize"]);
	if (arg['p'] != NULL)		  punum = atoi(arg['p']);
	if (arg["processor"] != NULL) punum = atoi(arg["processor"]);
	if (arg['s'] != NULL)		  cache_size = atoi(arg['s']);
	if (arg["cachesize"] != NULL) cache_size = atoi(arg["cachesize"]);
	if (arg['w'] != NULL)		  way_size = atoi(arg['w']);
	if (arg["way"] != NULL)		  way_size = atoi(arg["way"]);
	if (arg['z'] != NULL)		  DebugLevel = atoi(arg['z']);
	architecture_type com(punum);
	{
		for (size_t i = 0; i < com.processor_number(); i++) {
			processing_element_type& pe(com.processing_element(i));
			pe.set_local_memory_area(local_memory_address, local_memory_size);
			pe.set_file_table_size(16);
		}
		com.set_shared_memory_area(shared_memory_address, shared_memory_size);
		com.set_sync_memory_area(sync_memory_address, sync_memory_size);
	}
	ifstream fin;
	ofstream fout, ferr;
	if (arg["stdin"] != NULL) {
		fin.open(arg["stdin"], ios::in);
		if (!fin) {
			cerr << "open error: " << arg["stdin"] << endl;
			return 1;
		}
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_input_stream(fin);
		}
	} else {
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_input_stream(cin);
		}
	}
	if (arg["stdout"] != NULL) {
		fout.open(arg["stdout"], ios::out);
		if (!fout) {
			cerr << "open error: " << arg["stdout"] << endl;
			return 1;
		}
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_output_stream(fout);
		}
	} else {
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_output_stream(cout);
		}
	}
	if (arg["stderr"] != NULL) {
		ferr.open(arg["stderr"], ios::out);
		if (!ferr) {
			cerr << "open error: " << arg["stderr"] << endl;
			return 1;
		}
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_error_stream(ferr);
		}
	} else {
		for (size_t i = 0; i < com.processor_number(); i++) {
			com.processing_element(i).set_standard_error_stream(cerr);
		}
	}
	if (arg.argument()[0] == NULL) {
		if (VerboseMode) {
			version(to_basename(argv[0]));
			return 0;
		} else {
			cerr << "No executable file specified." << endl;
			return 1;
		}
	}
	com.set_cache_size(cache_size * 1024, line_size, way_size);
	{
		for (size_t i = 0; i < com.processor_number(); i++) {
			single_processing_element<r3000_word>&
				pe(com.processing_element(i));
			if (!pe.load(arg.argument()[0])) {
				cerr << arg.argument()[0]
					 << ": No such file or directory." << endl;
				return 1;
			}
			pe.set_commandline_argument(arg.argument());
		}
	}
	if (delay > 0) {
		com.shared_memory_read_wait(delay);
		com.shared_memory_write_wait(delay);
		com.sync_memory_read_wait(delay);
		com.sync_memory_write_wait(delay);
	}
	if (gdb_flag) {
		gdb_stub<r3000_word> stub;
		stub.set_multiprocessor(com);
		stub.set_debug_output(cerr);
		if (udp_flag) stub.use_udp();
		stub.service();
		return 0;
	} else if (cui_monitor_flag) {
		cui_debugger<processing_element_type> mon;
		mon.set(com);
		mon.set_prompt("> ");
		mon.interactive_mode();
		return 0;
	} else if (gui_monitor_flag) {
		gui_debugger<processing_element_type> mon;
		mon.set(com);
		mon.interactive_mode();
		return 0;
	} else {
		if (trace_flag) com.enable_trace(cout);
		int *nrun_run_c = 0, *nstl_run_c = 0, *nstl_stl_c = 0,
			*rrun_run_c = 0, *rstl_run_c = 0, *rstl_stl_c = 0,
			*wrun_run_c = 0, *wstl_run_c = 0, *wstl_stl_c = 0;
		if (PrintStatus) {
			nrun_run_c = new int[punum];
			nstl_run_c = new int[punum];
			nstl_stl_c = new int[punum];
			rrun_run_c = new int[punum];
			rstl_run_c = new int[punum];
			rstl_stl_c = new int[punum];
			wrun_run_c = new int[punum];
			wstl_run_c = new int[punum];
			wstl_stl_c = new int[punum];
			for (int i = 0; i < punum; i++) {
				nrun_run_c[i] = nstl_run_c[i] = nstl_stl_c[i] =
				rrun_run_c[i] = rstl_run_c[i] = rstl_stl_c[i] =
				wrun_run_c[i] = wstl_run_c[i] = wstl_stl_c[i] = 0;
			}
		}
		processing_element_type& pe(com.processing_element_actual(0));
		size_t count = 0;
		while (!pe.is_halt()) {
			if (DebugLevel > 0) {
				cerr << "--- clk:" << count << " ---" << endl;
				cerr << "in:" << endl << com << endl;
			}
			if ((++count & 0x3ff == 0) && com.is_bus_error()) {
				size_t i;
				for (i = 0; i < com.processor_number(); i++) {
					if (com.processing_element(i).is_bus_error()) break;
				}
				single_processing_element<r3000_word>&
					pe(com.processing_element(i));
				cout << "pu" << i << ": " << hex
					 << "bus error(I:0x" << pe.processor().program_counter()
					 << ", D:0x" << pe.bus_error_address() << ')'
					 << dec << endl;
				break;
			}
			if (PrintStatus) {
				for (int i = 0; i < punum; i++) {
					bool st_flag
						 = com.processing_element(i).processor().is_stall(),
						 rd_flag
						 = com.processing_element(i).processor().is_reading(),
						 wr_flag
						 = com.processing_element(i).processor().is_writing(),
						 cs_flag
						 = com.processing_element_actual(i).
							cache_controller().shared_access_is_finished();
					if (rd_flag) {
						if (!st_flag) {
							rrun_run_c[i]++;
						} else if (!cs_flag) {
							rstl_run_c[i]++;
						} else {
							rstl_stl_c[i]++;
						}
					} else if (wr_flag) {
						if (!st_flag) {
							wrun_run_c[i]++;
						} else if (!cs_flag) {
							wstl_run_c[i]++;
						} else {
							wstl_stl_c[i]++;
						}
					} else {
						if (!st_flag) {
							nrun_run_c[i]++;
						} else if (!cs_flag) {
							nstl_run_c[i]++;
						} else {
							nstl_stl_c[i]++;
						}
					}
				}
			}
			com.clock_in();
			if (DebugLevel > 0) {
				cerr << "out:" << endl << com << endl;
			}
			com.clock_out();
		}
		if (PrintStatus) {
			cout << "puid: "
					" no-r-r  rd-r-r  wr-r-r "
					" no-s-r  rd-s-r  wr-s-r "
					" no-s-s  rd-s-s  wr-s-s" << endl;
			for (int i = 0; i < punum; i++) {
				cout << setw(4) << i << ": "
					 << setw(7) << nrun_run_c[i] << ' '
					 << setw(7) << rrun_run_c[i] << ' '
					 << setw(7) << wrun_run_c[i] << ' '
					 << setw(7) << nstl_run_c[i] << ' '
					 << setw(7) << rstl_run_c[i] << ' '
					 << setw(7) << wstl_run_c[i] << ' '
					 << setw(7) << rstl_stl_c[i] << ' '
					 << setw(7) << wstl_stl_c[i] << ' '
					 << setw(7) << nstl_stl_c[i] << endl;
			}
			delete[] nrun_run_c; delete[] nstl_run_c; delete[] nstl_stl_c;
			delete[] rrun_run_c; delete[] rstl_run_c; delete[] rstl_stl_c;
			delete[] wrun_run_c; delete[] wstl_run_c; delete[] wstl_stl_c;
		}
		if (clock_flag) cout << "clock: " << pe.timer_clock_value() << endl;
		return pe.commandline_status();
	}
	/* not reached */
}
