/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2003 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< cache_line_illinois.h >>>
 * Cache line buffer for Illinos protocol
 */

#ifndef CACHE_LINE_ILLINOIS_H
#define CACHE_LINE_ILLINOIS_H

#include <cstddef>
#include <vector>
#include <iostream>
#include <iomanip>
#include <isis/cache_line_base.h>

template <
	typename address_type,
	typename data_type = address_type,
	class container_type = std::vector<data_type>
>
class cache_line_illinois;

template <typename A, typename D, class C>
class cache_line_illinois : public cache_line_base<A, D, C>
{
private:
	typedef cache_line_illinois<A, D, C> self_type;
	typedef cache_line_base<A, D, C> base_type;
	typedef C container_type;
	enum state_type {I, CE, CS, DE};
public:
	typedef A address_type;
	typedef D data_type;
	typedef typename container_type::size_type size_type;
	//
	explicit cache_line_illinois(
		size_type, address_type = address_type(0), data_type = data_type(0)
	);
	cache_line_illinois(const self_type&);
	virtual ~cache_line_illinois();
	void swap(self_type&);
	self_type& operator=(const self_type&);
	virtual void output(std::ostream&) const;
	//
	virtual bool is_valid() const
		{ return state_ != I; }
	bool is_exclusive() const
		{ return state_ == CE || state_ == DE; }
	bool is_clean_exclusive() const
		{ return state_ == CE; }
	bool is_clean_shared() const
		{ return state_ == CS; }
	bool is_dirty() const
		{ return state_ == DE; }
	//
	void set_invalid()
		{ state_ = I; }
	void set_clean_exclusive()
		{ state_ = CE; }
	void set_clean_shared()
		{ state_ = CS; }
	void set_dirty()
		{ state_ = DE; }
private:
	state_type state_;
	//
	// forbidden
	cache_line_illinois();
};

template <typename A, typename D, class C>
cache_line_illinois<A, D, C>::cache_line_illinois(
	cache_line_illinois<A, D, C>::size_type n,
	cache_line_illinois<A, D, C>::address_type adr,
	cache_line_illinois<A, D, C>::data_type dat
)
	:
	base_type(n, adr, dat),
	state_(I)
{}

template <typename A, typename D, class C>
cache_line_illinois<A, D, C>::cache_line_illinois(
	const cache_line_illinois<A, D, C>& a
)
	:
	base_type(a),
	state_(a.state_)
{}

template <typename A, typename D, class C>
cache_line_illinois<A, D, C>::~cache_line_illinois()
{}

template <typename A, typename D, class C>
void cache_line_illinois<A, D, C>::swap(cache_line_illinois<A, D, C>& a)
{
	using std::swap;
	base_type::swap(a);
	swap(state_, a.state_);
}

template <typename A, typename D, class C>
cache_line_illinois<A, D, C>& cache_line_illinois<A, D, C>::operator=(
	const cache_line_illinois<A, D, C>& a
)
{
	if (this != &a) self_type(a).swap(*this);
	return *this;
}

template <typename A, typename D, class C>
void cache_line_illinois<A, D, C>::output(std::ostream& os) const
{
	using namespace std;
	//
	if (!os) return;
	const char fill = os.fill();
	const ios::fmtflags flags = os.flags();
	os << hex << setfill('0');
	if (is_valid()) {
		os << setw(sizeof(address_type) * 2) << tag() << ": ";
		switch (state_) {
		case CE: os << "CE"; break;
		case CS: os << "CS"; break;
		case DE: os << "D "; break;
		default: os << "??"; break;
		}
		for (size_t i = 0; i < size(); i++) {
			os << ' ' << setw(sizeof(data_type) * 2) << data(i);
		}
	} else {
		os << "<invalid>";
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // CACHE_LINE_ILLINOIS_H
