/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2001 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< cache_controller_local.h >>>
 * local bus side cache controller
 */

#ifndef CACHE_CONTROLLER_LOCAL_H
#define CACHE_CONTROLLER_LOCAL_H

extern bool VerboseMode;

#include <cassert>
#include <iostream>
#include <isis/isis.h>
#include "cache_protocol.h"
#include "cache_controller_shared.h"

template <class Type>
class cache_controller_local : public root_object
{
private:
	typedef synchronous_unit inherited;
	typedef cache_line_PROTOCOL<Type, Type> cache_line_t;
	typedef cache_line_set<cache_line_t> cache_line_set_t;
	typedef set_associative_cache<cache_line_t> cache_t;

	enum state_t {
		READY,
		READ_HIT_ACK,
		READ_HIT_TRANS,
		READ_WAIT,
		READ_TRANS,
		SYNC_WAIT,
		SYNC_TRANS,
		WRITE_ACK,
		WRITE_TRANS,
		WRITE_WAIT_FOR_END
	};
	cache_controller_shared<Type>& cc_sh;
	bus_port<Type> l_bus;
	cache_t* cache_buf;
	Type shared_top_, shared_size_, sync_top_, sync_size_;
	state_t state;
	Type access_adr_, access_data_;
	// for verify
	mapped_memory<Type, Type, sizeof(Type), char> *shmem;
	mapped_memory<Type, Type, sizeof(Type), char> *symem;
	bus_port<Type> mem_bus;
public:
	cache_controller_local(cache_controller_shared<Type>&);
	cache_controller_local(const cache_controller_local<Type>&,
						   cache_controller_shared<Type>&);
	virtual ~cache_controller_local() {}
	cache_controller_local<Type>& operator=(
		const cache_controller_local<Type>&);
	virtual void output(std::ostream&) const;
	inline void clock_in();
	inline void clock_out();
	void reset() {
		state = READY; }
	const port& port_ref() const {
		return l_bus; }
	port& port_ref() {
		return l_bus; }
	inline int is_shared_address(Type) const;
	inline int is_sync_address(Type) const;
	inline int is_valid_address(Type) const;
	void set_shared_address(Type, Type);
	void set_sync_address(Type, Type);
	void connect(bus_port<Type>& a) {
		l_bus.connect(a); }
	void disconnect() {
		l_bus.disconnect(); }
	void disconnect(bus_port<Type>& a) {
		l_bus.disconnect(a); }
	int is_connected() const {
		return l_bus.is_connected(); }
	int is_connected(const bus_port<Type>& a) const {
		return l_bus.is_connected(a); }
	void connect_cache(cache_t& a) {
		cache_buf = &a; }
	void disconnect_cache() {
		cache_buf = NULL; }
	int is_connected_to_cache() const {
		return cache_buf != NULL; }
	// for verify
	void connect_mem(mapped_memory<Type, Type> *sh,
					 mapped_memory<Type, Type> *sy) {
		shmem = sh; symem = sy; }
	int is_connected_mem() {
		return (shmem != NULL) && (symem != NULL); }
	void disconnect_mem() {
		shmem = NULL; symem = NULL; }
	void connect_mem_port(bus_port<Type>& a) {
		mem_bus.connect(a); }
	void disconnect_mem_port() {
		mem_bus.disconnect(); }
	void disconnect_mem_port(bus_port<Type>& a) {
		mem_bus.disconnect(a); }
	int is_connected_mem_port() const {
		return mem_bus.is_connected(); }
	int is_connected_mem_port(const bus_port<Type>& a) const {
		return mem_bus.is_connected(a); }
	int verify();
};

template <class Type>
cache_controller_local<Type>::
cache_controller_local(cache_controller_shared<Type>& a)
	: cc_sh(a),
	  cache_buf(NULL),
	  shared_top_(Type(0)),
	  shared_size_(Type(0)),
	  sync_top_(Type(0)),
	  sync_size_(Type(0)),
	  state(READY),
	  shmem(NULL), symem(NULL)
{
}

template <class Type>
cache_controller_local<Type>::
cache_controller_local(const cache_controller_local<Type>& a,
					   cache_controller_shared<Type>& b)
	: cc_sh(b),
	  cache_buf(NULL),
	  shared_top_(a.shared_top_),
	  shared_size_(a.shared_size_),
	  sync_top_(a.sync_top_),
	  sync_size_(a.sync_size_),
	  state(READY),
	  shmem(a.shmem), symem(a.symem)
{
}

template <class Type>
cache_controller_local<Type>& cache_controller_local<Type>::
operator=(const cache_controller_local<Type>& a)
{
	if (this != &a) {
		shared_top_ = a.shared_top_;
		shared_size_ = a.shared_size_;
		sync_top_ = a.sync_top_;
		sync_size_ = a.sync_size_;
		state = READY;
		shmem = a.shmem;
		symem = a.symem;
	}
	return *this;
}

template <class Type>
inline int cache_controller_local<Type>::
is_shared_address(Type a) const
{
	return (a >= shared_top_) && (a < (shared_top_ + shared_size_));
}

template <class Type>
inline int cache_controller_local<Type>::
is_sync_address(Type a) const
{
	return (a >= sync_top_) && (a < (sync_top_ + sync_size_));
}

template <class Type>
inline int cache_controller_local<Type>::
is_valid_address(Type a) const
{
	return is_shared_address(a) || is_sync_address(a);
}

template <class Type>
void cache_controller_local<Type>::
set_shared_address(Type a, Type b)
{
	if (b > 0) shared_top_ = a, shared_size_ = b;
}

template <class Type>
void cache_controller_local<Type>::
set_sync_address(Type a, Type b)
{
	if (b > 0) sync_top_ = a, sync_size_ = b;
}

/*
 * ưåѴؿ
 */
template <class Type>
int cache_controller_local<Type>::verify()
{
	using namespace std;
	mapped_memory<Type, Type> *mem;
	const Type line_adr = cache_buf->tag_address(access_adr_);
	cache_line_set_t& cls = cache_buf->cache_line_set(line_adr);
	const int way_index = cls.way_of_hit(line_adr);

	if (way_index < 0)
		return 1;						// XXX: åʤǡ

	if (cls[way_index].is_dirty())
		return 1;						// XXX: ΤȤcleanå

	if (mem_bus.is_multi_write_request() ||
		mem_bus.is_multi_write_ack() ||
		mem_bus.is_multi_write_data())
		return 1;						// XXX: 񤭹

	if (is_sync_address(access_adr_)) {
		mem = symem;
	} else if (is_shared_address(access_adr_)) {
		mem = shmem;
	} else {
		return 0;						// XXX: å夵ʤ
	}

	const Type data = mem->read(access_adr_);
	if (data != access_data_) {
		const ios::fmtflags flags = cerr.flags();
		cerr << hex
			 << "Warning: inconsistent memory access at 0x"
			 << access_adr_ << endl
			 << "memory data: 0x" << data << endl
			 << "cache  data: 0x" << access_data_ << endl;
		cerr.flags(flags);
		return 0;
	} else {
		return 1;
	}
}

template <class Type>
inline void cache_controller_local<Type>::clock_in()
{
	assert(cache_buf != NULL &&
		   cache_buf->cache_size() > 0 &&
		   cache_buf->line_size() > 0 &&
		   cache_buf->way_size() > 0);

	switch (state) {
	case READY:
//		if (!cc_sh.shared_access_is_finished()) break; // delete this line

		//
		// 饤Ȥξ
		//
		if (l_bus.is_single_write_request()) {
			const Type adr = l_bus.address();
			if (is_valid_address(adr)) {
				// write from processor to shared memory, sync memory
				access_adr_ = adr;
				state = WRITE_ACK;
			}
			break;
		}
		//
		// ꡼ɤξ
		//
		if (l_bus.is_single_read_request()) {
			const Type adr = l_bus.address();
			const Type line_adr = cache_buf->tag_address(adr);
			//
			// å饤󤬶ͭХ¦ǻʤԤ
			//
			if (cc_sh.cache_line_is_busy(line_adr))
				break;
			//
			// åߥʤ鶦ͭХ¦˽Ǥ
			//
			access_adr_ = adr;
			cache_line_set_t& cls = cache_buf->cache_line_set(line_adr);
			const int way_index = cls.way_of_hit(line_adr);
			if (way_index < 0) {
				if (is_sync_address(adr)) {
					cc_sh.send_local_sync_request(adr);
					state = SYNC_WAIT;
				} else if (is_shared_address(adr)) {
					cc_sh.send_local_read_request(adr);
					state = READ_WAIT;
				}
				break;
			}
			//
			// ƱѥΰΥ꡼ɥҥå (F&D)
			// ŪWRITE_TRANSƱ
			//
			if (is_sync_address(adr)) {
				Type d = cls[way_index].data(cache_buf->line_offset(adr));
				//
				// 0ʲͤѹɤ߽Ф
				//
				if (d <= 0) {
					access_data_ = d;
					cls.increment_lru_counter(way_index);
					state = READ_HIT_ACK;
					break;
				}
				//
				// exclusiveʥå饤ξϽλ
				//
				else if (cls[way_index].is_exclusive()) {
					access_data_ = d--;
					cls[way_index].set_data(cache_buf->line_offset(adr), d);
					cls[way_index].set_dirty();
					cls.increment_lru_counter(way_index);
					state = READ_HIT_ACK;
					break;
				}
				//
				// ͭ饤ؤwrite뤿ᶦͭХ¦˽Ǥ
				//
				else {
					cc_sh.send_local_sync_request(adr);
					state = SYNC_WAIT;
					break;
				}
			}
			//
			// ͭΰΥ꡼ɥҥå
			//
			else if (is_shared_address(adr)) {
				access_data_ = cls[way_index].data(cache_buf->line_offset(adr));
				cls.increment_lru_counter(way_index);
				state = READ_HIT_ACK;
				break;
			}
		}
		break;

	case WRITE_TRANS:
		if (!l_bus.is_single_write_data()) {
			break;
		} else {
			// ͽ

			//
			// å饤󤬥Х¦ǻʤԤ
			//
			const Type line_adr = cache_buf->tag_address(access_adr_);
			if (cc_sh.cache_line_is_busy(line_adr)) {
				cc_sh.send_local_write_request(access_adr_, l_bus.data());
				state = WRITE_WAIT_FOR_END;
				break;
			}
			//
			// åߥʤshared¦˽Ǥ
			//
			cache_line_set_t& cls = cache_buf->cache_line_set(line_adr);
			const int way_index = cls.way_of_hit(line_adr);
			if (way_index < 0) {
				cc_sh.send_local_write_request(access_adr_, l_bus.data());
				state = WRITE_WAIT_FOR_END;
				break;
			}
			//
			// exclusivecache linecache hitϽλ
			//
			if (cls[way_index].is_exclusive()) {
				cls[way_index].set_data(
					cache_buf->line_offset(access_adr_), l_bus.data()
				);
				cls[way_index].set_dirty();
				cls.increment_lru_counter(way_index);
				state = READY;
				break;
			}
			//
			// sharedcache linecache hitshared¦˽Ǥ
			//
			else {
				cc_sh.send_local_write_request(access_adr_, l_bus.data());
				state = WRITE_WAIT_FOR_END;
				break;
			}
		}

	default:
		break;
	}
}

template <class Type>
inline void cache_controller_local<Type>::clock_out()
{
	assert(cache_buf != NULL &&
		   cache_buf->cache_size() > 0 &&
		   cache_buf->line_size() > 0 &&
		   cache_buf->way_size() > 0);

	switch (state) {
	case READ_HIT_ACK:
		l_bus.send_single_read_ack();
		state = READ_HIT_TRANS;
		break;
	case READ_TRANS:
		access_data_ = cc_sh.local_read_data();
	case READ_HIT_TRANS:
		if (VerboseMode) assert(verify());
		l_bus.send_single_read_data(access_data_);
		state = READY;
		break;
	case READ_WAIT:
		if (cc_sh.local_read_is_finished()) {
			l_bus.send_single_read_ack();
			state = READ_TRANS;
		}
		break;
	case SYNC_WAIT:
		if (cc_sh.local_sync_is_finished()) {
			l_bus.send_single_read_ack();
			state = SYNC_TRANS;
		}
		break;
	case SYNC_TRANS:
		if (VerboseMode) assert(verify());
		l_bus.send_single_read_data(cc_sh.local_sync_data());
		state = READY;
		break;
	case WRITE_ACK:
		l_bus.send_single_write_ack();
		state = WRITE_TRANS;
		break;
	case WRITE_WAIT_FOR_END:
		if (cc_sh.local_write_is_finished()) state = READY;
		break;
	default:
		break;
	}
}

template <class Type>
void cache_controller_local<Type>::output(std::ostream& os)
	const
{
	using namespace std;
	const ios::fmtflags flags = os.flags();
	os << hex;
	switch (state) {
	case READY:
		os << "ready";
		break;
	case READ_HIT_ACK:
		os << "read_hit_ack(0x" << access_adr_ << ')';
		break;
	case READ_HIT_TRANS:
		os << "read_hit_trans(0x" << access_adr_ << ')';
		break;
	case READ_WAIT:
		os << "read_wait(0x" << access_adr_ << ')';
		break;
	case READ_TRANS:
		os << "read_trans(0x" << access_adr_ << ')';
		break;
	case SYNC_WAIT:
		os << "sync_wait(0x" << access_adr_ << ')';
		break;
	case SYNC_TRANS:
		os << "sync_trans(0x" << access_adr_ << ')';
		break;
	case WRITE_ACK:
		os << "write_ack(0x" << access_adr_ << ')';
		break;
	case WRITE_TRANS:
		os << "write_trans(0x" << access_adr_ << ')';
		break;
	case WRITE_WAIT_FOR_END:
		os << "write_wait_for_end(0x" << access_adr_ << ')';
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // CACHE_CONTROLLER_LOCAL_H
