/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2001 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< bus_packet_shared.h >>>
 * bus packet for shared bus
 */

#ifndef BUS_PACKET_SHARED_H
#define BUS_PACKET_SHARED_H

#include <iostream>
#include <isis/isis.h>

template <class Type>
class bus_packet_shared : public bus_packet_base<Type>
{
private:
	typedef bus_packet_shared<Type> myclass;
	enum state_t {
		READ_REQ, READ_ACK, READ_GRANT, READ_NACK, READ_DATA,
		WRITE_REQ, WRITE_ACK, WRITE_GRANT, WRITE_NACK, WRITE_DATA,
		UPDATE_REQ, UPDATE_ACK, UPDATE_DATA,
		INV_REQ,
		LINE_COPY_REQ, LINE_COPY_ACK,
		LINE_MOVE_REQ, LINE_MOVE_ACK
	} state;
public:
	bus_packet_shared() {}
	virtual ~bus_packet_shared() {}
	virtual void input(std::istream&) {}
	virtual void output(std::ostream&) const;

	// bus_packet_baseν貾۴ؿ
	virtual bool is_read() const {
		return state == READ_REQ  || state == READ_ACK
			|| state == READ_NACK || state == READ_DATA; }
	virtual bool is_write() const {
		return state == WRITE_REQ  || state == WRITE_ACK
			|| state == WRITE_NACK || state == WRITE_DATA; }
	virtual bool is_request() const {
		return state == READ_REQ || state == WRITE_REQ
			|| state == INV_REQ  || state == UPDATE_REQ
			|| state == LINE_COPY_REQ || state == LINE_MOVE_REQ; }
	virtual bool is_grant() const {
		return state == READ_GRANT || state == WRITE_GRANT; }
	virtual bool is_ack() const {
		return state == READ_ACK || state == WRITE_ACK
			|| state == UPDATE_ACK
			|| state == LINE_COPY_ACK || state == LINE_MOVE_ACK; }
	virtual bool is_nack() const {
		return state == READ_NACK || state == WRITE_NACK; }
	virtual bool is_data() const {
		return state == READ_DATA || state == WRITE_DATA; }
	// 
	virtual void set_read() {}
	virtual void set_write() {}
	virtual void set_request() {}
	virtual void set_grant() {}
	virtual void set_ack() {}
	virtual void set_nack() {}
	virtual void set_data() {}

	// useful virtual functions

	// is_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_read_request() const {
		return state == READ_REQ; }
	virtual bool is_read_grant() const {
		return state == READ_GRANT; }
	virtual bool is_read_ack() const {
		return state == READ_ACK; }
	virtual bool is_read_nack() const {
		return state == READ_NACK; }
	virtual bool is_read_data() const {
		return state == READ_DATA; }
	virtual bool is_write_request() const {
		return state == WRITE_REQ; }
	virtual bool is_write_grant() const {
		return state == WRITE_GRANT; }
	virtual bool is_write_ack() const {
		return state == WRITE_ACK; }
	virtual bool is_write_nack() const {
		return state == WRITE_NACK; }
	virtual bool is_write_data() const {
		return state == WRITE_DATA; }

	// is_single_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_single_read_request() const {
		return is_single() && myclass::is_read_request(); }
	virtual bool is_single_read_grant() const {
		return is_single() && myclass::is_read_grant(); }
	virtual bool is_single_read_ack() const {
		return is_single() && myclass::is_read_ack(); }
	virtual bool is_single_read_nack() const {
		return is_single() && myclass::is_read_nack(); }
	virtual bool is_single_read_data() const {
		return is_single() && myclass::is_read_data(); }
	virtual bool is_single_write_request() const {
		return is_single() && myclass::is_write_request(); }
	virtual bool is_single_write_grant() const {
		return is_single() && myclass::is_write_grant(); }
	virtual bool is_single_write_ack() const {
		return is_single() && myclass::is_write_ack(); }
	virtual bool is_single_write_nack() const {
		return is_single() && myclass::is_write_nack(); }
	virtual bool is_single_write_data() const {
		return is_single() && myclass::is_write_data(); }

	// is_multi_{read,write}_{request,grant,ack,nack,data}()
	virtual bool is_multi_read_request() const {
		return is_multi() && myclass::is_read_request(); }
	virtual bool is_multi_read_grant() const {
		return is_multi() && myclass::is_read_grant(); }
	virtual bool is_multi_read_ack() const {
		return is_multi() && myclass::is_read_ack(); }
	virtual bool is_multi_read_nack() const {
		return is_multi() && myclass::is_read_nack(); }
	virtual bool is_multi_read_data() const {
		return is_multi() && myclass::is_read_data(); }
	virtual bool is_multi_write_request() const {
		return is_multi() && myclass::is_write_request(); }
	virtual bool is_multi_write_grant() const {
		return is_multi() && myclass::is_write_grant(); }
	virtual bool is_multi_write_ack() const {
		return is_multi() && myclass::is_write_ack(); }
	virtual bool is_multi_write_nack() const {
		return is_multi() && myclass::is_write_nack(); }
	virtual bool is_multi_write_data() const {
		return is_multi() && myclass::is_write_data(); }

	// set_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_read_request() {
		state = READ_REQ; }
	virtual void set_read_request(Type a) {
		myclass::set_read_request(), address() = a; }
	virtual void set_read_grant() {
		state = READ_GRANT; }
	virtual void set_read_ack() {
		state = READ_ACK; }
	virtual void set_read_nack() {
		state = READ_NACK; }
	virtual void set_read_data() {
		state = READ_DATA; }
	virtual void set_read_data(Type a) {
		myclass::set_read_data(), data() = a; }
	virtual void set_write_request() {
		state = WRITE_REQ; }
	virtual void set_write_request(Type a) {
		myclass::set_write_request(), address() = a; }
	virtual void set_write_grant() {
		state = WRITE_GRANT; }
	virtual void set_write_ack() {
		state = WRITE_ACK; }
	virtual void set_write_nack() {
		state = WRITE_NACK; }
	virtual void set_write_data() {
		state = WRITE_DATA; }
	virtual void set_write_data(Type a) {
		myclass::set_write_data(), data() = a; }

	// set_single_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_single_read_request() {
		set_single(), myclass::set_read_request(); }
	virtual void set_single_read_request(Type a) {
		set_single(), myclass::set_read_request(a); }
	virtual void set_single_read_grant() {
		set_single(), myclass::set_read_grant(); }
	virtual void set_single_read_ack() {
		set_single(), myclass::set_read_ack(); }
	virtual void set_single_read_nack() {
		set_single(), myclass::set_read_nack(); }
	virtual void set_single_read_data() {
		set_single(), myclass::set_read_data(); }
	virtual void set_single_read_data(Type a) {
		set_single(), myclass::set_read_data(a); }
	virtual void set_single_write_request() {
		set_single(), myclass::set_write_request(); }
	virtual void set_single_write_request(Type a) {
		set_single(), myclass::set_write_request(a); }
	virtual void set_single_write_grant() {
		set_single(), myclass::set_write_grant(); }
	virtual void set_single_write_ack() {
		set_single(), myclass::set_write_ack(); }
	virtual void set_single_write_nack() {
		set_single(), myclass::set_write_nack(); }
	virtual void set_single_write_data() {
		set_single(), myclass::set_write_data(); }
	virtual void set_single_write_data(Type a) {
		set_single(), myclass::set_write_data(a); }

	// set_multi_{read,write}_{request,grant,ack,nack,data}()
	virtual void set_multi_read_request() {
		set_multi(), myclass::set_read_request(); }
	virtual void set_multi_read_request(Type a, unsigned b) {
		set_multi(b), myclass::set_read_request(a); }
	virtual void set_multi_read_grant() {
		set_multi(), myclass::set_read_grant(); }
	virtual void set_multi_read_ack() {
		set_multi(), myclass::set_read_ack(); }
	virtual void set_multi_read_nack() {
		set_multi(), myclass::set_read_nack(); }
	virtual void set_multi_read_data() {
		set_multi(), myclass::set_read_data(); }
	virtual void set_multi_read_data(Type a, unsigned b, unsigned c) {
		set_multi(b, c), myclass::set_read_data(a); }
	virtual void set_multi_write_request(Type a, unsigned b) {
		set_multi(b), myclass::set_write_request(a); }
	virtual void set_multi_write_grant() {
		set_multi(), myclass::set_write_grant(); }
	virtual void set_multi_write_ack() {
		set_multi(), myclass::set_write_ack(); }
	virtual void set_multi_write_nack() {
		set_multi(), myclass::set_write_nack(); }
	virtual void set_multi_write_data() {
		set_multi(), myclass::set_write_data(); }
	virtual void set_multi_write_data(Type a, unsigned b, unsigned c) {
		set_multi(b, c), myclass::set_write_data(a); }

	// for snoop cache protocol
	bool is_update_request() const {
		return state == UPDATE_REQ; }
	bool is_update_ack() const {
		return state == UPDATE_ACK; }
	//
	bool is_invalidation_request() const {
		return state == INV_REQ; }
	//
	bool is_line_copy_request() const {
		return state == LINE_COPY_REQ; }
	bool is_line_copy_ack() const {
		return state == LINE_COPY_ACK; }
	//
	bool is_line_move_request() const {
		return state == LINE_MOVE_REQ; }
	bool is_line_move_ack() const {
		return state == LINE_MOVE_ACK; }
	//
	void set_update_request() {
		state = UPDATE_REQ; }
	void set_update_request(Type a) {
		set_update_request(), address() = a; }
	void set_update_ack() {
		state = UPDATE_ACK; }
	//
	void set_invalidation_request() {
		state = INV_REQ; }
	void set_invalidation_request(Type a) {
		set_invalidation_request(), address() = a; }
	//
	void set_line_copy_request() {
		state = LINE_COPY_REQ; }
	void set_line_copy_request(Type a) {
		set_line_copy_request(), address() = a; }
	void set_line_copy_ack() {
		state = LINE_COPY_ACK; }
	//
	void set_line_move_request() {
		state = LINE_MOVE_REQ; }
	void set_line_move_request(Type a) {
		set_line_move_request(), address() = a; }
	void set_line_move_ack() {
		state = LINE_MOVE_ACK; }
};

template <class Type>
void bus_packet_shared<Type>::output(std::ostream& os) const
{
	using namespace std;
	if (!os) return;
	const ios::fmtflags flags = os.flags();
	switch (state) {
	case READ_REQ:
		os << "r_req(0x" << hex << address();
		if (is_multi()) os << ", *" << dec << total_packet_count();
		os << ')';
		break;
	case WRITE_REQ:
		os << "w_req(0x" << hex << address();
		if (is_multi()) os << ", *" << dec << total_packet_count();
		os << ')';
		break;
	case READ_GRANT:
		os << "r_grnt";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case READ_ACK:
		os << "r_ack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case WRITE_GRANT:
		os << "w_grnt";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case WRITE_ACK:
		os << "w_ack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case READ_NACK:
		os << "r_nack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case WRITE_NACK:
		os << "w_nack";
		if (is_multi()) os << "(*" << dec << total_packet_count() << ')';
		break;
	case READ_DATA:
		os << "r_dat(0x" << hex << data();
		if (is_multi())
			os << ", " << dec
			   << packet_number() << '/' << total_packet_count();
		os << ')';
		break;
	case WRITE_DATA:
		os << "w_dat(0x" << hex << data();
		if (is_multi())
			os << ", " << dec
			   << packet_number() << '/' << total_packet_count();
		os << ')';
		break;
	case UPDATE_REQ:
		os << "upd_req(0x" << hex << address() << ')';
		break;
	case UPDATE_ACK:
		os << "upd_ack";
		break;
	case UPDATE_DATA:
		os << "upd_dat(0x" << hex << data() << ')';
		break;
	case INV_REQ:
		os << "inv_req(0x" << hex << address() << ')';
		break;
	case LINE_COPY_REQ:
		os << "lcopy_req(0x" << hex << address() << ')';
		break;
	case LINE_COPY_ACK:
		os << "lcopy_ack";
		break;
	case LINE_MOVE_REQ:
		os << "lmove_req(0x" << hex << address() << ')';
		break;
	case LINE_MOVE_ACK:
		os << "lmove_ack";
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // BUS_PACKET_SHARED_H
