/* -*- C++ -*-
 * Copyright (c) 1998 Keisuke Inoue
 *               2001 Masaki Wakabayashi
 *               Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< bus_bridge.h >>>
 * ͭХ(internal bus)Υǡ(big_word<Type>)ȥ
 * åΥǡ(Type)ۤʤ뤿ᡢꥳȥ
 * ΥХ(external bus)ϤΥ󥿥ե𤷤ƶͭ
 * ³롣
 *
 */

#ifndef BUS_BRIDGE_H
#define BUS_BRIDGE_H

#include <iostream>
#include <isis/isis.h>
#include "big_word.h"

template <class Type>
class bus_bridge : public synchronous_unit
{
private:
	typedef synchronous_unit inherited;
	typedef big_word<Type> bus_data_t;
	enum state_t {
		IDLE,
		SEND_MULTI_READ_REQ,
		MULTI_READ_ACK_WAIT,
		MULTI_READ_TRANS,
		SEND_SINGLE_READ_ACK,
		SEND_SINGLE_READ_TRANS,
		SEND_SINGLE_WRITE_ACK,
		SINGLE_WRITE_TRANS,
		SEND_MULTI_WRITE_REQ,
		MULTI_WRITE_ACK_WAIT,
		MULTI_WRITE_TRANS,
		CLEAR
	} state;
	bus_port<bus_data_t> i_bus;			// Х¦(ͭХ¦)
	bus_port<Type> e_bus;				// Х¦
	Type address;
	bus_data_t buf_adr, buf_dat;
	size_t bus_size;					// ХΥХ(word)
	size_t n_packet;					// number of packet
public:
	bus_bridge();
	bus_bridge(const bus_bridge<Type>&);
	~bus_bridge() {}
	// syncronous_unit饹ν貾۴ؿ
	virtual void clock_in();
	virtual void clock_out();
	virtual void reset();
	//
	virtual void output(std::ostream&) const;
	// Х
	const port& internal_bus_port() const {
		return i_bus; }
	port& internal_bus_port() {
		return i_bus; }
	void connect_internal_bus_port(port& a) {
		i_bus.connect(a); }
	void connect_internal_bus_port(single_port_synchronous_unit& a) {
		a.connect(i_bus); }
	void disconnect_internal_bus_port() {
		i_bus.disconnect(); }
	void disconnect_internal_bus_port(port& a) {
		i_bus.disconnect(a); }
	int is_connected_to_internal_bus_port() const {
		return i_bus.is_connected(); }
	int is_connected_to_internal_bus_port(const port& a) const {
		return i_bus.is_connected(a); }
	// Х
	const bus_port<Type>& external_bus_port() const {
		return e_bus; }
	bus_port<Type>& external_bus_port() {
		return e_bus; }
	void connect_external_bus_port(port& a) {
		e_bus.connect(a); }
	void disconnect_external_bus_port() {
		e_bus.disconnect(); }
	void disconnect_external_bus_port(port& a) {
		e_bus.disconnect(a); }
	int is_connected_to_external_bus_port() const {
		return e_bus.is_connected(); }
	int is_connected_to_external_bus_port(const port& a) const {
		return e_bus.is_connected(a); }
};

template <class Type>
bus_bridge<Type>::bus_bridge()
	: state(IDLE)
{
}

template <class Type>
bus_bridge<Type>::bus_bridge(const bus_bridge<Type>& a)
	: inherited(a),
	  state(IDLE),
	  i_bus(a.i_bus),
	  e_bus(a.e_bus),
	  address(a.address),
	  buf_adr(a.buf_adr),
	  buf_dat(a.buf_dat),
	  n_packet(a.n_packet)
{
}

template <class Type>
void bus_bridge<Type>::reset()
{
	state = IDLE;
}

/*
 * ꡼ɥȥ󥶥֥å
 * idle: ꥯԤ
 * r1o: ˥꡼ɥꥯȤФ
 * r2i: ꤫AckԤ
 * r3i: ꤫ƤǡƼ
 * r4o: å女ȥAck֤
 * r5o: å女ȥ˥ǡ
 *
 * 饤ȥȥ󥶥֥å
 * idle: ꥯԤ
 * w1o: å女ȥAck֤
 * w2i: å女ȥ餫ǡ
 * w3o: إ饤ȥꥯȤФ
 * w4i: ꤫AckԤ
 * w5o: إǡ
 */
template <class Type>
void bus_bridge<Type>::clock_in()
{
	switch (state) {
	case IDLE:
		if (i_bus.is_ready()) break;

		buf_adr = i_bus.address();
		bus_size = buf_adr.size();
		address = buf_adr[0];
		if (i_bus.is_single_read_request()) {
			state = SEND_MULTI_READ_REQ;
		} else if (i_bus.is_single_write_request()) {
			state = SEND_SINGLE_WRITE_ACK;
		}
		break;

	case MULTI_READ_ACK_WAIT://r2i
		if (e_bus.is_multi_read_ack()) {
			buf_dat.resize(bus_size);
			state = MULTI_READ_TRANS;
		}
		break;

	case MULTI_READ_TRANS://r3i
		if (e_bus.is_read_data()) {
			const size_t no_packet = e_bus.packet_number();
			assert(e_bus.total_packet_count() == bus_size);
			assert(no_packet < bus_size);
			buf_dat[no_packet] = e_bus.data();
			if ((no_packet+1) == bus_size) {
				state = SEND_SINGLE_READ_ACK;
			}
		}
		break;

	case SINGLE_WRITE_TRANS://w2i
		if (i_bus.is_write_data()) {
			buf_dat = i_bus.data();
			state = SEND_MULTI_WRITE_REQ;
		}
		break;

	case MULTI_WRITE_ACK_WAIT://w4i
		if (e_bus.is_multi_write_ack()) {
			n_packet = 0;
			state = MULTI_WRITE_TRANS;
		}
		break;

	default:
		break;
	}
}

template <class Type>
void bus_bridge<Type>::clock_out()
{
	switch (state) {
	case SEND_MULTI_READ_REQ://r1o
		e_bus.send_multi_read_request(address, bus_size);
		state = MULTI_READ_ACK_WAIT;
		break;

	case SEND_SINGLE_READ_ACK://r4o
		e_bus.clear();
		i_bus.send_single_read_ack();
		state = SEND_SINGLE_READ_TRANS;
		break;

	case SEND_SINGLE_READ_TRANS://r5o
		i_bus.send_single_read_data(buf_dat);
		state = IDLE;
		break;

	case SEND_SINGLE_WRITE_ACK://w1o
		i_bus.send_single_write_ack();
		state = SINGLE_WRITE_TRANS;
		break;

	case SEND_MULTI_WRITE_REQ://w3o
		e_bus.send_multi_write_request(address, bus_size);
		state = MULTI_WRITE_ACK_WAIT;
		break;

	case MULTI_WRITE_TRANS://w5o
		e_bus.send_multi_write_data(buf_dat[n_packet], bus_size, n_packet);
		if (++n_packet == bus_size) {
			state = CLEAR;
		}
		break;

	case CLEAR:
		e_bus.clear();
		state = IDLE;
		break;

	default:
		break;
	}
}

template <class Type>
void bus_bridge<Type>::output(std::ostream& os) const
{
	using namespace std;
	const ios::fmtflags flags = os.flags();
	os << hex;
	switch (state) {
	case IDLE:
		os << "idle";
		break;
	case SEND_MULTI_READ_REQ:
		os << "send_multi_read_req(0x" << address << ')';
		break;
	case MULTI_READ_ACK_WAIT:
		os << "multi_read_ack_wait(0x" << address << ')';
		break;
	case MULTI_READ_TRANS:
		os << "multi_read_trans(0x" << address << ')';
		break;
	case SEND_SINGLE_READ_ACK:
		os << "send_single_read_ack(0x" << address << ')';
		break;
	case SEND_SINGLE_READ_TRANS:
		os << "send_single_read_trans(0x" << address << ')';
		break;
	case SEND_SINGLE_WRITE_ACK:
		os << "send_single_write_ack(0x" << address << ')';
		break;
	case SINGLE_WRITE_TRANS:
		os << "single_write_trans(0x" << address << ')';
		break;
	case SEND_MULTI_WRITE_REQ:
		os << "send_multi_write_req(0x" << address << ')';
		break;
	case MULTI_WRITE_ACK_WAIT:
		os << "multi_write_ack_wait(0x" << address << ')';
		break;
	case MULTI_WRITE_TRANS:
		os << "multi_write_trans(0x" << address << ')';
		break;
	case CLEAR:
		os << "clear";
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class Type>
std::ostream& operator<<(std::ostream& os, const bus_bridge<Type>& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}

#endif // BUS_BRIDGE_H
