/*
 * <<< single_r3081.cc >>>
 *
 * --- Single r3081 architecture simulator 'isis_single_r3081'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <isis/isis.h>

using namespace std;

const r3000_word local_memory_address = 0;
const r3000_word local_memory_size = 0x20000000;

static const char* to_basename(const char* s)
{
	const char* head = s;
	while (*s != '\0') {
		if (*s == '/') head = s + 1;
		s++;
	}
	return head;
}

void usage(const char* name)
{
	cout << "usage: " << name
		 << " <target> [options] [-- [target options]]" << endl
		 << "options:" << endl
		 << "  -c,    --clock     print execution time in clock." << endl
		 << "  -d<n>, --delay=<n> set memory access delay."		  << endl
		 << "  -g,    --gdb       use gdb mode."				  << endl
		 << "  -h,    --help      print this message."			  << endl
		 << "  -m,    --monitor   use monitor mode."			  << endl
		 << "  -v,    --verbose   verbosely output."			  << endl
		 << "  --udp              use gdb mode with udp."		  << endl
		 << "  --trace            enable address trace output."	  << endl
		 << "  --version          print version information."	  << endl
		 << "  --stdin=<file>     set standard input file."		  << endl
		 << "  --stdout=<file>    set standard output file."	  << endl
		 << "  --stderr=<file>    set standard error file."		  << endl;
}

void version(const char* name)
{
	cout << name << " - " << isis::version_string() << endl;
}

int main(int, char** argv)
{
	typedef r3081_processing_element processing_element_type;
	unsigned int delay = 0;
	bool clock_flag = false, trace_flag = false, cui_monitor_flag = false,
		 gui_monitor_flag = false, gdb_flag = false, udp_flag = false,
		 verbose_flag = false;
	argument_parser arg((const char* const*)(argv + 1));
	if (arg.defined('h') || arg.defined("help"))
		{ usage(to_basename(argv[0])); return 0; }
	if (arg.defined("version"))
		{ version(to_basename(argv[0])); return 0; }
	if (arg.defined('c') || arg.defined("clock"))
		clock_flag = true;
	if (arg.defined('g') || arg.defined("gdb"))
		gdb_flag = true;
	if (arg.defined("gui"))
		gui_monitor_flag = true;
	if (arg.defined('m') || arg.defined("monitor"))
		cui_monitor_flag = true;
	if (arg.defined('v') || arg.defined("verbose"))
		verbose_flag = clock_flag = true;
	if (arg.defined("udp"))
		udp_flag = true;
	if (arg.defined("trace"))
		trace_flag = true;
	if (arg['d'] != NULL)	  delay = atoi(arg['d']);
	if (arg["delay"] != NULL) delay = atoi(arg["delay"]);
	processing_element_type pe;
	pe.set_local_memory_area(local_memory_address, local_memory_size);
	pe.set_local_memory_read_wait(delay);
	pe.set_local_memory_write_wait(delay);
	pe.set_file_table_size(16);
	ifstream fin;
	ofstream fout, ferr;
	if (arg["stdin"] != NULL) {
		fin.open(arg["stdin"], ios::in);
		if (!fin) {
			cerr << "open error: " << arg["stdin"] << endl;
			return 1;
		}
		pe.set_standard_input_stream(fin);
	} else {
		pe.set_standard_input_stream(cin);
	}
	if (arg["stdout"] != NULL) {
		fout.open(arg["stdout"], ios::out);
		if (!fout) {
			cerr << "open error: " << arg["stdout"] << endl;
			return 1;
		}
		pe.set_standard_output_stream(fout);
	} else {
		pe.set_standard_output_stream(cout);
	}
	if (arg["stderr"] != NULL) {
		ferr.open(arg["stderr"], ios::out);
		if (!ferr) {
			cerr << "open error: " << arg["stderr"] << endl;
			return 1;
		}
		pe.set_standard_error_stream(ferr);
	} else {
		pe.set_standard_error_stream(cerr);
	}
	if (arg.argument()[0] == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}
	if (!pe.load(arg.argument()[0])) {
		cerr << arg.argument()[0]
			 << ": No such file or directory." << endl;
		return 1;
	}
	pe.set_commandline_argument(arg.argument());
	if (gdb_flag) {
		gdb_stub<r3000_word> stub;
		stub.set_singleprocessor(pe);
		stub.set_debug_output(cerr);
		if (udp_flag) stub.use_udp();
		if (verbose_flag) stub.set_verbose();
		stub.service();
		return 0;
	} else if (cui_monitor_flag) {
		cui_debugger<processing_element_type> mon;
		mon.set(pe);
		mon.set_prompt("> ");
		mon.interactive_mode();
		return 0;
	} else if (gui_monitor_flag) {
		gui_debugger<processing_element_type> mon;
		mon.set(pe);
		mon.interactive_mode();
		return 0;
	} else {
		while (!pe.is_halt()) {
			if (verbose_flag) {
				cout << "--- clk:" << pe.timer_clock_value() << " ---" << endl
					 << pe.processor() << endl << endl;
			} else if (trace_flag) {
				cout << "0x" << hex << pe.processor().program_counter()
					 << dec << endl;
			}
			if (pe.is_bus_error()) {
				cerr << hex
					 << "bus error(I:0x" << pe.processor().program_counter()
					 << ", D:0x" << pe.bus_error_address() << ')'
					 << dec << endl;
				break;
			}
			pe.clock_in();
			pe.clock_out();
		}
		if (clock_flag) cout << "clock: " << pe.timer_clock_value() << endl;
		return pe.commandline_status();
	}
	/* not reached */
}
