/*
 * <<< shared_bus_architecture.cc >>>
 *
 * --- Shared bus architecture class 'shared_bus_architecture'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include "shared_bus_architecture.h"

using namespace std;

shared_bus_architecture::shared_bus_architecture(void)
{
	insert_synchronous_unit(shmem_ctl);
	insert_synchronous_unit(shfad_ctl);
	shmem_ctl.port_ref().connect(shfad_ctl.port_ref());
	shmem_ctl.connect_memory(shmem);
	shfad_ctl.connect_memory(shfad);
}

shared_bus_architecture::shared_bus_architecture(size_t a)
	: pe(a)
{
	insert_synchronous_unit(shmem_ctl);
	insert_synchronous_unit(shfad_ctl);
	shmem_ctl.port_ref().connect(shfad_ctl.port_ref());
	shmem_ctl.connect_memory(shmem);
	shfad_ctl.connect_memory(shfad);
	for (size_t i = 0; i < pe.size(); i++) {
		insert_synchronous_unit_set(pe[i]);
		shmem_ctl.port_ref().connect(pe[i].port_ref());
		pe[i].set_processor_number(pe.size());
		pe[i].set_processor_id(i);
		pe[i].memory_map().insert(shmem);
		pe[i].memory_map().insert(shfad);
	}
}

shared_bus_architecture::shared_bus_architecture(const
	shared_bus_architecture& a)
	: pe(a.pe),
	  shmem(a.shmem),
	  shfad(a.shfad),
	  shmem_ctl(a.shmem_ctl),
	  shfad_ctl(a.shfad_ctl)
{
	insert_synchronous_unit(shmem_ctl);
	insert_synchronous_unit(shfad_ctl);
	shmem_ctl.port_ref().connect(shfad_ctl.port_ref());
	shmem_ctl.connect_memory(shmem);
	shfad_ctl.connect_memory(shfad);
	for (size_t i = 0; i < pe.size(); i++) {
		insert_synchronous_unit_set(pe[i]);
		shmem_ctl.port_ref().connect(pe[i].port_ref());
		pe[i].memory_map().insert(shmem);
		pe[i].memory_map().insert(shfad);
		pe[i].set_shared_memory_area(shmem.top(), shmem.size());
		pe[i].set_sync_memory_area(shfad.top(), shfad.size());
	}
}

shared_bus_architecture::~shared_bus_architecture()
{}

void shared_bus_architecture::output(ostream& os) const
{
	for (size_t i = 0; i < pe.size(); i++) {
		os << "pe" << i << ": " << pe[i] << '\n';
	}
	os << "bus: " << pe[0].port_ref();
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

void shared_bus_architecture::set_processor_number(size_t a)
{
	if (a >= 0) {
		pe.resize(a);
		for (size_t i = 0; i < pe.size(); i++) {
			insert_synchronous_unit_set(pe[i]);
			shmem_ctl.port_ref().connect(pe[i].port_ref());
			pe[i].set_processor_number(pe.size());
			pe[i].set_processor_id(i);
			pe[i].memory_map().insert(shmem);
			pe[i].memory_map().insert(shfad);
			pe[i].set_shared_memory_area(shmem.top(), shmem.size());
			pe[i].set_sync_memory_area(shfad.top(), shfad.size());
		}
	}
}

bool shared_bus_architecture::is_halt(void) const
{
	for (size_t i = 0; i < pe.size(); i++) {
		if (pe[i].is_halt()) return true;
	}
	return false;
}

bool shared_bus_architecture::is_bus_error(void) const
{
	for (size_t i = 0; i < pe.size(); i++) {
		if (pe[i].is_bus_error()) return true;
	}
	return false;
}

void shared_bus_architecture::set_shared_memory_area
	(shared_bus_architecture::address_type top,
	 shared_bus_architecture::size_type size)
{
	shmem.set_top(top);
	shmem.resize(size);
	for (size_t i = 0; i < pe.size(); i++) {
		pe[i].set_shared_memory_area(top, size);
	}
}

void shared_bus_architecture::set_sync_memory_area
	(shared_bus_architecture::address_type top,
	 shared_bus_architecture::size_type size)
{
	shfad.set_top(top);
	shfad.resize(size);
	for (size_t i = 0; i < pe.size(); i++) {
		pe[i].set_sync_memory_area(top, size);
	}
}

void shared_bus_architecture::enable_coprocessor(void)
{
	for (size_t i = 0; i < pe.size(); i++) pe[i].enable_coprocessor();
}

void shared_bus_architecture::enable_instruction_cache(void)
{
	for (size_t i = 0; i < pe.size(); i++) pe[i].enable_instruction_cache();
}

void shared_bus_architecture::enable_data_cache(void)
{
	for (size_t i = 0; i < pe.size(); i++) pe[i].enable_data_cache();
}

void shared_bus_architecture::disable_coprocessor(void)
{
	for (size_t i = 0; i < pe.size(); i++) pe[i].disable_coprocessor();
}

void shared_bus_architecture::disable_instruction_cache(void)
{
	for (size_t i = 0; i < pe.size(); i++) pe[i].disable_instruction_cache();
}

void shared_bus_architecture::disable_data_cache(void)
{
	for (size_t i = 0; i < pe.size(); i++) pe[i].disable_data_cache();
}

void shared_bus_architecture::set_shared_memory_read_wait(unsigned int a)
{
	return shmem_ctl.set_read_wait(a);
}

void shared_bus_architecture::set_shared_memory_write_wait(unsigned int a)
{
	return shmem_ctl.set_write_wait(a);
}

void shared_bus_architecture::set_sync_memory_read_wait(unsigned int a)
{
	return shfad_ctl.set_read_wait(a);
}

void shared_bus_architecture::set_sync_memory_write_wait(unsigned int a)
{
	return shfad_ctl.set_write_wait(a);
}
